/* ***************************************************************************************
* FILE:          ScreenBrokerService.cpp
* SW-COMPONENT:  HMI-BASE
* DESCRIPTION:  HMIBaseTraceCmds.h is part of HMI-Base ScreenBroker
* COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
****************************************************************************************/


#include "ScreenBrokerService.h"
#if defined(SCREENBROKER_WITH_SYSTEMD)
#include <systemd/sd-daemon.h>
#endif
#include <ScreenBroker/Service/Service.h>

#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/ScreenBrokerService.cpp.trc.h"
#endif

using namespace ::asf::core;


namespace ScreenBroker {
namespace Service {

bool gIlmAccess = true;
#define SIGNAL_PRINTF(args, ...)                            \
    do {                                                    \
        char buffer[100];                                   \
        memset(buffer, '\0', sizeof(buffer));               \
        sprintf(buffer, args, ## __VA_ARGS__);              \
        write(STDOUT_FILENO, buffer, strlen(buffer));       \
            } while(0)

#define SIGACTION_CHECK(signal, action, context)            \
    do {                                                    \
        if (sigaction(signal, action, context) < 0) {       \
            perror ("sigaction");                           \
            return 1;                                       \
                        }                                                   \
            } while(0)

#define PRINT_VERSION printf("screenbroker service %s\n", SCREENBROKER_VERSION_STRING)

ScreenBrokerService::ScreenBrokerService()
{
   // Initialize screen broker ETG service logging
   vInitPlatformEtg();
   // Handle system signals
   vHandleSysSignals();
   //Initialize the Screenbroker service
   ETG_TRACE_USR1(("Starting screen broker service (version %40s)", SCREENBROKER_VERSION_STRING));
   ScreenBroker::Internal::Service& lScreenBrokerService =
      ScreenBroker::Internal::Service::GetInstance();
   if (lScreenBrokerService.Init(gIlmAccess))
   {
#if defined(SCREENBROKER_WITH_SYSTEMD)
      sd_notify(0, "READY=1");
#endif
      ScreenBroker::Internal::LayerManagerAccessor::Dump();
   }
   else
   {
      ETG_TRACE_FATAL(("Initialization of screen broker service failed"));
   }
}


ScreenBrokerService::~ScreenBrokerService()
{
}


int ScreenBrokerService::vHandleSysSignals()
{
#if 0 // don't use this as we are an ASF based component now, which already handles signals
   // Handle system signals for termination and interrupt
   struct sigaction actTerm;
   memset(&actTerm, '\0', sizeof(actTerm));
   actTerm.sa_flags = SA_SIGINFO;
   actTerm.sa_sigaction = &OnTermination;
   SIGACTION_CHECK(SIGINT, &actTerm, NULL);
   SIGACTION_CHECK(SIGTERM, &actTerm, NULL);
#endif

   // Handle system signals for reset
   struct sigaction actReset;
   memset(&actReset, '\0', sizeof(actReset));
   actReset.sa_flags = SA_SIGINFO;
   actReset.sa_sigaction = &OnReset;
   SIGACTION_CHECK(SIGUSR1, &actReset, NULL);

   // Handle system signals for diagnosis
   struct sigaction actDiagnosis;
   memset(&actDiagnosis, '\0', sizeof(actDiagnosis));
   actDiagnosis.sa_flags = SA_SIGINFO;
   actDiagnosis.sa_sigaction = &OnDiagnosis;
   SIGACTION_CHECK(SIGUSR2, &actDiagnosis, NULL);

   return 1;
}


// ------------------------------------------------------------------------
void ScreenBrokerService::OnTermination(int sig, siginfo_t* siginfo, void* context)
{
   (void)siginfo;
   (void)context;
   SIGNAL_PRINTF("Signal %d received. Shutting down.\n", sig);
   ScreenBroker::Internal::Service::GetInstance().Destroy();
}


// ------------------------------------------------------------------------
void ScreenBrokerService::OnReset(int sig, siginfo_t* siginfo, void* context)
{
   (void)siginfo;
   (void)context;
   SIGNAL_PRINTF("Signal %d received. Reset activator plugins.\n", sig);
   ScreenBroker::Internal::Service::GetInstance().Reset(ScreenBroker::Internal::ResetMode::ActivatorPlugins);
}


// ------------------------------------------------------------------------
void ScreenBrokerService::OnDiagnosis(int sig, siginfo_t* siginfo, void* context)
{
   (void)siginfo;
   (void)context;
   SIGNAL_PRINTF("Signal %d received. Calling plugin diagnosis.\n", sig);
   ScreenBroker::Internal::Service::GetInstance().Diagnosis();
}


} //namespace
}
