/* ***************************************************************************************
* FILE:          ScreenArea.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenArea.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "ScreenArea.h"
#include "LayerManagerAccessor.h"
#include "PluginManager.h"
#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/ScreenArea.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenBrokerService);

// ------------------------------------------------------------------------
ScreenArea::ScreenArea()
   : mId(0),
     mXPos(0),
     mYPos(0),
     mWidth(0),
     mHeight(0),
     mDisplayId(UInt32Max),
     mDisplayAlias(UInt32Max),
     mLayerId(0),
     mIsPopupArea(false),
     mIsManaged(true),
     mIsVisible(true),
     mIsNotifyOnPopupStatus(false)
{
}


// ------------------------------------------------------------------------
void ScreenArea::Reset()
{
   //SCREENBROKER_LOG_FN();
   if ((mIsManaged) && (0 != mLayerId))
   {
      // Remove ILM layer
      if (Internal::LayerManagerAccessor::RemoveLayer(mLayerId))
      {
         ETG_TRACE_USR1(("Remove ILM layer %u for screen area ID:%u",
                         mLayerId,
                         mId));
         mLayerId = 0;
      }
      else
      {
         ETG_TRACE_ERR(("Failed to remove ILM layer %u for screen area ID:%u",
                        mLayerId,
                        mId));
      }
   }
}


// ------------------------------------------------------------------------
ScreenArea::~ScreenArea()
{
   Reset();
}


// ------------------------------------------------------------------------
bool ScreenArea::Init(UInt32 id,
                      UInt32 width,
                      UInt32 height,
                      UInt32 xPos,
                      UInt32 yPos,
                      bool isPopupArea,
                      UInt32 displayId,
                      UInt32 displayAlias,
                      bool isManaged,
                      bool isVisible,
                      bool isNotifyOnPopupStatus)
{
   // Allow initialization only on clean (reset) objects
   bool lRc = (0 == mId);
   if (lRc)
   {
      SetId(id);
      SetWidth(width);
      SetHeight(height);
      SetXPos(xPos);
      SetYPos(yPos);
      SetPopupArea(isPopupArea);
      SetDisplayId(displayId);
      SetDisplayAlias(displayAlias);
      SetManaged(isManaged);
      SetVisible(isVisible);
      SetNotifyOnPopupStatus(isNotifyOnPopupStatus);
   }
   else
   {
      ETG_TRACE_ERR(("Screen area object (ID:%u) was already initialized. Failed to initialize screen area (ID:%u)",
                     mId,
                     id));
   }

   return lRc;
}


// ------------------------------------------------------------------------
void ScreenArea::ReInit(const ScreenArea& newObj)
{
   // Allow re-initialization only on valid objects
   bool lRc = (0 != mId);
   if (lRc)
   {
      if (mDisplayId != newObj.mDisplayId)
      {
         SetDisplayId(newObj.mDisplayId);
      }
      if (mXPos != newObj.mXPos)
      {
         SetXPos(newObj.mXPos);
      }
      if (mYPos != newObj.mYPos)
      {
         SetYPos(newObj.mYPos);
      }
      if (mWidth != newObj.mWidth)
      {
         SetWidth(newObj.mWidth);
      }
      if (mHeight != newObj.mHeight)
      {
         SetHeight(newObj.mHeight);
      }
      if (mIsPopupArea != newObj.mIsPopupArea)
      {
         SetPopupArea(newObj.mIsPopupArea);
      }
      if (mIsManaged != newObj.mIsManaged)
      {
         SetManaged(newObj.mIsManaged);
      }
      if (mIsVisible != newObj.mIsVisible)
      {
         SetVisible(newObj.mIsVisible);
      }
      if (mIsNotifyOnPopupStatus != newObj.mIsNotifyOnPopupStatus)
      {
         SetNotifyOnPopupStatus(newObj.mIsNotifyOnPopupStatus);
      }
   }
}


// ------------------------------------------------------------------------
bool ScreenArea::InitializeLayering()
{
   //SCREENBROKER_LOG_FN();
   bool lRc = false;
   if (mIsManaged)
   {
      if (0 == mLayerId)
      {
         // Use screen area ID as source value for layer ID
         UInt32 lLayerId = mId;
         lRc = Internal::LayerManagerAccessor::CreateLayer(lLayerId,
               mWidth,
               mHeight,
               mXPos,
               mYPos,
               mIsVisible);

         if (lRc)
         {
            ETG_TRACE_USR1(("Created ILM layer %u for screen area ID:%u",
                            lLayerId,
                            mId));
            mLayerId = mId;
         }
         else
         {
            ETG_TRACE_ERR(("Failed to create ILM layer %u for screen area ID:%u",
                           lLayerId,
                           mId));
         }
      }
      else
      {
         ETG_TRACE_SYS(("Screen area already got an associated layer (ID:%u)!", mLayerId));
      }
   }
   else
   {
      mLayerId = mId;
      lRc = true;
   }

   return lRc;
}


}
