/* ***************************************************************************************
* FILE:          PopupArea.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupArea.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_PopupManager_PopupArea_h)
#define ScreenBroker_Service_PopupManager_PopupArea_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Service/PopupManager/PopupState.h>
#include <ScreenBroker/Service/ScreenArea.h>
#include <set>
#include <vector>

namespace ScreenBroker {
namespace Internal {
/// @addtogroup PopupManager
/// @{

/**
 * @brief The PopupArea class defines a screen area which is dedicated for popups and manages the
 * list of popups per (popup) screen area.
 */
class PopupArea
{
   public:
      /**
       * @brief Constructor initializes the popup area with given screen area ID.
       *
       * @note A screen area ID 0 marks the screen area as invalid (should not be set explicitely).
       *
       * @param screenAreaId ID of the screen area this popup area is linked to.
       */
      PopupArea(UInt32 screenAreaId = 0);

      ///
      ~PopupArea();

      /**
       * @brief Resets the internal status of popup area.
       */
      void Reset();

      /**
       * @brief Provides the ID of the screen area this popup area is linked to.
       *
       * @return ID of screen area.
       */
      UInt32 GetScreenAreaId() const
      {
         return mScreenAreaId;
      }

      /**
       * @brief Informs about the number of popup states currently held in popup area.
       *
       * @return Number of active popup states for this popup area.
       */
      UInt32 GetPopupCount() const;

      /**
       * @brief Adds the given popup state to the pop-up list at the given position.
       *
       * If the position exceeds the length of the queue it is appended at the end of the list.
       *
       * @param popup Popup state to add.
       * @param position Position where the popup has to be added.
       */
      void AddPopupAt(PopupState& popup, UInt32 position);

      /**
       * @brief Adds the given popup state to the end of the pop-up list.
       *
       * @param popup Popup state to add.
       */
      void AddPopup(PopupState& popup);

      /**
       * @brief Provides the popup state at the given position.
       *
       * @param position Requested position.
       * @return The requested popup state object at given position.
       */
      PopupState* GetPopupAt(UInt32 position);

      /**
       * @brief Searches for the popup state in the popup list of popup state's popup area.
       *
       * @param popupState Popup state to search.
       * @param position Position of the popup state found in the popup list of of popup state's popup area
       *                 (if position pointer is 0, then this out parameter is ignored).
       * @return The requested popup state found in the popup list of of popup state's popup area or
       *         0 if not found (in this case position returned is not valid).
       */
      PopupState* FindPopup(PopupState& popupState, UInt32* position);

      /**
       * @brief Removes the popup state at the given position from the pop-up list.
       *
       * @param position Position of the popup to remove.
       */
      void RemovePopupAt(UInt32 position);

      /**
       * @brief Removes the popup state from the pop-up list.
       *
       * @param popupState Popup state to remove.
       */
      void RemovePopup(PopupState& popupState);

      /**
       * @brief Provides the first popup state with a timestamp less than the given timestamp
       * or null if no further popup is available.
       *
       * @param timestamp Current timestamp.
       *
       * @return First expired popup state.
       */
      PopupState* GetFirstExpiredPopupState(UInt64 timestamp);

      /**
       * @brief Provides the next popup state with a timestamp less than the given timestamp or
       * null if no further popup is available.
       *
       * @param timestamp Current timestamp.
       *
       * @return Next expired popup state.
       */
      PopupState* GetNextExpiredPopupState();

   private:
      typedef std::vector<PopupState*> PopupStateVector;
      typedef std::set<PopupState*> PopupStateExpirationSet;

      /// Vector of current active popup states
      PopupStateVector mPopupStateVector;

      /// Set of active popup states sorted after their expiration
      PopupStateExpirationSet mPopupStateExpirationSet;

      /// Iterator to walk through popup state expiration list created in GetFirstExpiredPopupState()
      PopupStateExpirationSet::iterator mPopupStateExpirationSetIterator;

      /// Popup screen area ID
      UInt32 mScreenAreaId;
};


/// @}
}


}
#endif
