/* ***************************************************************************************
* FILE:          PopupArea.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupArea.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "PopupArea.h"

#include <algorithm>
#include <iterator>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/PopupArea.cpp.trc.h"
#endif


namespace ScreenBroker {
namespace Internal {
// SCREENBROKER_LOG_SET_REALM(LogRealm::PopupManager);

// ------------------------------------------------------------------------
PopupArea::PopupArea(UInt32 screenAreaId) : mScreenAreaId(screenAreaId)
{
   mPopupStateExpirationSetIterator = mPopupStateExpirationSet.end();
}


// ------------------------------------------------------------------------
PopupArea::~PopupArea()
{
   Reset();
}


// ------------------------------------------------------------------------
void PopupArea::Reset()
{
   mPopupStateExpirationSet.clear();
   mPopupStateVector.clear();
}


// ------------------------------------------------------------------------
UInt32 PopupArea::GetPopupCount() const
{
   return UInt32(mPopupStateVector.size());
}


// ------------------------------------------------------------------------
void PopupArea::AddPopup(PopupState& popupState)
{
   mPopupStateVector.push_back(&popupState);
}


// ------------------------------------------------------------------------
void PopupArea::AddPopupAt(PopupState& popupState,
                           UInt32 position)
{
   if (position < mPopupStateVector.size())
   {
      mPopupStateVector.insert(mPopupStateVector.begin().operator + (static_cast<int>(position)), &popupState);
   }
   else
   {
      mPopupStateVector.push_back(&popupState);
   }
}


// ------------------------------------------------------------------------
PopupState* PopupArea::GetPopupAt(UInt32 position)
{
   if (position < mPopupStateVector.size())
   {
      return mPopupStateVector[position];
   }
   else
   {
      return 0;
   }
}


// ------------------------------------------------------------------------
PopupState* PopupArea::FindPopup(PopupState& popupState, UInt32* position)
{
   PopupState* foundPopupState = 0;

   // Find popup state in list
   PopupStateVector::iterator it = find(mPopupStateVector.begin(), mPopupStateVector.end(), &popupState);

   // Found list element
   if (mPopupStateVector.end() != it)
   {
      if (0 != position)
      {
         *position = static_cast<UInt32>(-1); // Reset position value to "Invalid position"
         *position = static_cast<UInt32>(std::distance(mPopupStateVector.begin(), it));
      }
      foundPopupState = (*it);
   }

   return foundPopupState;
}


// ------------------------------------------------------------------------
void PopupArea::RemovePopupAt(UInt32 position)
{
   if (position < mPopupStateVector.size())
   {
      mPopupStateVector.erase(mPopupStateVector.begin().operator + (static_cast<int>(position)));
   }
}


// ------------------------------------------------------------------------
void PopupArea::RemovePopup(PopupState& popupState)
{
   // Find popup state in list
   PopupStateVector::iterator it = find(mPopupStateVector.begin(), mPopupStateVector.end(), &popupState);

   // Erase found list element
   if (mPopupStateVector.end() != it)
   {
      mPopupStateVector.erase(it);
   }
}


// ------------------------------------------------------------------------
PopupState* PopupArea::GetFirstExpiredPopupState(UInt64 timestamp)
{
   // Clear expiration list first
   mPopupStateExpirationSet.clear();

   // Re-generate expiration list based on existing popup state vector
   for (PopupStateVector::const_iterator it = mPopupStateVector.begin();
         mPopupStateVector.end() != it;
         ++it)
   {
      PopupState* lPopupState = (*it);

      // If time-based update value is set, calculate next timer expiration
      if ((0 != lPopupState) && (0 != lPopupState->GetNextUpdatePeriod()))
      {
         Int32 lPassedTime = static_cast<Int32>(timestamp - lPopupState->GetNextUpdateTimeBase());
         Int32 lExpirationPeriod = static_cast<Int32>(lPopupState->GetNextUpdatePeriod() - static_cast<UInt32>(lPassedTime));

         ETG_TRACE_USR4(("surfaceId %u (userData: 0x%08X): lPassedTime: %dms, lExpirationPeriod: %dms",
                         lPopupState->GetSurfaceId(),
                         lPopupState->GetUserData(),
                         lPassedTime,
                         lExpirationPeriod));

         // Add popup to expired popup state set, if popup is expired
         if (lExpirationPeriod <= 0)
         {
            mPopupStateExpirationSet.insert(lPopupState);
         }
      }
   }

   ETG_TRACE_USR1(("Total number of expired popups: %d", mPopupStateExpirationSet.size()));

   // Set popup state expiration list iterator pointing to first element
   mPopupStateExpirationSetIterator = mPopupStateExpirationSet.begin();

   // Retrieve first expired popup state
   return GetNextExpiredPopupState();
}


// ------------------------------------------------------------------------
PopupState* PopupArea::GetNextExpiredPopupState()
{
   PopupState* lPopupState = 0;
   // Retrieve and remove next expired popup state
   if (mPopupStateExpirationSet.end().operator != (mPopupStateExpirationSetIterator))
   {
      lPopupState = (*mPopupStateExpirationSetIterator);
      if (0 != lPopupState)
      {
         ETG_TRACE_USR1(("Popup with surface ID:%u has expired",
                         lPopupState->GetSurfaceId()));
      }
      ++mPopupStateExpirationSetIterator;
   }

   return lPopupState;
}


}
}
