/* ***************************************************************************************
* FILE:          PluginManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PluginManager.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Using the "do { ... } while(0) construct within macros is a common practice in order
// to make the use of macros safe within conditions.
//lint -emacro({717}, BIND_PLUGIN)    "do ... while(0);"
//lint -emacro({717}, LOAD_AND_BIND_PLUGIN)    "do ... while(0);"
// =============================================================================

#include "PluginManager.h"

#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <ScreenBroker/Plugin/IConfigurator.h>

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include "LayerManagerAccessor.h"
#endif
#include <map>
#include <list>
#include <vector>
#include <string>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/PluginManager.cpp.trc.h"
#endif

#include <dlfcn.h>

#include "ScreenBrokerActivator/ScreenBrokerActivator.h"
#include "PopupManagerActivator/PopupManagerActivator.h"
#include "ScreenLayouter/ScreenLayouter.h"
#ifdef SCREENBROKER_PLUGIN_KEYHANDLING
#include "KeyHandler/KeyHandler.h"
#endif
using namespace std;

/// Macro for binding screen broker plugins
#define BIND_PLUGIN(pluginName)                                                             \
    do {                                                                                    \
        mPlugins[PluginId::pluginName] = LoadPlugin<I##pluginName>();                       \
        lRc = (0 != mPlugins[PluginId::pluginName]);                                        \
        if (lRc) {                                                                          \
            lRc = (mPlugins[PluginId::pluginName]->Init());                                 \
            if (lRc) {                                                                      \
                /* not required here: ETG_TRACE_USR1(("Binding plugin '%40s' succeeded!", #pluginName)); */        \
            } else {                                                                        \
                /* not required here: ETG_TRACE_ERR(("Initializing plugin '%40s' failed!", #pluginName)); */     \
            }                                                                               \
        } else {                                                                            \
            /* not required here: ETG_TRACE_ERR(("Loading plugin '%40s' failed!", #pluginName)); */              \
        }                                                                                   \
    } while(0)

#define LOAD_AND_BIND_PLUGIN(pluginName)                                                            \
    do {                                                                                            \
        lRc = LoadSharedLibrary(pluginName##Library());                                             \
        if (lRc) {                                                                                  \
            BIND_PLUGIN(pluginName);                                                                \
        } else {                                                                                    \
            /* not required here: ETG_TRACE_ERR(("Loading shared library '%40s' failed!", pluginName##Library())); */    \
        }                                                                                           \
    } while(0)


#ifndef SCREENBROKER_DYNAMIC_PLUGIN
/// Macro for binding screen broker plugins
#define UPDATE_PLUGIN_FACTORY(pluginName)                                                             \
    do {                                                                                    \
        gScreenBrokerPluginFactoryMap[ScreenBroker::I##pluginName::Name()] = ScreenBroker::pluginName::Create; \
    } while(0)
#endif
// ----------------------------------------------------------------------------
// Definition of global factory map for plugins
ScreenBrokerPluginFactoryMap gScreenBrokerPluginFactoryMap;
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/PluginManager.cpp.trc.h"
#endif


namespace ScreenBroker {
namespace Internal {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::PluginManager);

// ------------------------------------------------------------------------
PluginManager& PluginManager::GetInstance()
{
   // Lazy initialization of screen broker service singleton
   static PluginManager lPluginManager;
   return lPluginManager;
}


// ------------------------------------------------------------------------
PluginManager::PluginManager()
{
}


// ------------------------------------------------------------------------
PluginManager::~PluginManager()
{
   mPlugins.clear();
}


// ------------------------------------------------------------------------
bool PluginManager::Load()
{
   bool lRc = false;
#ifdef SCREENBROKER_DYNAMIC_PLUGIN
   //SCREENBROKER_LOG_FN();
   lRc = LoadSharedLibrary(ScreenBrokerPluginLibrary());

   // Screen broker plugin container library is found
   if (lRc)
   {
      BIND_PLUGIN(ScreenBrokerActivator);
      BIND_PLUGIN(PopupManagerActivator);
      BIND_PLUGIN(ScreenLayouter);
#ifdef SCREENBROKER_PLUGIN_KEYHANDLING
      BIND_PLUGIN(KeyHandler);
#endif
   }
   // No screen broker plugin library, which holds all plugins, is found -> load each plugin separately
   else
   {
      ETG_TRACE_USR1(("No screen broker container plugin library '%40s' found, load each plugin separately!", ScreenBrokerPluginLibrary()));
      LOAD_AND_BIND_PLUGIN(ScreenBrokerActivator);
      LOAD_AND_BIND_PLUGIN(PopupManagerActivator);
      LOAD_AND_BIND_PLUGIN(ScreenLayouter);
#ifdef SCREENBROKER_PLUGIN_KEYHANDLING
      LOAD_AND_BIND_PLUGIN(KeyHandler);
#endif
   }
#else
   //Update class details to global map
   UPDATE_PLUGIN_FACTORY(ScreenBrokerActivator);
   UPDATE_PLUGIN_FACTORY(PopupManagerActivator);
   UPDATE_PLUGIN_FACTORY(ScreenLayouter);

   LOAD_AND_BIND_PLUGIN(Configurator);

   BIND_PLUGIN(ScreenBrokerActivator);
   BIND_PLUGIN(PopupManagerActivator);
   BIND_PLUGIN(ScreenLayouter);
#ifdef SCREENBROKER_PLUGIN_KEYHANDLING
   UPDATE_PLUGIN_FACTORY(KeyHandler);
   BIND_PLUGIN(KeyHandler);
#endif
#endif
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   LayerManagerAccessor::RegisterWithIlm();
#endif

   return lRc;
}


// ------------------------------------------------------------------------
void PluginManager::Unload()
{
   //SCREENBROKER_LOG_FN();
   // Unload plugins in reverse add order
   for (PluginMap::reverse_iterator rit = mPlugins.rbegin();
         rit != mPlugins.rend();
         ++rit)
   {
      if (0 != (*rit).second)
      {
         (*rit).second->Reset();
      }
   }

   // Close shared libraries
   for (SharedLibraryList::iterator it = mSharedLibraryList.begin();
         it != mSharedLibraryList.end();
         ++it)
   {
      SharedLibraryContext lContext = (*it);
      ETG_TRACE_USR1(("Unloading shared library %40s", lContext.name));
      dlclose(lContext.handle);
   }
}


// ------------------------------------------------------------------------
IScreenBrokerPlugin* PluginManager::GetPlugin(PluginId::Enum pluginId)
{
   PluginMap::iterator it = mPlugins.find(pluginId);
   return (mPlugins.end().operator != (it)) ? (*it).second : 0;
}


// ------------------------------------------------------------------------
template <typename T> T* PluginManager::LoadPlugin()
{
   //SCREENBROKER_LOG_FN();
   // Plugin creation (if plugin exists)
   IScreenBrokerPlugin::Create* lPluginFactory = gScreenBrokerPluginFactoryMap[T::Name()];
   if (0 != lPluginFactory)
   {
      T& lPlugin = static_cast<T&>(lPluginFactory());
      ETG_TRACE_USR4(("Created plugin object '%40s'", T::Name()));
      return &lPlugin;
   }
   else
   {
      ETG_TRACE_SYS(("Failed to create plugin object '%40s'", T::Name()));
      return 0;
   }
}


// ------------------------------------------------------------------------
bool PluginManager::LoadSharedLibrary(const Char* sharedLibraryName)
{
   //SCREENBROKER_LOG_FN();
   /* not required here: ETG_TRACE_USR4(("Loading shared library %40s", sharedLibraryName)); */

   // Open shared library and bind all undefined symbols immediately (needed for plugin factory access)
   void* lHandle = dlopen(sharedLibraryName, RTLD_NOW);
   SharedLibraryContext context = {lHandle, sharedLibraryName };
   bool lRc = (0 != context.handle);
   if (lRc)
   {
      // Add the handle to list
      mSharedLibraryList.push_back(context);
   }
   else
   {
      ETG_TRACE_ERR(("Failed to load shared library %30s! %40s!", sharedLibraryName, dlerror()));
   }
   return lRc;
}


}
}
