/* ***************************************************************************************
* FILE:          LayerManagerAccessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LayerManagerAccessor.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Using the "do { ... } while(0) construct within macros is a common practice in order
// to make the use of macros safe within conditions.
//lint -emacro({717}, SAFE_FREE)    "do ... while(0);"
// =============================================================================

#include "LayerManagerAccessor.h"
#include "PluginManager.h"

#ifndef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include <ilm/ilm_client.h>
#endif
#if !defined(LAYERMANAGER_VERSION_0_9_9)
#include <ilm/ilm_control.h>
#endif
#include <stdlib.h>

///
#define SAFE_FREE(array)                \
    do {                                \
        if (0 != array) free(array);    \
        array = 0;                      \
    } while(0)
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/LayerManagerAccessor.cpp.trc.h"
#endif


namespace ScreenBroker {
namespace Internal {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::LayerManagerAccessor);

// ------------------------------------------------------------------------
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#define ILM_SURFACE_CONFIGURATION_TIMEOUT 100
CriticalSection LayerManagerAccessor::mCsl;
std::list<UInt32> LayerManagerAccessor::configuredSurfaceList;
UInt32 LayerManagerAccessor::validatingSurfaceID = 0;
::hmibase::util::Condition LayerManagerAccessor::mCondition;

#endif

bool LayerManagerAccessor::mInitialized = false;

// ------------------------------------------------------------------------
bool LayerManagerAccessor::Init()
{
   //SCREENBROKER_LOG_FN();
   mInitialized = true;
   if (ILM_SUCCESS != ilm_init())
   {
      ETG_TRACE_ERR(("Layer manager client could not be initialized."));
      return false;
   }
   return true;
}


// ------------------------------------------------------------------------
void LayerManagerAccessor::Destroy()
{
   //SCREENBROKER_LOG_FN();
   if (mInitialized)
   {
      (void) ilm_destroy();
   }
   mInitialized = false;
}


// ------------------------------------------------------------------------
bool LayerManagerAccessor::CreateLayer(UInt32 layerId,
                                       UInt32 width,
                                       UInt32 height,
                                       UInt32 xPos,
                                       UInt32 yPos,
                                       bool visible)
{
   //SCREENBROKER_LOG_FN();
   bool lRc = true;
   if (mInitialized)
   {
      // Do not accept layerId 0 as it will request a new layerId from ILM
      if (0 == layerId)
      {
         ETG_TRACE_ERR(("ILM cannot create layer with ID 0!"));
         return false;
      }

      // Request ILM layer
      t_ilm_layer lLayerId = static_cast<t_ilm_layer>(layerId);
      lRc = (ILM_SUCCESS == ilm_layerCreateWithDimension(&lLayerId,
             width,
             height));

      // Set position
      lRc = lRc && (ILM_SUCCESS == ilm_layerSetDestinationRectangle(lLayerId, xPos, yPos, width, height));

      // Set visibility
      lRc = lRc && (ILM_SUCCESS == ilm_layerSetVisibility(lLayerId, visible));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool LayerManagerAccessor::RemoveLayer(UInt32 layerId)
{
   //SCREENBROKER_LOG_FN();
   bool lRc = true;
   if (mInitialized)
   {
      // Remove ILM layer
      lRc = (ILM_SUCCESS == ilm_layerRemove(static_cast<t_ilm_layer>(layerId)));

      // Commit changes to ILM
      lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool LayerManagerAccessor::AddSurfaceToLayer(UInt32 layerId,
      UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   bool lRc = true;
   if (mInitialized)
   {
      // Adds surface to given layer
      lRc = (ILM_SUCCESS == ilm_layerAddSurface(layerId, surfaceId));
   }

   return lRc;
}


// ------------------------------------------------------------------------
bool LayerManagerAccessor::RemoveSurfaceFromLayer(UInt32 layerId,
      UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   bool lRc = true;
   if (mInitialized)
   {
      // Remove surface to given layer
      lRc = (ILM_SUCCESS == ilm_layerRemoveSurface(layerId, surfaceId));

      // Commit changes to ILM
      lRc = lRc && (ILM_SUCCESS == ilm_commitChanges());
   }
   return lRc;
}


// ------------------------------------------------------------------------
void LayerManagerAccessor::Dump()
{
   //SCREENBROKER_LOG_FN();
   if (mInitialized)
   {
      ETG_TRACE_USR4(("Dump ILM status"));
      t_ilm_uint pDLength;
      t_ilm_uint* pDArray;
      ilmErrorTypes rcGetScreen = ilm_getScreenIDs(&pDLength, &pDArray);
      if (ILM_SUCCESS == rcGetScreen)
      {
         ETG_TRACE_USR4(("ILM: %u displays found", pDLength));
         for (Int d = 0; d < Int(pDLength); ++d)
         {
            ETG_TRACE_USR4(("  ILM display ID:%u found", pDArray[d]));
         }
      }
      else
      {
         ETG_TRACE_USR4(("ILM: no displays found"));
      }

      // Read all available layers
      t_ilm_int pALLength;
      t_ilm_layer* ppALArray;
      if (ILM_SUCCESS == ilm_getLayerIDs(&pALLength, &ppALArray))
      {
         ETG_TRACE_USR4(("ILM: %u layers found", pALLength));
         for (Int i = 0; i < pALLength; ++i)
         {
            ETG_TRACE_USR4(("  ILM layer ID:%u found", ppALArray[i]));
         }
         SAFE_FREE(ppALArray);
      }
      else
      {
         ETG_TRACE_USR4(("ILM: no layers found"));
      }

      // Read all available surfaces
      t_ilm_int pASLength;
      t_ilm_layer* ppASArray;
      if (ILM_SUCCESS == ilm_getSurfaceIDs(&pASLength, &ppASArray))
      {
         ETG_TRACE_USR4(("ILM: %u surfaces found", pASLength));
         for (Int l = 0; l < pASLength; ++l)
         {
            ETG_TRACE_USR4(("  ILM surface ID:%u found", ppASArray[l]));
         }
         SAFE_FREE(ppASArray);
      }
      else
      {
         ETG_TRACE_USR4(("ILM: no surfaces found"));
      }

      // Read available displays
      if (ILM_SUCCESS == rcGetScreen)
      {
         ETG_TRACE_USR4(("ILM: %u displays found", pDLength));
         for (Int d = 0; d < Int(pDLength); ++d)
         {
            ETG_TRACE_USR4(("ILM display ID:%u found", pDArray[d]));

            // Read available layers on display
            t_ilm_int pLLength = 0;
            t_ilm_layer* ppLArray;
            if (ILM_SUCCESS == ilm_getLayerIDsOnScreen(pDArray[d], &pLLength, &ppLArray))
            {
               ETG_TRACE_USR4((" ILM: %u layers on display %u found", pLLength, pDArray[d]));
               for (Int l = 0; l < pLLength; ++l)
               {
                  ETG_TRACE_USR4((" ILM: layer %u on display %u found", ppLArray[l], pDArray[d]));
               }
               SAFE_FREE(ppLArray);
            }
            else
            {
               ETG_TRACE_USR4((" ILM: no layers found on display %u", pDArray[d]));
            }
         }
         SAFE_FREE(pDArray);
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
void LayerManagerAccessor::RegisterWithIlm()
{
   //SCREENBROKER_LOG_FN();
   if (ILM_SUCCESS != ilm_registerNotification(RegistercallbackFunction, NULL))
   {
      ETG_TRACE_ERR_THR(("registerNotification with ILM Failed."));
   }
}


void LayerManagerAccessor::ConfigureILMSurface(t_ilm_uint id, struct ilmSurfaceProperties* sp)
{
   //SCREENBROKER_LOG_FN();
   if (ILM_SUCCESS != ilm_surfaceSetDestinationRectangle(id, 0, 0, sp->origSourceWidth, sp->origSourceHeight))
   {
      ETG_TRACE_ERR_THR(("ConfigureILMSurface : surfaceSetDestinationRectangle with ILM Failed."));
   }
   if (ILM_SUCCESS != ilm_surfaceSetSourceRectangle(id, 0, 0, sp->origSourceWidth, sp->origSourceHeight))
   {
      ETG_TRACE_ERR_THR(("ConfigureILMSurface : surfaceSetSourceRectangle with ILM Failed."));
   }
   if (ILM_SUCCESS != ilm_surfaceSetOpacity(id, 1))
   {
      ETG_TRACE_ERR_THR(("ConfigureILMSurface : surfaceSetOpacity with ILM Failed."));
   }
   IScreenLayouter* lScreenLayouter = PluginManager::GetInstance().Plugin<IScreenLayouter>();
   if (0 != lScreenLayouter)
   {
      UInt32 lLayerId;
      bool lIsSurfaceObserved = false;
      //get the layer info of the surface from plugin
      lIsSurfaceObserved = lScreenLayouter->IsSurfaceIsObserved(UInt32(id));
      if (lIsSurfaceObserved)
      {
         lLayerId = lScreenLayouter->GetScreenAreaIdOfSurface(UInt32(id));
         // Adds given surface to given layer
         ScreenBroker::Internal::LayerManagerAccessor::AddSurfaceToLayer(lLayerId, id);

         if (ILM_SUCCESS != ilm_surfaceSetVisibility(id, 1))
         {
            ETG_TRACE_ERR_THR(("ConfigureILMSurface : surfaceSetVisibility with ILM Failed."));
         }
      }
      else
      {
         ETG_TRACE_USR4_THR(("ConfigureILMSurface: surface %d is not observed to add it to layer", id));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("ConfigureILMSurface: ScreenLayouter-NULL."));
   }
   if (ILM_SUCCESS != ilm_commitChanges())
   {
      ETG_TRACE_ERR_THR(("ConfigureILMSurface : ILM commit Failed."));
   }
   else
   {
      pushToCofiguredSurfaceList(id);
      if (LayerManagerAccessor::getValidatingSurfaceId() == UInt32(id))
      {
         if (mCondition.signal())
         {
            ETG_TRACE_USR4_THR(("Condition Wait Signaled"));
         }
      }
   }
}


void LayerManagerAccessor::RegistercallbackFunction(ilmObjectType object, t_ilm_uint id, t_ilm_bool created, void* user_data)
{
   //SCREENBROKER_LOG_FN();
   (void)user_data;

   if (object == ILM_SURFACE)
   {
      if (created)
      {
         struct ilmSurfaceProperties sp;
         if (ILM_SUCCESS == ilm_getPropertiesOfSurface(id, &sp))
         {
            if ((sp.origSourceWidth != 0) && (sp.origSourceHeight != 0))
            {
               ScreenBroker::Internal::LayerManagerAccessor::ConfigureILMSurface(id, &sp);
            }
            else
            {
               if (ILM_SUCCESS != ilm_surfaceAddNotification(id, SurfaceCallbackFunction))
               {
                  ETG_TRACE_ERR_THR(("RegistercallbackFunction : surfaceAddNotification with ILM Failed."));
               }
               if (ILM_SUCCESS != ilm_commitChanges())
               {
                  ETG_TRACE_ERR_THR(("RegistercallbackFunction : ILM commit Failed."));
               }
            }
         }
         else
         {
            ETG_TRACE_ERR_THR(("ilm_getPropertiesOfSurface : Failed."));
         }
      }
   }
}


void LayerManagerAccessor::SurfaceCallbackFunction(t_ilm_uint id, struct ilmSurfaceProperties* sp, t_ilm_notification_mask m)
{
   ETG_TRACE_USR4_THR(("SurfaceCallbackFunction notification mask[%u]", m));
   //SCREENBROKER_LOG_FN();
   if ((unsigned)m & ILM_NOTIFICATION_CONFIGURED)
   {
      ScreenBroker::Internal::LayerManagerAccessor::ConfigureILMSurface(id, sp);
   }
}


bool LayerManagerAccessor::isSurfaceConfigured(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&Internal::LayerManagerAccessor::mCsl);
   bool configured = false;
   std::list<UInt32>::iterator iter = Internal::LayerManagerAccessor::configuredSurfaceList.begin();
   while (iter != Internal::LayerManagerAccessor::configuredSurfaceList.end())
   {
      if (surfaceId == (*iter))
      {
         configured = true;
         break;
      }
      else
      {
         ++iter;
      }
   }
   ETG_TRACE_USR4_THR(("Surface is Configured:%d", configured));
   return configured;
}


void LayerManagerAccessor::pushToCofiguredSurfaceList(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&mCsl);
   std::list<UInt32>::iterator iter = configuredSurfaceList.begin();
   while (iter != configuredSurfaceList.end())
   {
      if (surfaceId == (*iter))
      {
         break;
      }
      else
      {
         ++iter;
      }
   }
   if (iter == configuredSurfaceList.end())
   {
      configuredSurfaceList.push_back(surfaceId);
      ETG_TRACE_USR4_THR(("Pushed Surface in Configured List[%u]", surfaceId));
   }
}


void LayerManagerAccessor::popFromCofiguredSurfaceList(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&mCsl);
   std::list<UInt32>::iterator iter = configuredSurfaceList.begin();
   while (iter != configuredSurfaceList.end())
   {
      if (surfaceId == (*iter))
      {
         configuredSurfaceList.erase(iter);
         ETG_TRACE_USR4_THR(("Poped Surface from Configured List[%u]", surfaceId));
         return;
      }
      else
      {
         ++iter;
      }
   }
}


void LayerManagerAccessor::clearCofiguredSurfaceList()
{
   CriticalSectionLocker lLock(&mCsl);
   configuredSurfaceList.clear();
}


void LayerManagerAccessor::setValidatingSurfaceId(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&mCsl);
   validatingSurfaceID = surfaceId;
}


UInt32 LayerManagerAccessor::getValidatingSurfaceId()
{
   CriticalSectionLocker lLock(&mCsl);
   return validatingSurfaceID;
}


void LayerManagerAccessor::clearValidatingSurfaceId()
{
   LayerManagerAccessor::setValidatingSurfaceId(0);
}


bool LayerManagerAccessor::isValidSurface(UInt32 surfaceId)
{
   if (!LayerManagerAccessor::isSurfaceConfigured(surfaceId))
   {
      LayerManagerAccessor::setValidatingSurfaceId(surfaceId);
      if (0 == mCondition.wait((uint32_t)ILM_SURFACE_CONFIGURATION_TIMEOUT))
      {
         ETG_TRACE_ERR(("Requested Surface ID : %d Validation Timed Out", surfaceId));
         return false;
      }
      if (LayerManagerAccessor::getValidatingSurfaceId() == surfaceId)
      {
         LayerManagerAccessor::clearValidatingSurfaceId();
      }
   }
   return LayerManagerAccessor::isSurfaceConfigured(surfaceId);
}


#endif
}


}
