/* ***************************************************************************************
* FILE:          ClientState.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ClientState.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "ClientState.h"

#include <ScreenBroker/RequestArg.h>
#include <ScreenBroker/Service/PopupManager/PopupState.h>
#include <ScreenBroker/Service/PluginManager.h>
#include <ScreenBroker/Service/Service.h>
#include <ScreenBroker/Service/ServiceAdaptor.h>
#include <ScreenBroker/Service/LayerManagerAccessor.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/ClientState.cpp.trc.h"
#endif


namespace ScreenBroker {
namespace Internal {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenBrokerService);

// ------------------------------------------------------------------------
ClientState::ClientState()
{
}


// ------------------------------------------------------------------------
ClientState::ClientState(const std::string& clientId) : mClientId(clientId)
{
}


// ------------------------------------------------------------------------
ClientState::~ClientState()
{
   //SCREENBROKER_LOG_FN();
   // Deactivate all surfaces (remove link to layers) of client
   for (SurfaceContextMap::iterator it = mSurfaceContextMap.begin();
         mSurfaceContextMap.end().operator != (it);
         ++it)
   {
      UInt32 lSurfaceId = (*it).first;
      const ScreenArea* lScreenArea = (*it).second.ScreenArea();
      // Check if surface context entry is valid
      if (0 != lScreenArea)
      {
         ETG_TRACE_USR4(("Client '%10s': Request deactivation of surface %u from screen area %u at plugin",
                         mClientId.c_str(),
                         lSurfaceId,
                         lScreenArea->GetId()));

         // Removes surface from given layer
         bool lRc = LayerManagerAccessor::RemoveSurfaceFromLayer(lScreenArea->GetLayerId(), lSurfaceId);
         if (lRc)
         {
            ETG_TRACE_USR1(("Surface %d removed from layer %d", lSurfaceId, lScreenArea->GetLayerId()));
         }
         else
         {
            ETG_TRACE_SYS(("Failed to remove surface %d on layer %d", lSurfaceId, lScreenArea->GetLayerId()));
         }
      }
   }
   mSurfaceContextMap.clear();
}


// ------------------------------------------------------------------------
void ClientState::Reset()
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1(("Reset of client state %40s", mClientId.c_str()));
   mPopupStateMap.clear();
   mSurfaceContextMap.clear();
}


// ------------------------------------------------------------------------
const ScreenArea* ClientState::GetScreenArea(UInt32 surfaceId)
{
   SurfaceContextMap::iterator it = mSurfaceContextMap.find(surfaceId);

   // Register surface only if it wasn't done before
   if (mSurfaceContextMap.end().operator != (it))
   {
      return (*it).second.ScreenArea();
   }

   return 0;
}


// ------------------------------------------------------------------------
void ClientState::RegisterSurface(const ServiceRequestArg& serviceRequestArg,
                                  const ScreenArea& screenArea,
                                  UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   // Register surface only if it wasn't done before
   const ScreenArea* lScreenArea = GetScreenArea(surfaceId);
   if (0 == lScreenArea)
   {
      ETG_TRACE_USR1(("Client '%10s': Register surface %u to screen area %u",
                      mClientId.c_str(),
                      surfaceId,
                      screenArea.GetId()));
      // Generate new surface context
      mSurfaceContextMap[surfaceId] = SurfaceContext(&screenArea, serviceRequestArg.UserData());
   }
   else
   {
      ETG_TRACE_SYS(("Client '%10s': Failed to register surface %u to screen area %u: surface already registered to %u!",
                     mClientId.c_str(),
                     surfaceId,
                     screenArea.GetId(),
                     lScreenArea->GetId()));
   }
}


// ------------------------------------------------------------------------
void ClientState::DeregisterSurface(UInt32 surfaceId,
                                    bool sendToIlm)
{
   //SCREENBROKER_LOG_FN();

   SurfaceContextMap::iterator it = mSurfaceContextMap.find(surfaceId);

   if (mSurfaceContextMap.end().operator != (it))
   {
      ETG_TRACE_USR4(("Client '%10s': Deregister surface %u",
                      mClientId.c_str(),
                      surfaceId));

      if (sendToIlm)
      {
         const ScreenArea* lScreenArea = GetScreenArea(surfaceId);
         if (0 != lScreenArea)
         {
            // Removes surface from given layer
            bool lRc = LayerManagerAccessor::RemoveSurfaceFromLayer(
                          lScreenArea->GetLayerId(),
                          surfaceId);
            if (lRc)
            {
               ETG_TRACE_USR1(("Surface %d removed from layer %d",
                               surfaceId,
                               lScreenArea->GetLayerId()));
            }
            else
            {
               ETG_TRACE_SYS(("Failed to remove surface %d from layer %d",
                              surfaceId,
                              lScreenArea->GetLayerId()));
            }
         }
         else
         {
            ETG_TRACE_SYS(("Failed to remove surface %d from layer: Surface not registered to a screen area",
                           surfaceId));
         }
      }

      mSurfaceContextMap.erase(it);
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': Failed to deregister surface %u: surface is not registered!",
                     mClientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ClientState::Activate(const ServiceRequestArg& serviceRequestArg,
                           UInt32 surfaceId,
                           UInt32 entryCustomAnimationType,
                           UInt32 exitCustomAnimationType)
{
   //SCREENBROKER_LOG_FN();
   IScreenBrokerActivator* lScreenBrokerActivator =
      PluginManager::GetInstance().Plugin<IScreenBrokerActivator>();

   if (0 != lScreenBrokerActivator)
   {
      // Check if surface is registered
      const ScreenArea* lScreenArea = GetScreenArea(surfaceId);
      if (0 != lScreenArea)
      {
         if (!lScreenArea->IsPopupArea())
         {
            UInt32 lHint = mSurfaceContextMap[surfaceId].Hint();
            ETG_TRACE_USR4(("Client '%10s': Activate surface %u to screen area %u providing hint %u with entryCustomAnimationType %u and exitCustomAnimationType %u",
                            mClientId.c_str(),
                            surfaceId,
                            lScreenArea->GetId(),
                            lHint,
                            entryCustomAnimationType,
                            exitCustomAnimationType));
            lScreenBrokerActivator->Activate(serviceRequestArg,
                                             lScreenArea->GetLayerId(),
                                             surfaceId,
                                             entryCustomAnimationType,
                                             exitCustomAnimationType,
                                             lHint);
         }
         else
         {
            ETG_TRACE_ERR(("Client '%10s': Failed to activate surface %u: surface is registered at popup screen area %u!",
                           mClientId.c_str(),
                           surfaceId,
                           lScreenArea->GetId()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Failed to activate surface %u: surface is not registered!",
                        mClientId.c_str(),
                        surfaceId));
      }
   }
}


// ------------------------------------------------------------------------
void ClientState::TransitionAnimation(const ServiceRequestArg& serviceRequestArg,
                                      UInt32 surfaceId,
                                      UInt32 animationType,
                                      UInt32 hint)
{
   //SCREENBROKER_LOG_FN();
   IScreenBrokerActivator* lScreenBrokerActivator =
      PluginManager::GetInstance().Plugin<IScreenBrokerActivator>();

   if (0 != lScreenBrokerActivator)
   {
      // Check if surface is registered
      const ScreenArea* lScreenArea = GetScreenArea(surfaceId);
      if (0 != lScreenArea)
      {
         if (!lScreenArea->IsPopupArea())
         {
            ETG_TRACE_USR4(("Client '%10s': Establish animation for surface %u to screen area %u providing hint %u",
                            mClientId.c_str(),
                            surfaceId,
                            lScreenArea->GetId(),
                            hint));
            lScreenBrokerActivator->TransitionAnimation(serviceRequestArg,
                  lScreenArea->GetLayerId(),
                  surfaceId,
                  animationType,
                  hint);
         }
         else
         {
            ETG_TRACE_ERR(("Client '%10s': Failed to Establish animation for surface %u: surface is registered at popup screen area %u!",
                           mClientId.c_str(),
                           surfaceId,
                           lScreenArea->GetId()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Failed to Establish animation for surface %u: surface is not registered!",
                        mClientId.c_str(),
                        surfaceId));
      }
   }
}


// ------------------------------------------------------------------------
void ClientState::AttachandAnimateLayer(const ServiceRequestArg& serviceRequestArg,
                                        UInt32 layerId,
                                        std::vector<UInt32> surfaceIdList,
                                        UInt32 animationType,
                                        UInt32 hint)
{
   //SCREENBROKER_LOG_FN();
   IScreenBrokerActivator* lScreenBrokerActivator =
      PluginManager::GetInstance().Plugin<IScreenBrokerActivator>();

   if (0 != lScreenBrokerActivator)
   {
      lScreenBrokerActivator->TransitionAnimationOfLayer(serviceRequestArg, layerId, surfaceIdList, animationType, hint);
   }
}


// ------------------------------------------------------------------------
void ClientState::ShowPopup(const ServiceRequestArg& serviceRequestArg,
                            UInt32 surfaceId,
                            const PopupPresentationArg& popupPresentationArg)
{
   //SCREENBROKER_LOG_FN();
   IPopupManagerActivator* lPopupManagerActivator =
      PluginManager::GetInstance().Plugin<IPopupManagerActivator>();

   if (0 != lPopupManagerActivator)
   {
      // Check if surface is registered on a popup dedicated screen area
      const ScreenArea* lScreenArea = GetScreenArea(surfaceId);
      if (0 != lScreenArea)
      {
         if (lScreenArea->IsPopupArea())
         {
            // First search if popup state with given surface ID is already registered, else
            // retrieve popup state with given surface ID (if not yet exists, a new one will be created)
            PopupStateMap::iterator it = mPopupStateMap.find(surfaceId);
            bool lCreatePopupState = (mPopupStateMap.end().operator == (it));
            PopupState& lPopupState = (lCreatePopupState) ? GetPopupState(surfaceId) : (*it).second;

            // Update popup properties
            lPopupState.ApplyProperties(lScreenArea->GetId(),
                                        lScreenArea->GetLayerId(),
                                        popupPresentationArg,
                                        serviceRequestArg);

            UInt32 lHint = mSurfaceContextMap[surfaceId].Hint();
            ETG_TRACE_USR4(("Client '%10s': Show popup with surface %u to popup screen area %u providing hint %u%40s",
                            mClientId.c_str(),
                            surfaceId,
                            lScreenArea->GetId(),
                            lHint,
                            lCreatePopupState ? " (backup available)" : ""));
            lPopupManagerActivator->ShowPopup(serviceRequestArg,
                                              lPopupState,
                                              lHint);
         }
         else
         {
            ETG_TRACE_ERR(("Client '%10s': Failed to show popup with surface %u: surface is registered at non popup screen area %u!",
                           mClientId.c_str(),
                           surfaceId,
                           lScreenArea->GetId()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Failed to show popup with surface %u: surface is not registered!",
                        mClientId.c_str(),
                        surfaceId));
      }
   }
}


// ------------------------------------------------------------------------
void ClientState::HidePopup(const ServiceRequestArg& serviceRequestArg,
                            UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   IPopupManagerActivator* lPopupManagerActivator =
      PluginManager::GetInstance().Plugin<IPopupManagerActivator>();

   if (0 != lPopupManagerActivator)
   {
      // Check if surface is registered on a popup dedicated screen area
      const ScreenArea* lScreenArea = GetScreenArea(surfaceId);
      if (0 != lScreenArea)
      {
         if (lScreenArea->IsPopupArea())
         {
            // Find registered popup state with given surface ID
            PopupStateMap::iterator it = mPopupStateMap.find(surfaceId);

            if (mPopupStateMap.end().operator != (it))
            {
               ETG_TRACE_USR4(("Client '%10s': Hide popup with surface %u to popup screen area %u",
                               mClientId.c_str(),
                               surfaceId,
                               lScreenArea->GetId()));
               lPopupManagerActivator->HidePopup(serviceRequestArg,
                                                 (*it).second);
            }
            else
            {
               ETG_TRACE_ERR(("Client '%10s': Failed to hide popup with surface %u: popup state not not found!",
                              mClientId.c_str(),
                              surfaceId));
            }
         }
         else
         {
            ETG_TRACE_ERR(("Client '%10s': Failed to hide popup with surface %u: surface is registered at non popup screen area %u!",
                           mClientId.c_str(),
                           surfaceId,
                           lScreenArea->GetId()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Failed to hide popup with surface %u: surface is not registered!",
                        mClientId.c_str(),
                        surfaceId));
      }
   }
}


// ------------------------------------------------------------------------
PopupState& ClientState::GetPopupState(UInt32 surfaceId)
{
   PopupStateMap::iterator it = mPopupStateMap.find(surfaceId);

   // If not found then create a new entry
   if (mPopupStateMap.end().operator == (it))
   {
      ETG_TRACE_USR1(("Creating popup state with surface %u", surfaceId));
      mPopupStateMap[surfaceId] = PopupState(surfaceId);
      return mPopupStateMap[surfaceId];
   }

   return (*it).second;
}


}
}
