/* ***************************************************************************************
 * FILE:          AsfServiceAdaptor.cpp
 * SW-COMPONENT:  HMI-BASE
 * DESCRIPTION:  AsfServiceAdaptor.cpp is part of HMI-Base ScreenBroker
 * COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "AsfServiceAdaptor.h"
#include <ScreenBroker/RequestArg.h>

// need this here until only one hierarchy level if supported in ETG_I_CENUM
//using namespace Screenbroker;

#include "ScreenBroker/ScreenBroker_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AsfServiceAdaptor.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::screenbroker;
using namespace ::bosch::cm::ai::hmi::screenbroker::Service;

namespace ScreenBroker {
namespace Internal {

// ========================================================================

AsfServiceAdaptor::AsfServiceAdaptor() :
   ServiceStub("SBServicePort_o")
{
   setVersion(SCREENBROKER_VERSION_STRING);
}


// ------------------------------------------------------------------------
AsfServiceAdaptor::~AsfServiceAdaptor()
{
}


// ------------------------------------------------------------------------
ServiceAdaptor* AsfServiceAdaptor::DoGetInstance()
{
   static AsfServiceAdaptor theInstance;
   return &theInstance;
}


// ------------------------------------------------------------------------
bool AsfServiceAdaptor::Init()
{
   // all things happen in Run
   return ServiceAdaptor::Init();
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::Run()
{
   // dummy interface
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::Terminate()
{
}


// ------------------------------------------------------------------------
const Char* AsfServiceAdaptor::GetServiceId() const
{
   return "ASFScreenbrokerService";
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onClientTerminationRequest(
   const ::boost::shared_ptr<ClientTerminationRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onClientTerminationRequest"));

   HandleClientTermination(request->getClientId());
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onGetDisplayIdRequest(
   const ::boost::shared_ptr<GetDisplayIdRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onGetDisplayIdRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleGetDisplayId(std::string(request->getClientId().c_str()), reqarg,
                      static_cast<UInt32>(request->getScreenAreaId()));
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onGetPreferredDimensionsRequest(
   const ::boost::shared_ptr<GetPreferredDimensionsRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onGetPreferredDimensionsRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleGetPreferredDimensions(std::string(request->getClientId().c_str()), reqarg,
                                static_cast<UInt32>(request->getScreenAreaId()));
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onRegisterSurfaceRequest(
   const ::boost::shared_ptr<RegisterSurfaceRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onRegisterSurfaceRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleRegisterSurface(
      std::string(request->getClientId().c_str()), reqarg,
      static_cast<UInt32>(request->getScreenAreaId()),
      static_cast<UInt32>(request->getSurfaceId()));
}


// ------------------------------------------------------------------------

void AsfServiceAdaptor::onDeregisterSurfaceRequest(
   const ::boost::shared_ptr<DeregisterSurfaceRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onDeregisterSurfaceRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleDeregisterSurface(
      std::string(request->getClientId().c_str()), reqarg,
      static_cast<UInt32>(request->getSurfaceId()),
      request->getSendToIlm());
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onActivateApplicationRequest(
   const ::boost::shared_ptr<ActivateApplicationRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onActivateApplicationRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleActivateApplication(
      std::string(request->getClientId().c_str()), reqarg,
      static_cast<UInt32>(request->getSurfaceId()), static_cast<UInt32>(request->getEntryCustomAnimationType()), static_cast<UInt32>(request->getExitCustomAnimationType()));
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onStartAnimationRequest(
   const ::boost::shared_ptr<StartAnimationRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onStartAnimationRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleStartAnimation(
      std::string(request->getClientId().c_str()), reqarg,
      static_cast<UInt32>(request->getSurfaceId()), static_cast<UInt32>(request->getAnimationType()),
      static_cast<UInt32>(request->getHint()));
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onAttachandAnimateLayerRequest(
   const ::boost::shared_ptr<AttachandAnimateLayerRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onAttachandAnimateLayerRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   HandleAttachandAnimateLayer(std::string(request->getClientId().c_str()), reqarg,
                               static_cast<UInt32>(request->getLayerID()),
                               static_cast<std::vector<UInt32> >(request->getSurfaceID()),
                               static_cast<UInt32>(request->getAnimationType()),
                               static_cast<UInt32>(request->getHint()));
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onShowPopupRequest(
   const ::boost::shared_ptr<ShowPopupRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onShowPopupRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   PopupPresentationArg popupPresArg(
      request->getPopupPresentationArg().getPriority(),
      (Modality::Enum) request->getPopupPresentationArg().getModality(),
      request->getPopupPresentationArg().getPresentationTime(),
      request->getPopupPresentationArg().getMinimumPresentationTime(),
      request->getPopupPresentationArg().getValidityPeriod(),
      (HorizontalAlignment::Enum) request->getPopupPresentationArg().getHorizontalAlignment(),
      (VerticalAlignment::Enum) request->getPopupPresentationArg().getVerticalAlignment(),
      (Scaling::Enum) request->getPopupPresentationArg().getScaling(),
      request->getPopupPresentationArg().getFocusPriority());

   // forward the handling to the ServiceAdaptor class
   HandleShowPopup(
      std::string(request->getClientId().c_str()), reqarg,
      static_cast<UInt32>(request->getSurfaceId()), popupPresArg);
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onHidePopupRequest(
   const ::boost::shared_ptr<HidePopupRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onHidePopupRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   // forward the handling to the ServiceAdaptor class
   HandleHidePopup(
      std::string(request->getClientId().c_str()), reqarg,
      static_cast<UInt32>(request->getSurfaceId()));
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onCloseOnExternalTouchRequest(
   const ::boost::shared_ptr<CloseOnExternalTouchRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onCloseOnExternalTouchRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   HandleCloseOnExternalTouch(
      std::string(request->getClientId().c_str()), reqarg);
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onSetPopupFilterRequest(
   const ::boost::shared_ptr<SetPopupFilterRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onSetPopupFilterRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());
   PopupPresentationArg popupPresArg(
      request->getPopupPresentationArg().getPriority(),
      (Modality::Enum) request->getPopupPresentationArg().getModality(),
      request->getPopupPresentationArg().getPresentationTime(),
      request->getPopupPresentationArg().getMinimumPresentationTime(),
      request->getPopupPresentationArg().getValidityPeriod(),
      (HorizontalAlignment::Enum) request->getPopupPresentationArg().getHorizontalAlignment(),
      (VerticalAlignment::Enum) request->getPopupPresentationArg().getVerticalAlignment(),
      (Scaling::Enum) request->getPopupPresentationArg().getScaling(),
      request->getPopupPresentationArg().getFocusPriority());

   HandleSetPopupFilter(std::string(request->getClientId().c_str()), reqarg, request->getDisableAll(), popupPresArg);
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onClearPopupFilterRequest(
   const ::boost::shared_ptr<ClearPopupFilterRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onClearPopupFilterRequest"));

   RequestArg reqarg(request->getRequestArg().getRequestId(),
                     request->getRequestArg().getAppId(),
                     request->getRequestArg().getUserData());

   HandleClearPopupFilter(
      std::string(request->getClientId().c_str()), reqarg);
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onKeyFeedbackRequest(
   const ::boost::shared_ptr<KeyFeedbackRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onKeyFeedbackRequest"));

   HandleKeyFeedback(std::string(request->getClientId().c_str()),
                     request->getUserData(),
                     request->getKeyCode(),
                     (KeyState::Enum)request->getKeyState(),
                     request->getConsumed());
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onActionRequest(
   const ::boost::shared_ptr<ActionRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onActionRequest"));

   HandleAction(std::string(request->getClientId().c_str()), request->getActionId(), request->getActionData());
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onSetInputFocusRequest(
   const ::boost::shared_ptr<SetInputFocusRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onSetInputFocusRequest"));

   HandleSetInputFocus(std::string(request->getClientId().c_str()),
                       request->getSurfaceId(),
                       request->getKeyboardFocus(),
                       request->getPointerFocus(),
                       request->getPriority(),
                       request->getFocusData());
}


// ------------------------------------------------------------------------
void AsfServiceAdaptor::onRequestCurrentStatusRequest(
   const ::boost::shared_ptr<RequestCurrentStatusRequest>& request)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor: onRequestCurrentStatusRequest"));

   HandleRequestCurrentStatus(std::string(request->getClientId().c_str()), request->getRequestId());
}


///
void AsfServiceAdaptor::SendOnGetDisplayId(const std::string& clientId,
      const RequestArg& reqArg, UInt32 displayId)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::Send OnGetDisplayId signal"));
   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      reqArg.RequestId(), reqArg.AppId(),
      reqArg.UserData());

   //send event to proxies
   sendOnGetDisplayIdSignal(clientId, reqarg, displayId);
}


///
void AsfServiceAdaptor::SendOnGetPreferredDimensions(
   const std::string& clientId, const RequestArg& reqArg,
   const DimensionArg& dimArg)
{
   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      reqArg.RequestId(), reqArg.AppId(),
      reqArg.UserData());
   bosch::cm::ai::hmi::screenbroker::Arguments::DimensionArg dimarg(dimArg.Width(), dimArg.Height());

   ETG_TRACE_USR1(("AsfServiceAdaptor::Send OnGetPreferredDimensions signal"));
   //send event to proxies
   sendOnGetPreferredDimensionsSignal(clientId, reqarg, dimarg);
}


///
void AsfServiceAdaptor::SendNotifySurfaceStateChange(
   const std::string& clientId, UInt32 userData, UInt32 surfaceId,
   UInt32 displayId, UInt32 displayAlias, SurfaceState::Enum state)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::Send NotifySurfaceStateChange signal"));
   //send surface state change notification  to registered clients
   sendOnSurfaceStateChangedSignal(clientId, userData, surfaceId, displayId, displayAlias,
                                   (bosch::cm::ai::hmi::screenbroker::Types::SurfaceState) state);
}


///
void AsfServiceAdaptor::SendNotifyCloseOnExternalTouch(bool status)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::Send NotifyCloseOnExternalTouch signal"));
   //send event to proxies
   sendOnCloseOnExternalTouchSignal(status);
}


///
void AsfServiceAdaptor::SendNotifyError(
   const ServiceRequestArg& serviceRequestArg, Int32 systemErrno,
   Int32 presentationErrno)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::Send SendNotifyError signal"));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      serviceRequestArg.RequestId(), serviceRequestArg.AppId(),
      serviceRequestArg.UserData());
   sendOnErrorSignal(serviceRequestArg.ClientId(), reqarg, systemErrno, presentationErrno);
}


///
void AsfServiceAdaptor::SendNotifyKeyEvent(const std::string& clientId,
      UInt32 userData, UInt32 surfaceId, UInt32 keyCode,
      KeyState::Enum keyState)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::Send NotifyKeyEvent signal"));
   //send event to proxies
   sendOnKeyEventSignal(clientId, userData, surfaceId, keyCode, (bosch::cm::ai::hmi::screenbroker::Types::KeyState)keyState);
}


///
void AsfServiceAdaptor::SendNotifyCurrentStatus(const std::string& clientId,
      UInt32 requestId, UInt32 surfaceId, UInt32 status, UInt32 userData)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::SendNotifyCurrentStatus signal"));
   //send event to proxies
   sendOnCurrentStatusSignal(clientId, requestId, surfaceId, status, userData);
}


///
void AsfServiceAdaptor::SendNotifyLayerStateChanged(const std::string& clientId,
      UInt32 requestId, UInt32 layerId, LayerState::Enum state)
{
   ETG_TRACE_USR1(("AsfServiceAdaptor::SendNotifyLayerStateChanged signal"));
   //send event to proxies
   sendOnLayerStateChangedSignal(clientId, requestId, layerId, (bosch::cm::ai::hmi::screenbroker::Types::LayerState)state);
}


void AsfServiceAdaptor::UpdateBuildInLayerInformation(UInt32 layerId, UInt32 surfaceId, UInt32 screenId, UInt32 width, UInt32 height, std::string connectorName)
{
   Arguments::BuildInLayerInfo info(layerId, surfaceId, screenId, connectorName, Arguments::DimensionArg(width, height));
   getBuildInLayersMutable().push_back(info);
}


} // namespace
} // namespace
