/* ***************************************************************************************
* FILE:          IScreenLayouter.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IScreenLayouter.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Plugin_IScreenLayouter_h)
#define ScreenBroker_Plugin_IScreenLayouter_h

#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <ScreenBroker/Service/ScreenArea.h>
#include <list>
#include <vector>
#include <string>

namespace ScreenBroker {
/// @addtogroup ScreenBrokerPlugin
/// @{

/**
 * @brief The IScreenLayouter class defines the interace of for a ScreenLayouter plugin.
 *
 * The ScreenLayouter plugin is dedicated to provide a Z-ordered sorted list of screen areas to
 * the screen broker service.
 */
class IScreenLayouter : public IScreenBrokerPlugin
{
   public:
      SCREENBROKER_PLUGIN_STATIC_HEADER(ScreenLayouter)

      ///
      typedef std::list<ScreenArea> ScreenAreaList;

      ///
      typedef std::vector<std::string> SeatList;

      ///
      typedef std::vector<UInt32> DisplayIdList;

      /**
       * @brief Returns the identification of a plugin instance.
       *
       * @return The identification of the plugin instance.
       */
      virtual PluginId::Enum GetId() const
      {
         return Id();
      }

      /**
       * @brief Returns the name of a plugin instance.
       *
       * @return Name of the plugin instance.
       */
      virtual const Char* GetName() const
      {
         return Name();
      }

      /**
       * @brief Provides the ScreenArea object with given ID.
       *
       * @param screenAreaId ID for which the screen area object is requested.
       * @return Screen area object with requested ID if exists, 0 otherwise
       */
      virtual ScreenArea* GetScreenArea(UInt32 screenAreaId) = 0;

      /**
       * @brief List of screen areas of initial screen layout.
       *
       * @note The list must be sorted by Z-order (increasing).
       *
       * @return Z-order sorted list of screen areas.
       */
      virtual ScreenAreaList& GetScreenAreaList() = 0;

      /**
       * @brief Get Id of display where surface is shown.
          *
       * @note This information is not processed in screenbroker core!
          *
          *
       * @param surfaceId The surface the information is requested.
       * @return DisplayId of display where the surface is shown.
       */
      virtual UInt32 GetDisplayIdOfSurface(UInt32 surfaceId) = 0;
      /**
      * @brief Get Id of ScreenArea where surface is shown.
      *
      * @note This information is not processed in screenbroker core!
      *
      *
      * @param surfaceId The surface the information is requested.
      * @return ScreenAreaId of ScreenArea where the surface is shown.
      */
      virtual UInt32 GetScreenAreaIdOfSurface(UInt32 surfaceId) = 0;
      /**
      * @brief check if observeAndAdd attribute is set for the given surface
      *
      * @note This information is not processed in screenbroker core!
      *
      *
      * @param surfaceId The surface information is requested.
      * @return bool val true if the attribute is set.
      */
      virtual bool IsSurfaceIsObserved(UInt32 surfaceId) = 0;
      /**
       * @brief Get Id of display where screenArea is shown.
          *
       * @note This information is not processed in screenbroker core!
          *
          *
       * @param screenArea The screanArea the information is requested.
       * @return DisplayId of display where the surface is shown.
       */
      virtual UInt32 GetDisplayIdOfScreenArea(UInt32 screenArea) = 0;

      /**
       * @brief Get list of seats for this surface
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param surfaceId The surface the information is requested.
       * @return List of seats
       */
      virtual SeatList* GetSeatList(UInt32 surfaceId) = 0;

      /**
       * @brief GetDisplayIdList
       * @return
       */
      virtual DisplayIdList& GetDisplayIdList() = 0;

      /**
       * @brief Get Animation Method from configuration
       *
       * @param method Animation method is returned in this variable
       *
       * @return Is animation read
       */
      virtual bool getAnimationMethod(UInt& method) = 0;
};


/// @}
}


#endif
