/* ***************************************************************************************
* FILE:          IScreenBrokerActivator.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IScreenBrokerActivator.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Plugin_IScreenBrokerActivator_h)
#define ScreenBroker_Plugin_IScreenBrokerActivator_h

#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <ScreenBroker/DimensionArg.h>

namespace ScreenBroker {
class ServiceRequestArg;
/// @addtogroup ScreenBrokerPlugin
/// @{

/**
 * @brief The IScreenBrokerActivator class defines the interace of for a ScreenBrokerActivator plugin.
 *
 * The ScreenBrokerActivator plugin is triggered by the screen broker when client requests
 * to manipulate surface presentation. This plugin has to take care to apply the project specific
 * surface arbitration logic at layer manager.
 */
class IScreenBrokerActivator : public IScreenBrokerPlugin
{
   public:
      SCREENBROKER_PLUGIN_STATIC_HEADER(ScreenBrokerActivator)

      /**
       * @brief Returns the identification of a plugin instance.
       *
       * @return The identification of the plugin instance.
       */
      virtual PluginId::Enum GetId() const
      {
         return Id();
      }

      /**
       * @brief Returns the name of a plugin instance.
       *
       * @return Name of the plugin instance.
       */
      virtual const Char* GetName() const
      {
         return Name();
      }

      /**
       * @brief Activation request for given surface triggered by client application.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param layerId Layer where the given surface is located in.
       * @param surfaceId Surface which is being activated from the client application.
       * @param hint Static surface hinting information, which was defined by user data provided
       * with surface registration by client application.
       */
      virtual void Activate(const ServiceRequestArg& serviceRequestArg,
                            UInt32 layerId,
                            UInt32 surfaceId,
                            UInt32 entryCustomAnimationType,
                            UInt32 exitCustomAnimationType,
                            UInt32 hint) = 0;

      /**
       * @brief TransitionAnimation request for given surface triggered by client application.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param layerId Layer where the given surface is located in.
       * @param surfaceId Surface which is being activated from the client application.
       * @param hint Static surface hinting information, which was defined by user data provided
       * with surface registration by client application.
       */
      virtual void TransitionAnimation(const ServiceRequestArg& serviceRequestArg,
                                       UInt32 layerId,
                                       UInt32 surfaceId,
                                       UInt32 animationType,
                                       UInt32 hint) = 0;

      /**
       * @brief Transition Animation request for given Layer triggered by client application.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param layerId Layer where the given surface is located in.
       * @param hint Static surface hinting information, which was defined by user data provided
       * with surface registration by client application.
       */
      virtual void TransitionAnimationOfLayer(const ServiceRequestArg& serviceRequestArg,
                                              UInt32 layerId,
                                              std::vector<UInt32> surfaceIdList,
                                              UInt32 animationType,
                                              UInt32 hint) = 0;
      /**
       * @brief Requests preferred dimensions for surfaces (e.g. indicators) on the given layer.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin. The request argument is returned unmodified in the
       * application callback OnGetPreferredDimensions for mapping the request and callback (using the request ID).
       * @param layerId Layer for which the preferred dimensions of surfaces is requested.
       * @param defaultDimension Default dimension of the layer.
       * @return Preferred dimensions for sufaces on given layer.
       */
      virtual DimensionArg GetPreferredDimensions(const ServiceRequestArg& serviceRequestArg,
            UInt32 layerId,
            const DimensionArg& defaultDimension) = 0;
      /**
       * @brief SetInputFocus request for given surface triggerd by client application.
       *
       * @param surfaceId Surface which should get focus.
       * @param keyboardFocus If keyboard focus should be set.
       * @param pointerFocus If pointer focus should be set.
       * @param priority Focus priority (0 default priority, high value -> low priority)
       * @param focusData Used to force or unforce the input focus.
       */
      virtual void SetInputFocus(UInt32 surfaceId,
                                 bool keyboardFocus,
                                 bool pointerFocus,
                                 Int32 priority,
                                 UInt32 focusData) = 0;

      /**
       * @brief Deregister request for given surface triggered by client application.
       *
       * @param surfaceId The surface ID of the surface which is deregistered.
       */
      virtual void DeregisterSurface(UInt32 surfaceId) = 0;
};


/// @}
}


#endif
