/* ***************************************************************************************
* FILE:          ServiceProxy.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ServiceProxy.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Client_ServiceProxy_h)
#define ScreenBroker_Client_ServiceProxy_h

// =============================================================================
// This class ServiceProxy::Api is intended always to be called with a valid
// ServiceProxy object. Additionally this is done only once per statically
// chosen DBus binding.
// Furthermore this class will never be created dynamically or copied in any
// way, which makes it safe to omit the default constructor intentionally.
//lint -esym(1712, Api) "default constructor not defined for class"
// =============================================================================

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/KeyState.h>
#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/RequestArg.h>
#include <ScreenBroker/BuildInLayerInfo.h>
#include <ScreenBroker/SurfaceState.h>
#include <ScreenBroker/Client/ClientApi.h>
#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <hmibase/util/Condition.h>
#include <string>
#include <vector>

namespace asf {
namespace core {
class BaseComponent;
}


}

namespace ScreenBroker {
namespace Internal {
/// @addtogroup ScreenBrokerClient
/// @{

/**
 * @brief The ServiceProxy class wraps the communication layer and manages its interface to the service.
 */
class ServiceProxy
{
   public:
      /**
       * @brief get the service proxy singleton
       * there can only be one proxy object in the system.
       * @return the ServiceProxy object or 0 (null) in case of errors or multiple Create requests
       */
      static ServiceProxy* Create(ClientApi* clientApi, std::string& clientId, ::asf::core::BaseComponent* baseComponent = 0);

      ///
      virtual ~ServiceProxy();

      ///
      ClientApi* SetClientApi(ClientApi* clientApi);

      /**
       * @brief Initializes the service
       */
      void Init();

      /**
       * @brief Run the service, returns only after service terminates
       */
      void Run();

      /**
       * @brief Synchronizes the calling (startup) thread with the screen broker client thread context.
       */
      void Synchronize();

      /**
       * @brief Terminates the service
       */
      void Terminate();

      ///
      virtual const Char* GetVersion() = 0;

      ///
      virtual bool GetBuildInLayerInfo(std::vector<BuildInLayerInfo>& info) = 0;

      ///
      virtual void GetDisplayId(const RequestArg& requestArg,
                                UInt32 screenAreaId) = 0;

      ///
      virtual void GetPreferredDimensions(const RequestArg& requestArg,
                                          UInt32 screenAreaId) = 0;

      ///
      virtual void RegisterSurface(const RequestArg& requestArg,
                                   UInt32 screenAreaId,
                                   UInt32 surfaceId) = 0;

      ///
      virtual void DeregisterSurface(const RequestArg& requestArg,
                                     UInt32 surfaceId,
                                     bool sendToIlm) = 0;

      ///
      virtual void ActivateApplication(const RequestArg& requestArg,
                                       UInt32 surfaceId,
                                       UInt32 entryCustomAnimationType,
                                       UInt32 exitCustomAnimationType) = 0;

      ///
      virtual void StartAnimation(const RequestArg& requestArg,
                                  UInt32 surfaceId,
                                  UInt32 animationType,
                                  UInt32 hint) = 0;
      ///
      virtual void StartLayerAnimation(const RequestArg& requestArg,
                                       UInt32 layerId,
                                       std::vector<UInt32> surfaceIdList,
                                       UInt32 animationType,
                                       UInt32 hint) = 0;

      ///
      virtual void ShowPopup(const RequestArg& requestArg,
                             UInt32 surfaceId,
                             const PopupPresentationArg& popupPresentationArg) = 0;

      ///
      virtual void HidePopup(const RequestArg& requestArg,
                             UInt32 surfaceId) = 0;

      ///
      virtual void CloseOnExternalTouch(const RequestArg& requestArg) = 0;

      ///
      virtual void SetPopupFilter(const RequestArg& requestArg,
                                  bool disableAll,
                                  const PopupPresentationArg& popupPresentationArg) = 0;

      ///
      virtual void ClearPopupFilter(const RequestArg& requestArg) = 0;

      ///
      virtual void KeyFeedback(UInt32 userData,
                               UInt32 keyCode,
                               KeyState::Enum keyState,
                               bool consumed) = 0;

      ///
      virtual void Action(UInt32 actionId, UInt32 actionData) = 0;

      ///
      virtual void SetInputFocus(UInt32 surfaceId,
                                 bool keyboardFocus,
                                 bool pointerFocus,
                                 Int32 priority,
                                 UInt32 focusData) = 0;

      ///
      virtual void RequestCurrentStatus(UInt32 requestId) = 0;

   protected:
#if defined(SCREENBROKER_WITH_CLIENTSYNC)
      /// Semaphore for run synchronization
      hmibase::util::Condition _cond;
#endif

      /**
       * @brief Initializes the service
       */
      virtual void DoInit() = 0;

      /**
       * @brief Run the service, returns only after service terminates
       */
      virtual void DoRun() = 0;

      /**
       * @brief Terminates the service
       */
      virtual void DoTerminate() = 0;

      ///
      ServiceProxy(ClientApi* clientApi = 0);

      class Api : public CriticalSectionLocker
      {
         public:
            Api(ServiceProxy* proxy)
               : CriticalSectionLocker(&proxy->mCs),
                 mClientApi(proxy->mClientApi)
            {
            }

            bool Ok() const
            {
               return mClientApi != 0;
            }

            ClientApi* operator->()
            {
               return mClientApi;
            }

            ClientApi* operator*()
            {
               return mClientApi;
            }

         private:
            ClientApi* mClientApi;
      };

   private:
      ClientApi* mClientApi;
      CriticalSection mCs;
};


/// @}
}


}
#endif
