/* ***************************************************************************************
* FILE:          Client.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Client.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Client_Client_h)
#define ScreenBroker_Client_Client_h

// =============================================================================
// This class is intended to be a singleton, for this reason it is ok, that the
// constructor and the copy constructor have only private access.
//lint -esym(1704, ScreenBroker::Client::Client)   "Constructor has private access specification"
// =============================================================================

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <string>

namespace asf {
namespace core {
class BaseComponent;
}


}

namespace ScreenBroker {
// Forward declarations
class ClientApi;
/// @addtogroup ScreenBrokerClient
/// @{

/// Working state of client.
namespace ClientWorkingState {
enum Enum
{
   ///
   Initialized,
   ///
   Running,
   ///
   Terminated
};


}

/**
 * @brief The Client class provides the screen broker client instance.
 *
 * The client shall always run in a separate thread context, as it will block for listening
 * the service-side communication.
 *
 * The client shall be initialized with a application defined ClientApi instance,
 * in order to forward the service-side notifications to the client application.
 */
class Client
{
      friend class ClientApi;
   public:
      /**
       * @brief Get the singleton instance of the screen broker client.
       *
       * @return Screen broker client instance.
       */
      static Client& GetInstance();

      ///
      ~Client();

      /**
       * @brief Initialize the client with an application defined ClientApi object, having the
       * callback interfaces implemented.
       *
       * @param clientApi Pointer to the application defined clientApi instance.
       * @return True, if initialization was successful, false otherwise.
       */
      bool InitClientApi(ClientApi* clientApi, const char* clientId, ::asf::core::BaseComponent* baseComponent);

      /**
       * @brief Set (or change) the instance of the clientApi object.
       *
       * As screen broker client is a singleton and doesn't allow multiple registrations of clientApi
       * instances, it sometime may come handy if another application module (e.g. a library) may temporarily overwrite
       * (resp. set it's own) clientApi instance, for issuing request (e.g. GetDisplayId()), and give controls
       * back to the main application clientApi, once the request is finished.
       *
       * @param clientApi The instance of clientApi which shall be used from now on.
       * @return The previous set instance of clientApi.
       */
      ClientApi* SetClientApi(ClientApi* clientApi);

      /**
       * @brief Starts the screen broker client.
       *
       * @note This interface block the current execution, so it is recommended to start the client within a
       * separate thread context.
       */
      void Run();

      /**
       * @brief Synchronizes the calling (startup) thread with the screen broker client thread context.
       *
       * @note This interface blocks the execution of the calling thread until the screen broker client
       * is successfully initialized and running.
       */
      void Synchronize();

      /**
       * @brief Requests the unique client ID (normally provided by the respective transport layer instance).
       *
       * @return Unique client ID.
       */
      const Char* ClientId() const;

      /**
       * @brief Informs if the screen broker client is up and running.
       *
       * @return True, if screen broker client is running, false otherwise.
       */
      bool IsRunning();

      /**
       * @brief Destroys the screen broker client instance and informs the screen broker service
       * about the client termination.
       */
      void Destroy();

   private:
      ClientWorkingState::Enum mState;
      Internal::ServiceProxy* mServiceProxy;

      CriticalSection mCs;

      std::string mClientId;

      ///
      Client();

      // Dis-allow copying of singleton
      Client(const Client&);
      Client& operator=(const Client&);

      ///
      Internal::ServiceProxy* GetServiceProxy()
      {
         return mServiceProxy;
      }
};


/// @}
}


#endif
