/* ***************************************************************************************
* FILE:          Client.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Client.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Connection.h>
#include "Client.h"
#include "ServiceProxy.h"

#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERCLIENT
#include "trcGenProj/Header/Client.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenBrokerClient);

// ------------------------------------------------------------------------
Client& Client::GetInstance()
{
   // Lazy initialization of screen broker service singleton
   static Client lClient;
   return lClient;
}


// ------------------------------------------------------------------------
Client::Client() :
   mState(ClientWorkingState::Terminated),
   mServiceProxy(0)
{
   ETG_TRACE_USR4_THR(("Client created"));
}


// ------------------------------------------------------------------------
Client::~Client()
{
   mServiceProxy = 0;
   ETG_TRACE_USR4_THR(("Client '%10s' destroyed", ClientId()));
}


// ------------------------------------------------------------------------
bool Client::InitClientApi(ClientApi* clientApi, const char* clientId, ::asf::core::BaseComponent* baseComponent)
{
   //SCREENBROKER_LOG_FN();
   bool lRc = (0 != clientApi);
   if (lRc)
   {
      CriticalSectionLocker lLock(&mCs);
      mClientId = clientId;
      lRc = (0 == mServiceProxy);
      if (lRc)
      {
         ETG_TRACE_USR4_THR(("Client initializing:ClientId:%s", mClientId.c_str()));

         mServiceProxy = Internal::ServiceProxy::Create(clientApi, mClientId, baseComponent);
         lRc = (0 != mServiceProxy);

         if (lRc)
         {
            mServiceProxy->Init();
            // Set screen broker client state
            mState = ClientWorkingState::Initialized;
         }
         else
         {
            ETG_TRACE_ERR_THR(("Failed to create client!"));
         }
      }
      else
      {
         ETG_TRACE_SYS_THR(("Client '%10s' already initialized!", ClientId()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("ClientApi instance is not valid in Client '%40s'!", ClientId()));
   }

   return lRc;
}


// ------------------------------------------------------------------------
ClientApi* Client::SetClientApi(ClientApi* clientApi)
{
   //SCREENBROKER_LOG_FN();
   CriticalSectionLocker lLock(&mCs);
   if (0 != mServiceProxy)
   {
      return (mServiceProxy->SetClientApi(clientApi));
   }
   else
   {
      ETG_TRACE_ERR_THR(("Couldn't modify client api. Client not yet initialized!"));
      return 0;
   }
}


// ------------------------------------------------------------------------
void Client::Run()
{
   //SCREENBROKER_LOG_FN();
   if (0 != mServiceProxy)
   {
      ETG_TRACE_USR1_THR(("Client starting"));
      {
         CriticalSectionLocker lLock(&mCs);
         mState = ClientWorkingState::Running;
      }
      mServiceProxy->Run();
   }
   else
   {
      ETG_TRACE_FATAL_THR(("Client not initialized! Terminate.."));
   }
}


// ------------------------------------------------------------------------
void Client::Synchronize()
{
   //SCREENBROKER_LOG_FN();
#if defined(SCREENBROKER_WITH_CLIENTSYNC)
   if (!IsRunning())
   {
      if (0 != mServiceProxy)
      {
         mServiceProxy->Synchronize();
         ETG_TRACE_USR1_THR(("Client '%10s' synced & running", ClientId()));
      }
      else
      {
         ETG_TRACE_SYS_THR(("Client not initialized! No syncing done.."));
      }
   }
   else
   {
      ETG_TRACE_SYS_THR(("Client '%10s' already running! No syncing done..", ClientId()));
   }
#endif
}


// ------------------------------------------------------------------------
const Char* Client::ClientId() const
{
   return mClientId.c_str();
}


// ------------------------------------------------------------------------
bool Client::IsRunning()
{
   CriticalSectionLocker lLock(&mCs);
   return (ClientWorkingState::Running == mState);
}


// ------------------------------------------------------------------------
void Client::Destroy()
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR4_THR(("Destroying client '%40s'", ClientId()));
   CriticalSectionLocker lLock(&mCs);
   if ((0 != mServiceProxy) && (ClientWorkingState::Terminated != mState))
   {
      mServiceProxy->Terminate();
      mState = ClientWorkingState::Terminated;
   }
   else
   {
      ETG_TRACE_SYS_THR(("Client not running!"));
   }
}


}
