/* ***************************************************************************************
 * FILE:          AsfServiceProxy.cpp
 * SW-COMPONENT:  HMI-BASE
 * DESCRIPTION:  AsfServiceProxy.cpp is part of HMI-Base ScreenBroker
 * COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include <ScreenBroker/DimensionArg.h>
#include <ScreenBroker/RequestArg.h>
#include <ScreenBroker/Connection.h>
#include <ScreenBroker/BuildInLayerInfo.h>

#include <ScreenBroker/Client/Client.h>
#include <ScreenBroker/Client/ClientApi.h>
#include "AsfServiceProxy.h"

#include "ScreenBroker/ScreenBroker_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERCLIENT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AsfServiceProxy.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::screenbroker;
using namespace ::bosch::cm::ai::hmi::screenbroker::Service;

namespace ScreenBroker {
namespace Internal {

// ========================================================================
static AsfServiceProxy* sInstance = 0;  // to be checked
// ========================================================================

// ========================================================================
// to get the client api instance
class ServiceApi
{
   public:
      ServiceApi() :
         mApi(sInstance)
      {
      }

      bool Ok() const
      {
         return mApi.Ok(); //get service proxy instance
      }

      ClientApi* operator->()
      {
         return *mApi;
      }

   private:
      AsfServiceProxy::Api mApi;
};


// -------------------------------------------------------------------------
AsfServiceProxy* AsfServiceProxy::DoCreate(ClientApi* clientApi,
      std::string& clientId,
      ::asf::core::BaseComponent* baseComponent)
{
   if (sInstance != 0)
   {
      return 0;
   }

   static Internal::AsfServiceProxy serviceProxy(clientId);
   // set the client api and sInstance
   serviceProxy.SetClientApi(clientApi);
   sInstance = &serviceProxy;

   serviceProxy.setBaseComponent(baseComponent);

   return sInstance;
}


// -------------------------------------------------------------------------
AsfServiceProxy::AsfServiceProxy(std::string& clientId) :
   mClientId(clientId),
   mversion(""),
   _SBServiceProxy_CurrentStatus_RegId(0),
   _SBServiceProxy_LayerStateChanged_RegId(0),
   _SBServiceProxy_SurfaceStateChanged_RegId(0),
   _SBServiceProxy_GetPreferredDimensions_RegId(0),
   _SBServiceProxy_KeyEvent_RegId(0),
   _SBServiceProxy_Error_RegId(0),
   _SBServiceProxy_CloseOnExternalTouch_RegId(0),
   _SBServiceProxy_GetDisplayId_RegId(0),
   _SBServiceProxy_Version_RegId(0),
   _baseComponent(0),
   _SBServiceProxy(bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy::createProxy("SBServicePort", *this))
{
   ETG_TRACE_USR4_THR(("AsfServiceProxy constructor"));
   _AsfServiceProxyAdaptor.setProxy(_SBServiceProxy);
}


// -------------------------------------------------------------------------
AsfServiceProxy::~AsfServiceProxy()
{
}


// -------------------------------------------------------------------------
const Char* AsfServiceProxy::GetClientId()
{
   return (sInstance != 0) ? sInstance->mClientId.c_str() : "";
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onAvailable(
   const ::boost::shared_ptr<asf::core::Proxy>& proxy,
   const ServiceStateChange& /*stateChange*/)
{
   if (_SBServiceProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("AsfServiceProxy onAvailable"));

      _SBServiceProxy_CurrentStatus_RegId =
         _SBServiceProxy->sendOnCurrentStatusRegister(*this);

      _SBServiceProxy_LayerStateChanged_RegId =
         _SBServiceProxy->sendOnLayerStateChangedRegister(*this);

      _SBServiceProxy_GetDisplayId_RegId =
         _SBServiceProxy->sendOnGetDisplayIdRegister(*this);

      _SBServiceProxy_GetPreferredDimensions_RegId =
         _SBServiceProxy->sendOnGetPreferredDimensionsRegister(*this);

      _SBServiceProxy_SurfaceStateChanged_RegId =
         _SBServiceProxy->sendOnSurfaceStateChangedRegister(*this);

      _SBServiceProxy_Error_RegId = _SBServiceProxy->sendOnErrorRegister(
                                       *this);

      _SBServiceProxy_KeyEvent_RegId =
         _SBServiceProxy->sendOnKeyEventRegister(*this);

      _SBServiceProxy_CloseOnExternalTouch_RegId =
         _SBServiceProxy->sendOnCloseOnExternalTouchRegister(*this);

      ScreenBroker::Client::GetInstance().Run(); //client

      ServiceApi api;
      if (api.Ok())
      {
         api->OnServiceStatusChanged(true);
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onUnavailable(
   const ::boost::shared_ptr<asf::core::Proxy>& proxy,
   const ServiceStateChange& /*stateChange*/)
{
   if (_SBServiceProxy == proxy)
   {
      ETG_TRACE_USR4_THR(("AsfServiceProxy onUnavailable"));
      _SBServiceProxy->sendOnCurrentStatusDeregister(
         _SBServiceProxy_CurrentStatus_RegId);
      _SBServiceProxy->sendOnLayerStateChangedDeregister(
         _SBServiceProxy_LayerStateChanged_RegId);
      _SBServiceProxy->sendOnGetDisplayIdDeregister(
         _SBServiceProxy_GetDisplayId_RegId);
      _SBServiceProxy->sendOnGetPreferredDimensionsDeregister(
         _SBServiceProxy_GetPreferredDimensions_RegId);
      _SBServiceProxy->sendOnKeyEventDeregister(
         _SBServiceProxy_KeyEvent_RegId);
      _SBServiceProxy->sendOnErrorDeregister(_SBServiceProxy_Error_RegId);
      _SBServiceProxy->sendOnCloseOnExternalTouchDeregister(
         _SBServiceProxy_CloseOnExternalTouch_RegId);
      _SBServiceProxy->sendOnSurfaceStateChangedDeregister(
         _SBServiceProxy_SurfaceStateChanged_RegId);

      if (sInstance != 0)
      {
         sInstance->Reset();//cleanup
      }

      ServiceApi api;
      if (api.Ok())
      {
         api->OnServiceStatusChanged(false);
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::setBaseComponent(::asf::core::BaseComponent* baseComponent)
{
   _baseComponent = baseComponent;
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnGetDisplayIdError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnGetDisplayIdError > & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnGetDisplayIdSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnGetDisplayIdSignal > & signal)
{
   if (_SBServiceProxy && sInstance
         && sInstance->IsEventAddressee(signal->getClientId().c_str()))
   {
      ServiceApi api;
      if (api.Ok())
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive onOnGetDisplayIdSignal(%40s,%u, %u,%u,%u)", signal->getClientId().c_str(),
                             signal->getRequestArg().getRequestId(),
                             signal->getRequestArg().getAppId(),
                             signal->getRequestArg().getUserData(),
                             signal->getDisplayId()));

         RequestArg reqarg(signal->getRequestArg().getRequestId(),
                           signal->getRequestArg().getAppId(),
                           signal->getRequestArg().getUserData());

         api->OnGetDisplayId(reqarg,
                             static_cast<UInt32>(signal->getDisplayId()));
      }
      else
      {
         ETG_TRACE_SYS_THR(("error receiving onOnGetDisplayId signal"));
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnGetPreferredDimensionsError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnGetPreferredDimensionsError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnGetPreferredDimensionsSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnGetPreferredDimensionsSignal > & signal)
{
   if (_SBServiceProxy && sInstance
         && sInstance->IsEventAddressee(signal->getClientId().c_str()))
   {
      ServiceApi api;
      if (api.Ok())
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive onOnGetPreferredDimensionsSignal(%40s,{%u,%u,%u},{%u,%u})", signal->getClientId().c_str(),
                             signal->getRequestArg().getRequestId(),
                             signal->getRequestArg().getAppId(),
                             signal->getRequestArg().getUserData(),
                             signal->getDimensionArg().getWidth(),
                             signal->getDimensionArg().getHeight()));

         RequestArg reqarg(signal->getRequestArg().getRequestId(),
                           signal->getRequestArg().getAppId(),
                           signal->getRequestArg().getUserData());
         DimensionArg dimarg(signal->getDimensionArg().getWidth(), signal->getDimensionArg().getHeight());

         api->OnGetPreferredDimensions(reqarg,
                                       dimarg);
      }
      else
      {
         ETG_TRACE_SYS_THR(("error receiving GetPreferredDimensionsSignal signal"));
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnErrorError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnErrorError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnErrorSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnErrorSignal > & signal)
{
   if (_SBServiceProxy && sInstance
         && sInstance->IsEventAddressee(signal->getClientId().c_str()))
   {
      ServiceApi api;
      if (api.Ok())
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive OnError(%40s, {%u, %u, %u}, %d, %d)",
                             signal->getClientId().c_str(),
                             signal->getRequestArg().getRequestId(),
                             signal->getRequestArg().getAppId(),
                             signal->getRequestArg().getUserData(),
                             signal->getSystemErrno(),
                             signal->getPresentationErrno()));

         RequestArg reqarg(signal->getRequestArg().getRequestId(),
                           signal->getRequestArg().getAppId(),
                           signal->getRequestArg().getUserData());

         api->OnError(reqarg,
                      Int32(signal->getSystemErrno()),
                      Int32(signal->getPresentationErrno()));
      }
      else
      {
         ETG_TRACE_SYS_THR(("error receiving OnError signal"));
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnKeyEventError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnKeyEventError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnKeyEventSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnKeyEventSignal > & signal)
{
   if (_SBServiceProxy && sInstance
         && sInstance->IsEventAddressee(signal->getClientId().c_str()))
   {
      ServiceApi api;
      if (api.Ok())
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive OnKeyEvent(%40s, %u, %u, %u, %d)",
                             signal->getClientId().c_str(),
                             signal->getUserData(),
                             signal->getSurfaceId(),
                             signal->getKeyCode(),
                             signal->getKeyState()));

         api->OnKeyEvent(static_cast<UInt32>(signal->getUserData()),
                         static_cast<UInt32>(signal->getSurfaceId()),
                         static_cast<UInt32>(signal->getKeyCode()),
                         static_cast<KeyState::Enum>(signal->getKeyState()));
      }
      else
      {
         ETG_TRACE_SYS_THR(("error receiving OnKeyEvent signal"));
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnSurfaceStateChangedError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnSurfaceStateChangedError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnSurfaceStateChangedSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnSurfaceStateChangedSignal > & signal)
{
   if (_SBServiceProxy && sInstance
         && sInstance->IsEventAddressee(signal->getClientId().c_str()))
   {
      ServiceApi api;
      if (api.Ok())
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive onOnSurfaceStateChangedSignal(%40s, %u, %u, %u, %d)", signal->getClientId().c_str(), signal->getUserData(), signal->getSurfaceId(), signal->getDisplayId(), signal->getSurfaceState()));
         api->OnSurfaceStateChanged(
            static_cast<UInt32>(signal->getUserData()),
            static_cast<UInt32>(signal->getSurfaceId()),
            static_cast<UInt32>(signal->getDisplayId()),
            static_cast<UInt32>(signal->getDisplayAlias()),
            static_cast<SurfaceState::Enum>(signal->getSurfaceState()));
      }
      else
      {
         ETG_TRACE_SYS_THR(("error receiving OnSurfaceStateChanged signal"));
      }
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnCurrentStatusError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnCurrentStatusError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnCurrentStatusSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnCurrentStatusSignal > & signal)
{
   ServiceApi api;
   if (api.Ok() && (_SBServiceProxy && sInstance))
   {
      if (sInstance->IsEventAddressee(signal->getClientId().c_str()))
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive onOnCurrentStatusSignal(%40s, %u, %u, %u, %u)", signal->getClientId().c_str(), signal->getRequestId(),
                             signal->getSurfaceId(), signal->getStatus(), signal->getUserData()));

         api->OnCurrentStatus(static_cast<UInt32>(signal->getRequestId()),
                              static_cast<UInt32>(signal->getSurfaceId()),
                              static_cast<UInt32>(signal->getStatus()),
                              static_cast<UInt32>(signal->getUserData()));
      }
      else
      {
         ETG_TRACE_USR1_THR(("Receive onOnCurrentStatusSignal without matching clientid"));
         api->OnCurrentStatus(static_cast<UInt32>(signal->getRequestId()),
                              static_cast<UInt32>(signal->getSurfaceId()),
                              static_cast<UInt32>(signal->getStatus()),
                              static_cast<UInt32>(signal->getUserData()));
      }
   }
   else
   {
      ETG_TRACE_SYS_THR(("error receiving onOnCurrentStatusSignal signal"));
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnLayerStateChangedError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnLayerStateChangedError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnLayerStateChangedSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnLayerStateChangedSignal > & signal)
{
   ServiceApi api;
   if (api.Ok() && (_SBServiceProxy && sInstance))
   {
      if (sInstance->IsEventAddressee(signal->getClientId().c_str()))
      {
         //SCREENBROKER_LOG_SCOPE(apiOk);
         ETG_TRACE_USR1_THR(("Receive onOnLayerStateChangedSignal(%40s, %u ,%u, %u)", signal->getClientId().c_str(),
                             signal->getRequestId(), signal->getLayerID(), signal->getLayerState()));

         api->OnLayerStateChanged(static_cast<UInt32>(signal->getRequestId()),
                                  static_cast<UInt32>(signal->getLayerID()),
                                  static_cast<LayerState::Enum>(signal->getLayerState()));
      }
      else
      {
         ETG_TRACE_USR1_THR(("Receive onOnLayerStateChangedSignal without matching clientid"));
         api->OnLayerStateChanged(static_cast<UInt32>(signal->getRequestId()),
                                  static_cast<UInt32>(signal->getLayerID()),
                                  static_cast<LayerState::Enum>(signal->getLayerState()));
      }
   }
   else
   {
      ETG_TRACE_SYS_THR(("error receiving onOnLayerStateChangedSignal signal"));
   }
}


// -------------------------------------------------------------------------

void AsfServiceProxy::onOnCloseOnExternalTouchError(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnCloseOnExternalTouchError >  & /*error*/)
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::onOnCloseOnExternalTouchSignal(
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy > & /*proxy*/,
   const ::boost::shared_ptr <
   bosch::cm::ai::hmi::screenbroker::Service::OnCloseOnExternalTouchSignal > & signal)
{
   ServiceApi api;
   if (api.Ok())
   {
      //SCREENBROKER_LOG_SCOPE(apiOk);
      ETG_TRACE_USR1_THR(("Receive onOnCloseOnExternalTouchSignal(%u)", signal->getStatus()));
      api->OnCloseOnExternalTouch(static_cast<bool>(signal->getStatus()));
   }
}


void AsfServiceProxy::onVersionError(const ::boost::shared_ptr< bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy>& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::screenbroker::Service::VersionError >& /*error*/) {}
void AsfServiceProxy::onVersionUpdate(const ::boost::shared_ptr< bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy>& /*proxy*/, const ::boost::shared_ptr< bosch::cm::ai::hmi::screenbroker::Service::VersionUpdate >& update)
{
   mversion = update->getVersion();

   ETG_TRACE_USR1_THR(("Receive onVersionUpdate(%s)", mversion.c_str()));
}


//
void AsfServiceProxy::onBuildInLayersError(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy >& /*proxy*/, const ::boost::shared_ptr< ::bosch::cm::ai::hmi::screenbroker::Service::BuildInLayersError >& /*error*/) {}
void AsfServiceProxy::onBuildInLayersUpdate(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::screenbroker::Service::ServiceProxy >& /*proxy*/, const ::boost::shared_ptr< ::bosch::cm::ai::hmi::screenbroker::Service::BuildInLayersUpdate >& update)
{
   ETG_TRACE_USR1_THR(("Receive onBuildInLayersUpdate, size %d", update->getBuildInLayers().size()));

   ServiceApi api;
   if (api.Ok())
   {
      std::vector<BuildInLayerInfo> info;
      std::vector<Arguments::BuildInLayerInfo> propertyVector = update->getBuildInLayers();

      for (size_t i = 0; i < propertyVector.size(); ++i)
      {
         BuildInLayerInfo tmp(propertyVector[i].getLayerId(),
                              propertyVector[i].getSurfaceId(),
                              propertyVector[i].getScreenId(),
                              propertyVector[i].getConnectorName(),
                              DimensionArg(propertyVector[i].getDimensions().getWidth(), propertyVector[i].getDimensions().getHeight()));
         info.push_back(tmp);
      }
      api->OnBuildInLayerInfo(info);
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::GetDisplayId(const RequestArg& requestArg,
                                   UInt32 screenAreaId)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1_THR(("Send GetDisplayId(%40s, {%u, %u, %u}, %u)",
                       GetClientId(),
                       requestArg.RequestId(),
                       requestArg.AppId(),
                       requestArg.UserData(),
                       screenAreaId));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   GetDisplayIdPayload payload(GetClientId(), reqarg, screenAreaId);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::GetPreferredDimensions(const RequestArg& requestArg,
      UInt32 screenAreaId)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1_THR(("Send GetPreferredDimensions(%40s, {%u, %u, %u}, %u)",
                       GetClientId(),
                       requestArg.RequestId(),
                       requestArg.AppId(),
                       requestArg.UserData(),
                       screenAreaId));
   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   GetPreferredDimensionsPayload payload(GetClientId(), reqarg, screenAreaId);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::RegisterSurface(const RequestArg& requestArg,
                                      UInt32 screenAreaId, UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1(
      ("Send RegisterSurface(%40s, {%u, %u, %u}, %u, %u)", GetClientId(), requestArg.RequestId(), requestArg.AppId(), requestArg.UserData(), screenAreaId, surfaceId));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   RegisterSurfacePayload payload(GetClientId(), reqarg, screenAreaId, surfaceId);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::DeregisterSurface(const RequestArg& requestArg,
                                        UInt32 surfaceId, bool sendToIlm)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1(
      ("Send DeregisterSurface(%40s, {%u, %u, %u}, %u, %d)", GetClientId(), requestArg.RequestId(), requestArg.AppId(), requestArg.UserData(), surfaceId, sendToIlm));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   DeregisterSurfacePayload payload(GetClientId(), reqarg, sendToIlm, surfaceId);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::ActivateApplication(const RequestArg& requestArg,
      UInt32 surfaceId,
      UInt32 entryCustomAnimationType,
      UInt32 exitCustomAnimationType)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1_THR(("Send ActivateApplication(%40s, {%u, %u, %u}, %u, %u, %u)",
                       GetClientId(),
                       requestArg.RequestId(),
                       requestArg.AppId(),
                       requestArg.UserData(),
                       surfaceId,
                       entryCustomAnimationType,
                       exitCustomAnimationType));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   ActivateApplicationPayload payload(GetClientId(), reqarg, surfaceId, entryCustomAnimationType, exitCustomAnimationType);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::StartAnimation(const RequestArg& requestArg,
                                     UInt32 surfaceId,
                                     UInt32 animationType,
                                     UInt32 hint)
{
   ETG_TRACE_USR1_THR(("Send StartAnimation(%40s, {%u, %u, %u}, %u, %u, %u)",
                       GetClientId(),
                       requestArg.RequestId(),
                       requestArg.AppId(),
                       requestArg.UserData(),
                       surfaceId,
                       animationType,
                       hint));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   StartAnimationPayload payload(GetClientId(), reqarg,
                                 surfaceId, animationType, hint);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::StartLayerAnimation(const RequestArg& requestArg,
      UInt32 layerId,
      std::vector<UInt32> surfaceIdList,
      UInt32 animationType,
      UInt32 hint)
{
   ETG_TRACE_USR1_THR(("Send StartAnimation(%40s, {%u, %u, %u}, %u, %u, %u)",
                       GetClientId(),
                       requestArg.RequestId(),
                       requestArg.AppId(),
                       requestArg.UserData(),
                       layerId,
                       animationType,
                       hint));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   AttachandAnimateLayerPayload payload(GetClientId(), reqarg,
                                        layerId, surfaceIdList, animationType, hint);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::ShowPopup(const RequestArg& requestArg, UInt32 surfaceId,
                                const PopupPresentationArg& popupPresentationArg)
{
   {
      char str[230];
      sprintf(str,
              "Send ShowPopup(%s, {%u, %u, %u}, %u, {%u, %d, %u, %u, %u, %d, %d, %d, %d})",
              GetClientId(), requestArg.RequestId(), requestArg.AppId(),
              requestArg.UserData(), surfaceId,
              popupPresentationArg.Priority(),
              popupPresentationArg.Modality(),
              popupPresentationArg.PresentationTime(),
              popupPresentationArg.MinimumPresentationTime(),
              popupPresentationArg.ValidityPeriod(),
              popupPresentationArg.HorizontalAlignment(),
              popupPresentationArg.VerticalAlignment(),
              popupPresentationArg.Scaling(),
              popupPresentationArg.FocusPriority());
      ETG_TRACE_USR1_THR(("%s", str));
   }
   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqArg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());
   bosch::cm::ai::hmi::screenbroker::Arguments::PopupPresentationArg popupPresArg(
      popupPresentationArg.Priority(),
      (::bosch::cm::ai::hmi::screenbroker::Types::Modality) popupPresentationArg.Modality(),
      popupPresentationArg.PresentationTime(),
      popupPresentationArg.MinimumPresentationTime(),
      popupPresentationArg.ValidityPeriod(),
      (::bosch::cm::ai::hmi::screenbroker::Types::HorizontalAlignment) popupPresentationArg.HorizontalAlignment(),
      (::bosch::cm::ai::hmi::screenbroker::Types::VerticalAlignment) popupPresentationArg.VerticalAlignment(),
      (::bosch::cm::ai::hmi::screenbroker::Types::Scaling) popupPresentationArg.Scaling(),
      popupPresentationArg.FocusPriority());

   ShowPopupPayload payload(GetClientId(), reqArg, surfaceId,
                            popupPresArg);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::HidePopup(const RequestArg& requestArg,
                                UInt32 surfaceId)
{
   ETG_TRACE_USR1_THR(("Send HidePopup(%40s, {%u, %u, %u}, %u)", GetClientId(), requestArg.RequestId(), requestArg.AppId(), requestArg.UserData(), surfaceId));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   HidePopupPayload payload(GetClientId(), reqarg, surfaceId);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------

void AsfServiceProxy::SetInputFocus(UInt32 surfaceId, bool keyboardFocus,
                                    bool pointerFocus, Int32 priority, UInt32 focusData)
{
   ETG_TRACE_USR1_THR(("SetInputFocus(%40s, %u, %d, %d, %d, %u)",
                       GetClientId(),
                       surfaceId,
                       keyboardFocus,
                       pointerFocus,
                       priority,
                       focusData));

   SetInputFocusPayload payload(GetClientId(), surfaceId, keyboardFocus, pointerFocus, priority, focusData);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::KeyFeedback(UInt32 userData, UInt32 keyCode,
                                  KeyState::Enum keyState, bool consumed)
{
   ETG_TRACE_USR1_THR(("Send KeyFeedback(%40s, %u, %u, %d, %d)",
                       GetClientId(),
                       userData,
                       keyCode,
                       keyState,
                       consumed));

   KeyFeedbackPayload payload(GetClientId(), userData, keyCode, bosch::cm::ai::hmi::screenbroker::Types::KeyState(keyState), consumed);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::CloseOnExternalTouch(const RequestArg& requestArg)
{
   ETG_TRACE_USR1_THR(("Send CloseOnExternalTouch(%40s, {%u, %u, %u})", GetClientId(), requestArg.RequestId(), requestArg.AppId(), requestArg.UserData()));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   CloseOnExternalTouchPayload payload(GetClientId(), reqarg);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::ClearPopupFilter(const RequestArg& requestArg)
{
   ETG_TRACE_USR1_THR(("Send ClearPopupFilter(%40s, {%u, %u, %u})",
                       GetClientId(),
                       requestArg.RequestId(),
                       requestArg.AppId(),
                       requestArg.UserData()));

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   ClearPopupFilterPayload payload(GetClientId(), reqarg);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::Action(UInt32 actionId, UInt32 actionData)
{
   ETG_TRACE_USR1_THR(("Send Action(%40s, %u, %u)",
                       GetClientId(),
                       actionId,
                       actionData));

   ActionPayload payload(GetClientId(), actionId, actionData);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::RequestCurrentStatus(UInt32 requestId)
{
   ETG_TRACE_USR1_THR(("Send RequestCurrentStatus(%40s,%u)", GetClientId(), requestId));

   RequestCurrentStatusPayload payload(GetClientId(), requestId);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::SetPopupFilter(const RequestArg& requestArg,
                                     bool disableAll, const PopupPresentationArg& popupPresentationArg)
{
   {
      char str[230];
      sprintf(str,
              "Send SetPopupFilter(%s, {%u, %u, %u}, %u, {%u, %d, %u, %u, %u, %d, %d, %d})",
              GetClientId(), requestArg.RequestId(), requestArg.AppId(),
              requestArg.UserData(), disableAll,
              popupPresentationArg.Priority(),
              popupPresentationArg.Modality(),
              popupPresentationArg.PresentationTime(),
              popupPresentationArg.MinimumPresentationTime(),
              popupPresentationArg.ValidityPeriod(),
              popupPresentationArg.HorizontalAlignment(),
              popupPresentationArg.VerticalAlignment(),
              popupPresentationArg.Scaling());
      ETG_TRACE_USR1_THR(("%s", str));
   }

   bosch::cm::ai::hmi::screenbroker::Arguments::RequestArg reqarg(
      requestArg.RequestId(), requestArg.AppId(),
      requestArg.UserData());

   bosch::cm::ai::hmi::screenbroker::Arguments::PopupPresentationArg popupPresArg(
      popupPresentationArg.Priority(),
      (::bosch::cm::ai::hmi::screenbroker::Types::Modality) popupPresentationArg.Modality(),
      popupPresentationArg.PresentationTime(),
      popupPresentationArg.MinimumPresentationTime(),
      popupPresentationArg.ValidityPeriod(),
      (::bosch::cm::ai::hmi::screenbroker::Types::HorizontalAlignment) popupPresentationArg.HorizontalAlignment(),
      (::bosch::cm::ai::hmi::screenbroker::Types::VerticalAlignment) popupPresentationArg.VerticalAlignment(),
      (::bosch::cm::ai::hmi::screenbroker::Types::Scaling) popupPresentationArg.Scaling(),
      popupPresentationArg.FocusPriority());

   //send request to client
   SetPopupFilterPayload payload(GetClientId(), reqarg, disableAll, popupPresArg);
   sendLocalMessageAsfService(payload);
}


// -------------------------------------------------------------------------

const Char* AsfServiceProxy::GetVersion()
{
   if (_SBServiceProxy)
   {
      return mversion.c_str();
   }
   return "";
}


bool AsfServiceProxy::GetBuildInLayerInfo(std::vector<BuildInLayerInfo>& info)
{
   if (_SBServiceProxy)
   {
      if (_SBServiceProxy->hasBuildInLayers())
      {
         std::vector<Arguments::BuildInLayerInfo> propertyVector = _SBServiceProxy->getBuildInLayers();

         for (size_t i = 0; i < propertyVector.size(); ++i)
         {
            BuildInLayerInfo tmp(propertyVector[i].getLayerId(),
                                 propertyVector[i].getSurfaceId(),
                                 propertyVector[i].getScreenId(),
                                 propertyVector[i].getConnectorName(),
                                 DimensionArg(propertyVector[i].getDimensions().getWidth(), propertyVector[i].getDimensions().getHeight()));
            info.push_back(tmp);
         }
      }
      else
      {
         _SBServiceProxy->sendBuildInLayersGet(*this);
      }
   }

   return (info.size() > 0);
}


// -------------------------------------------------------------------------
void AsfServiceProxy::DoInit()
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::DoRun()
{
   if (_SBServiceProxy)
   {
      _SBServiceProxy->sendVersionGet(*this);
#if defined(SCREENBROKER_WITH_CLIENTSYNC)
      // signal that "run" thread is now track
      if (false == _cond.signal())
      {
         ETG_TRACE_ERR_THR(("Signal run synchronization failed! errno=%d", errno));
      }
#endif
   }
   else
   {
      ETG_TRACE_ERR((" GetVersion signal failed"));
   }
}


// -------------------------------------------------------------------------
void AsfServiceProxy::DoTerminate()
{
}


// -------------------------------------------------------------------------
void AsfServiceProxy::Reset()
{
   SetClientApi(0);
   mClientId.clear();
   mversion.clear();
}


} //Internal
} //Screenbroker
