/* ***************************************************************************************
* FILE:          GetVisibleScreen.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  GetVisibleScreen.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "GetVisibleScreen.h"
#include "hmibase/trace/HMIBaseTraceCmds.h"
#include "bosch/cm/ai/hmi/appctrl/trace/CommandsConst.h"

#include "hmibase/trace/HMIBaseTraceCmds_Trace.h"

using namespace hmibase::trace::command;

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_TRACEDAEMON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL               TR_CHANNEL_HMI_BASE
#define ETG_I_TTFIS_CMD_PREFIX            "HMI_"
#define ETG_I_FILE_PREFIX                 hmibase::trace::command::GetVisibleScreen::
#include "trcGenProj/Header/GetVisibleScreen.cpp.trc.h"
#endif

using namespace ::bosch::cm::ai::hmi::appctrl::trace::Commands;

namespace hmibase {
namespace trace {
namespace command {

GetVisibleScreen& GetVisibleScreen::GetInstance()
{
   static GetVisibleScreen theInstance;
   return theInstance;
}


GetVisibleScreen::GetVisibleScreen() : _currentRequest(ALL_SCENES)
{
   ETG_I_REGISTER_FILE();
}


GetVisibleScreen::~GetVisibleScreen()
{
   ETG_I_UNREGISTER_FILE();
}


void GetVisibleScreen::Configure(HMIBaseTraceCmds* baseComponent)
{
   TraceCmdHandler::Configure(baseComponent);

   if (_stub)
   {
      _stub->RegisterMethodHandler(this);
   }
}


void GetVisibleScreen::onLocalMessage(::boost::shared_ptr< asf::core::PayloadTuple1< GetVisibleScreenPayload > >& payload)
{
   if (_stub)
   {
      if (_timer.isStopped())
      {
         _currentRequest = payload->get0().category;
         _timer.start(*this, 3000, asf::core::Timer::Once);

         if ((_currentRequest == ALL_SCENES)
               || (_currentRequest == MAIN_SCENE_CATEGORY)
               || (_currentRequest == OTHER_SCENE_CATEGORY)
               || (_currentRequest == MAIN_AND_POPUP_SCENE_CATEGORIES))
         {
            _stub->sendTraceQuerySignal(TraceQueryData(0, traceQueryId__GET_CURRENT_MAIN_VIEW, std::vector< ::std::string >()));
         }

         if ((_currentRequest == ALL_SCENES)
               || (_currentRequest == POPUP_SCENE_CATEGORY)
               || (_currentRequest == MAIN_AND_POPUP_SCENE_CATEGORIES))
         {
            _stub->sendTraceQuerySignal(TraceQueryData(0, traceQueryId__GET_CURRENT_VISIBLE_POPUP_VIEW, std::vector< ::std::string >()));
         }
      }
      else
      {
         ETG_TRACE_FATAL(("Previous request not finished yet. Current request will not be processed"));
      }
   }
}


void GetVisibleScreen::onExpired(::asf::core::Timer& timer, ::boost::shared_ptr<asf::core::TimerPayload> /*payload*/)
{
   ETG_TRACE_FATAL(("Timer expired"));
   if (&timer == &_timer)
   {
      ETG_TRACE_FATAL(("Force output on HMI_GET_VISIBLE_SCREEN after timeout"));
      print(true);
   }
}


bool GetVisibleScreen::Handle(const ::boost::shared_ptr< ::bosch::cm::ai::hmi::appctrl::trace::Commands::UpdateTraceQueryResultRequest >& data)
{
   traceQueryId queryId = data->getData().getQueryId();
   if ((queryId == traceQueryId__GET_CURRENT_MAIN_VIEW)
         || (queryId == traceQueryId__GET_CURRENT_VISIBLE_POPUP_VIEW)
         || (queryId == traceQueryId__GET_CURRENT_OTHER_VIEW))
   {
      std::string s;
      _stub->GetClientNameFromPid(data->getData().getPid(), s);
      if (_timer.isActive())
      {
         if (_visibleScreenData.empty())
         {
            std::vector<pid_t> pids;
            _stub->GetClientPids(pids);
            for (std::vector<pid_t>::iterator it = pids.begin(); it != pids.end(); ++it)
            {
               visibleScreenInfo d;

               // fill unrequested categories with dummy data
               switch (_currentRequest)
               {
                  case MAIN_SCENE_CATEGORY:
                  {
                     d.popupSceneDataReceived = true;
                     d.otherSceneDataReceived = true;
                  }
                  break;
                  case POPUP_SCENE_CATEGORY:
                  {
                     d.mainSceneDataReceived = true;
                     d.otherSceneDataReceived = true;
                  }
                  break;
                  case OTHER_SCENE_CATEGORY:
                  {
                     d.popupSceneDataReceived = true;
                     d.mainSceneDataReceived = true;
                  }
                  break;
                  case MAIN_AND_POPUP_SCENE_CATEGORIES:
                  {
                     d.otherSceneDataReceived = true;
                  }
                  break;
                  default:
                     break;
               }

               _visibleScreenData[*it] = d;
            }
         }

         std::map<int, visibleScreenInfo>::iterator iter = _visibleScreenData.find(data->getData().getPid());

         if (iter != _visibleScreenData.end())
         {
            if (queryId == traceQueryId__GET_CURRENT_MAIN_VIEW)
            {
               iter->second.mainSceneDataReceived = true;
               iter->second.mainScenes = data->getData().getQueryInput();

               // split into main scenes and others like status or footer bar
               for (std::vector<std::string>::iterator it = iter->second.mainScenes.begin(); it != iter->second.mainScenes.end(); ++it)
               {
                  ETG_TRACE_USR1(("Update on GET_CURRENT_MAIN_VIEW [pid %25s], %s", s.c_str(), it->c_str()));
               }
            }
            else if (queryId == traceQueryId__GET_CURRENT_OTHER_VIEW)
            {
               iter->second.otherSceneDataReceived = true;
               iter->second.otherScenes = data->getData().getQueryInput();

               // split into main scenes and others like status or footer bar
               for (std::vector<std::string>::iterator it = iter->second.otherScenes.begin(); it != iter->second.otherScenes.end(); ++it)
               {
                  ETG_TRACE_USR1(("Update on GET_CURRENT_OTHER_VIEW [pid %25s], %s", s.c_str(), it->c_str()));
               }
            }
            else if (queryId == traceQueryId__GET_CURRENT_VISIBLE_POPUP_VIEW)
            {
               iter->second.popupSceneDataReceived = true;
               iter->second.popupScenes = data->getData().getQueryInput();

               for (std::vector<std::string>::iterator it = iter->second.popupScenes.begin(); it != iter->second.popupScenes.end(); ++it)
               {
                  ETG_TRACE_USR1(("Update on GET_CURRENT_VISIBLE_POPUP_VIEW [pid %25s], %s", s.c_str(), it->c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("Unsupported query Id %d", queryId));
            }
            print();
         }
      }
      else
      {
         // answer after timer has stopped, so info is outdated
         ETG_TRACE_ERR(("view info from %32s received after timeout, queryId %d", s.c_str(), queryId));
      }
      return true;
   }
   return false;
}


void GetVisibleScreen::print(bool force)
{
   bool print = true;
   for (std::map<int, visibleScreenInfo>::iterator it = _visibleScreenData.begin(); it != _visibleScreenData.end(); ++it)
   {
      if (force)
      {
         std::string s;
         _stub->GetClientNameFromPid(it->first, s);
         if (it->second.mainSceneDataReceived == false)
         {
            ETG_TRACE_FATAL(("\tno answer from %s for main scenes", s.c_str()));
         }
         if (it->second.popupSceneDataReceived == false)
         {
            ETG_TRACE_FATAL(("\tno answer from %s for popup scenes", s.c_str()));
         }
         if (it->second.otherSceneDataReceived == false)
         {
            ETG_TRACE_FATAL(("\tno answer from %s for other scenes", s.c_str()));
         }
      }
      print = (print && it->second.mainSceneDataReceived && it->second.popupSceneDataReceived && it->second.otherSceneDataReceived);
   }

   if (print || force)
   {
      std::string mainScenes;
      std::string popupScenes;
      std::string otherScenes;

      _timer.stop();

      for (std::map<int, visibleScreenInfo>::iterator it = _visibleScreenData.begin(); it != _visibleScreenData.end(); ++it)
      {
         for (std::vector<std::string>::iterator viewIterator = it->second.mainScenes.begin(); viewIterator != it->second.mainScenes.end(); ++viewIterator)
         {
            if (viewIterator->empty() == false)
            {
               mainScenes += "\n\t" + *viewIterator;
            }
         }
         for (std::vector<std::string>::iterator viewIterator = it->second.popupScenes.begin(); viewIterator != it->second.popupScenes.end(); ++viewIterator)
         {
            if (viewIterator->empty() == false)
            {
               popupScenes += "\n\t" + *viewIterator;
            }
         }
         for (std::vector<std::string>::iterator viewIterator = it->second.otherScenes.begin(); viewIterator != it->second.otherScenes.end(); ++viewIterator)
         {
            if (viewIterator->empty() == false)
            {
               otherScenes += "\n\t" + *viewIterator;
            }
         }
      }

      std::string s;

      if ((_currentRequest == ALL_SCENES)
            || (_currentRequest == MAIN_SCENE_CATEGORY)
            || (_currentRequest == MAIN_AND_POPUP_SCENE_CATEGORIES))
      {
         s = mainScenes;
      }
      if ((_currentRequest == ALL_SCENES)
            || (_currentRequest == POPUP_SCENE_CATEGORY)
            || (_currentRequest == MAIN_AND_POPUP_SCENE_CATEGORIES))
      {
         if (popupScenes.empty() == false)
         {
            s += popupScenes;
         }
      }
      if ((_currentRequest == ALL_SCENES)
            || (_currentRequest == OTHER_SCENE_CATEGORY))
      {
         if (otherScenes.empty() == false)
         {
            s += otherScenes;
         }
      }

      ETG_TRACE_FATAL(("@TML %s", s.c_str()));

      _visibleScreenData.clear();
      _currentRequest = ALL_SCENES;
   }
}


ETG_I_CMD_DEFINE((TraceCommand, "Get_Visible_Screen"))
ETG_I_CMD_DEFINE((TraceCommand, "Get_Visible_Screen %d", ETG_I_CENUM(enSceneCategory)))
void GetVisibleScreen::TraceCommand(enSceneCategory e)
{
   if (GetInstance()._stub)
   {
      GetVisibleScreenPayload payload(e);
      GetInstance()._stub->sendLocalMessage(GetInstance(), payload);
   }
}


}
}


}
