/* ***************************************************************************************
* FILE:          HMIBaseTraceCmds.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIBaseTraceCmds.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "HMIBaseTraceCmds.h"
#include "hmibase/util/StringUtils.h"
#include <systemd/sd-daemon.h>

#include "hmibase/trace/command/AbortSBAnimation.h"
#include "hmibase/trace/command/CollectTextMetaData.h"
#include "hmibase/trace/command/DumpCanderaPerfMeasures.h"
#include "hmibase/trace/command/DumpScene.h"
#include "hmibase/trace/command/DumpScreen.h"
#include "hmibase/trace/command/EnableArabicPatch.h"
#include "hmibase/trace/command/ForceRedraw.h"
#include "hmibase/trace/command/GetAllVisibleTextIDInListItem.h"
#include "hmibase/trace/command/GetAllVisibleTextInAllListWidgets.h"
#include "hmibase/trace/command/GetAllVisibleTextInGivenList.h"
#include "hmibase/trace/command/GetAllVisibleTextInView.h"
#include "hmibase/trace/command/GetCourierMessageStatistics.h"
#include "hmibase/trace/command/GetFEATMemoryStatistics.h"
#include "hmibase/trace/command/GetFocusedWidgetInstanceInfo.h"
#include "hmibase/trace/command/GetPopupStack.h"
#include "hmibase/trace/command/GetPossibleTouchPointForElemWithName.h"
#include "hmibase/trace/command/GetPossibleTouchPointForElemWithText.h"
#include "hmibase/trace/command/GetPossibleTouchPointForElemWithTextId.h"
#include "hmibase/trace/command/GetPossibleTouchPointForListElem.h"
#include "hmibase/trace/command/GetScreenInfo.h"
#include "hmibase/trace/command/GetSMInfo.h"
#include "hmibase/trace/command/GetStartupInvestigation.h"
#include "hmibase/trace/command/GetTopActiveSurface.h"
#include "hmibase/trace/command/GetTopActiveSurfaceNonPermanent.h"
#include "hmibase/trace/command/GetTopPopup.h"
#include "hmibase/trace/command/GetTopVisiblePopup.h"
#include "hmibase/trace/command/GetTouchSessionInfo.h"
#include "hmibase/trace/command/GetVisibleScreen.h"
#include "hmibase/trace/command/GetWidgetInstanceInfo.h"
#include "hmibase/trace/command/GetWidgetPropertyInfo.h"
#include "hmibase/trace/command/SceneAction.h"
#include "hmibase/trace/command/SetCgiTraceLevel.h"
#include "hmibase/trace/command/SetLocale.h"
#include "hmibase/trace/command/SetWidgetProperty.h"
#include "hmibase/trace/command/SimEnterKey.h"
#include "hmibase/trace/command/SimJoystick.h"
#include "hmibase/trace/command/SimRotateEncoder.h"
#include "hmibase/trace/command/SimSetFocus.h"
#include "hmibase/trace/command/SimTouchByIndex.h"
#include "hmibase/trace/command/SimTouchByName.h"
#include "hmibase/trace/command/SimTouchByText.h"
#include "hmibase/trace/command/SimTouchByTextId.h"
#include "hmibase/trace/command/SimTouch.h"
#include "hmibase/trace/command/ToggleInvalidation.h"
#include "hmibase/trace/command/TraverseViews.h"
#include "hmibase/trace/command/ToggleRenderStatisticsOverlay.h"
#include "hmibase/trace/command/StartAnimation.h"
#include "hmibase/trace/command/SetFrameRate.h"
#include "hmibase/trace/command/GetAllLoadedScenes.h"

#include "hmibase/trace/HMIBaseTraceCmds_Trace.h"

using namespace ::bosch::cm::ai::hmi::appctrl::trace::Commands;

// need this here until only one hierarchy level is supported in ETG_I_CENUM
using namespace hmibase::trace;


#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_TRACEDAEMON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL               TR_CHANNEL_HMI_BASE
#define ETG_I_TTFIS_CMD_PREFIX            "HMI_"
#define ETG_I_FILE_PREFIX                 hmibase::trace::HMIBaseTraceCmds::
#include "trcGenProj/Header/HMIBaseTraceCmds.cpp.trc.h"
#endif

using namespace ::asf::core;
using namespace ::hmibase::util;


namespace hmibase {
namespace trace {

HMIBaseTraceCmds::HMIBaseTraceCmds()
   : bosch::cm::ai::hmi::appctrl::trace::Commands::CommandsStub("port")
{
   vInitPlatformEtg();

   ETG_I_REGISTER_CHN(TraceCmd_NotProcessedMsg);
   ETG_I_REGISTER_FILE();

   //_lcmAppComponent.setAppStartReady();
   sd_notifyf(0, "READY=1\n"	// Tells the init system that daemon startup is finished.
              // This is only used by systemd if the service definition file has Type=notify set.
              "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
              // This is free-form and can be used for various purposes
              "MAINPID=%lu\nPARENTID=%lu",
              (unsigned long) getpid(),
              (unsigned long) getppid());

   command::AbortSBAnimation::GetInstance().Configure(this);
   command::CollectTextMetaData::GetInstance().Configure(this);
   command::DumpCanderaPerfMeasures::GetInstance().Configure(this);
   command::DumpScene::GetInstance().Configure(this);
   command::DumpScreen::GetInstance().Configure(this);
   command::EnableArabicPatch::GetInstance().Configure(this);
   command::ForceRedraw::GetInstance().Configure(this);
   command::GetAllVisibleTextIDInListItem::GetInstance().Configure(this);
   command::GetAllVisibleTextInAllListWidgets::GetInstance().Configure(this);
   command::GetAllVisibleTextInGivenList::GetInstance().Configure(this);
   command::GetAllVisibleTextInView::GetInstance().Configure(this);
   command::GetCourierMessageStatistics::GetInstance().Configure(this);
   command::GetFEATMemoryStatistics::GetInstance().Configure(this);
   command::GetFocusedWidgetInstanceInfo::GetInstance().Configure(this);
   command::GetPopupStack::GetInstance().Configure(this);
   command::GetPossibleTouchPointForElemWithName::GetInstance().Configure(this);
   command::GetPossibleTouchPointForElemWithText::GetInstance().Configure(this);
   command::GetPossibleTouchPointForElemWithTextId::GetInstance().Configure(this);
   command::GetPossibleTouchPointForListElem::GetInstance().Configure(this);
   command::GetScreenInfo::GetInstance().Configure(this);
   command::GetSMInfo::GetInstance().Configure(this);
   command::GetStartupInvestigation::GetInstance().Configure(this);
   command::GetTopActiveSurface::GetInstance().Configure(this);
   command::GetTopActiveSurfaceNonPermanent::GetInstance().Configure(this);
   command::GetTopPopup::GetInstance().Configure(this);
   command::GetTopVisiblePopup::GetInstance().Configure(this);
   command::GetTouchSessionInfo::GetInstance().Configure(this);
   command::GetVisibleScreen::GetInstance().Configure(this);
   command::GetWidgetInstanceInfo::GetInstance().Configure(this);
   command::GetWidgetPropertyInfo::GetInstance().Configure(this);
   command::SceneAction::GetInstance().Configure(this);
   command::SetCgiTraceLevel::GetInstance().Configure(this);
   command::SetLocale::GetInstance().Configure(this);
   command::SetWidgetProperty::GetInstance().Configure(this);
   command::SimEnterKey::GetInstance().Configure(this);
   command::SimJoystick::GetInstance().Configure(this);
   command::SimRotateEncoder::GetInstance().Configure(this);
   command::SimSetFocus::GetInstance().Configure(this);
   command::SimTouchByIndex::GetInstance().Configure(this);
   command::SimTouchByName::GetInstance().Configure(this);
   command::SimTouchByText::GetInstance().Configure(this);
   command::SimTouchByTextId::GetInstance().Configure(this);
   command::SimTouch::GetInstance().Configure(this);
   command::ToggleInvalidation::GetInstance().Configure(this);
   command::TraverseViews::GetInstance().Configure(this);
   command::ToggleRenderStatisticsOverlay::GetInstance().Configure(this);
   command::StartAnimation::GetInstance().Configure(this);
   command::SetFrameRate::GetInstance().Configure(this);
   command::GetAllLoadedScenes::GetInstance().Configure(this);
}


HMIBaseTraceCmds::~HMIBaseTraceCmds()
{
   ETG_I_UNREGISTER_FILE();
   ETG_I_UNREGISTER_CHN();
}


void HMIBaseTraceCmds::RegisterMethodHandler(::hmibase::trace::TraceQueryResponseHandler* handler)
{
   _traceQueryResponseHandler.insert(handler);
}


bool HMIBaseTraceCmds::getClientsPid(const std::string& clientName, std::vector<pid_t>& result)
{
   if (_clientMap.empty())
   {
      ETG_TRACE_ERR(("No clients registered"));
      return false;
   }
   else
   {
      std::map<std::string, clientInfo>::iterator it = clientName.empty() ? _clientMap.begin() : _clientMap.find(clientName);

      if (it != _clientMap.end())
      {
         ETG_TRACE_USR1(("Matching client %s found", clientName.c_str()));
         result.push_back(it->second.pid); // name match found
      }
      else
      {
         for (it = _clientMap.begin(); it != _clientMap.end(); ++it)
         {
            if (it->first.find(clientName) != std::string::npos)
            {
               ETG_TRACE_USR1(("SubString match %25s found in process name %s", clientName.c_str(), it->first.c_str()));
               result.push_back(it->second.pid); // name substring match found
            }
         }

         if (result.size() == 0)
         {
            ETG_TRACE_FATAL(("No match for name %25s, return first registered client %s", clientName.c_str(), _clientMap.begin()->first.c_str()));
            ETG_TRACE_FATAL(("Known clients:"));
            for (it = _clientMap.begin(); it != _clientMap.end(); ++it)
            {
               ETG_TRACE_FATAL(("\t- %s", it->first.c_str()));
            }
            result.push_back(_clientMap.begin()->second.pid); // return first element when no match for name found
         }
      }
   }

   return true;
}


void HMIBaseTraceCmds::GetClientNameFromPid(pid_t pid, std::string& s)
{
   std::map<std::string, clientInfo>::iterator it;
   for (it = _clientMap.begin(); it != _clientMap.end(); ++it)
   {
      if (it->second.pid == pid)
      {
         s = it->first;
      }
   }
}


void HMIBaseTraceCmds::onUpdateClientStatusRequest(const ::boost::shared_ptr< bosch::cm::ai::hmi::appctrl::trace::Commands::UpdateClientStatusRequest >& request)
{
   clientInfo c;
   c.pid = static_cast<pid_t>(request->getPid());
   c.surfaceIds = request->getSurfaces();
   _clientMap[request->getName()] = c;
}


void HMIBaseTraceCmds::onUpdateTraceQueryResultRequest(const ::boost::shared_ptr< UpdateTraceQueryResultRequest >& data)
{
   for (std::set<hmibase::trace::TraceQueryResponseHandler*>::iterator it = _traceQueryResponseHandler.begin(); it != _traceQueryResponseHandler.end(); ++it)
   {
      (*it)->Handle(data);
   }
}


void HMIBaseTraceCmds::GetClientPids(std::vector<pid_t>& pids)
{
   for (std::map<std::string, clientInfo>::iterator it = _clientMap.begin(); it != _clientMap.end(); ++it)
   {
      pids.push_back(it->second.pid);
   }
}


// sink for not processed ttfis input messages
void HMIBaseTraceCmds::TraceCmd_NotProcessedMsg(const unsigned char* pcu8Data)
{
   if (pcu8Data)
   {
      ETG_TRACE_FATAL(("TraceCmd_NotProcessedMsg() : 0x%*x", ETG_LIST_LEN(pcu8Data[0]), ETG_LIST_PTR_T8(pcu8Data)));
   }
}


}
}
