/* ***************************************************************************************
* FILE:          ViewIdentifier.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewIdentifier.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef __ViewIdentifier_h_
#define __ViewIdentifier_h_

#include <string>
#include <climits>
#include <functional>
#include <string.h>


namespace hmibase {
namespace view {

class ViewIdentifier
{
   public:
      ViewIdentifier() : _viewStr(""), _viewId(0)
      {
      }

      explicit ViewIdentifier(const char* viewName) : _viewStr(viewName)
      {
#if __cplusplus > 199711L  //c++11
         std::string str(_viewStr);
         _viewId = std::hash<std::string> {}(str);
#else
         _viewId = calcHash(viewName);
#endif
      }

      ViewIdentifier(const ViewIdentifier& id) : _viewStr(id._viewStr), _viewId(id._viewId)
      {
      }

      ViewIdentifier& operator=(const ViewIdentifier& id)
      {
         if (this != &id)
         {
            _viewStr = id._viewStr;
            _viewId = id._viewId;
         }
         return *this;
      }

      const char* CStr() const
      {
         return _viewStr;
      }

      unsigned int GetHashId() const
      {
         return _viewId;
      }

      bool operator==(const ViewIdentifier& id) const
      {
         return ((id._viewId == _viewId && _viewId != 0) || (strcmp(id._viewStr, _viewStr) == 0));
      }

      bool operator!=(const ViewIdentifier& id) const
      {
         return !((id._viewId == _viewId && _viewId != 0) || (strcmp(id._viewStr, _viewStr) == 0));
      }

   private:
      unsigned int calcHash(std::string str)
      {
         const char* s = str.c_str();
         register unsigned int hash = 0;
         unsigned int seed = 11;
         for (;;)
         {
            while (*s != '\0')
            {
               hash += (unsigned int)static_cast<unsigned char>(*s);
               hash += (hash << 10);
               hash ^= (hash >> 6);
               ++s;
            }

            hash += (hash << 3);
            hash ^= (hash >> 11);
            hash += (hash << 15);

            if (hash != /*0xFFFFFFFF*/ UINT_MAX)
            {
               break;
            }
            hash = seed;
            seed += 2;
         }
         return hash;
      }

      const char*   _viewStr;
      unsigned int  _viewId;
};


} // namespace view
} // namespace hmibase


#endif // guard
