/* ***************************************************************************************
* FILE:          Widget.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Widget.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef HMIBASE_VIEW_WIDGET_H
#define HMIBASE_VIEW_WIDGET_H

#include "DelegateBase.h"
#include <Courier/Visualization/FrameworkWidget.h>
#include <TouchGestures/GestureListener.h>
#include <AppUtils/ObjectGuard.h>
#include <bitset>

namespace Courier {
struct TouchInfo;
}


namespace Candera {
class CanderaObject;
class Vector2;
class WidgetBase;
class Node;
class Node2D;
}


class BaseWidget2D;
class BaseWidget3D;


namespace hmibase {
namespace widget {
namespace flags {
enum Enum_Widget
{
   Active,
   Enabled,
   Focused,
   Pressed,
   Touched,
   Visible,
   ControllerValid,
   ItemTemplate,
   ItemClone,

   //last value
   _LastValue_Widget,

   //max value used for bit set
   _MaxValue = 32
};


}

class WidgetCloneHelper;
class WidgetController;
class WidgetControllerData;
class WidgetControllerEvent;

/*****************************************************************************/
class Widget : public Courier::FrameworkWidget, public hmibase::input::gesture::GestureListener, public hmibase::util::GuardedObject, public DelegateBase
{
      typedef Courier::FrameworkWidget Base;

   public:
      HMIBASE_TYPEDEF_OBJECT_ACCESSOR(Widget);

      Widget();
      virtual ~Widget();

      /* returns the default widget controller */
      virtual WidgetController* GetDefaultController() const;

      /* sets the controller id */
      void SetControllerId(ControllerIdType id);

      /* processes an event */
      virtual bool OnEvent(const WidgetControllerEvent& e);

      /* prevents the dynamic cast */
      virtual const BaseWidget2D* ToBaseWidget2D() const;

      /* prevents the dynamic cast */
      virtual BaseWidget2D* ToBaseWidget2D();

      /* returns true if this is a 2D widget */
      virtual bool Is2D() const;

      /* returns the associated node for 2D widgets or NULL otherwise */
      virtual Candera::Node2D* GetNode2D() const;

      /* prevents the dynamic cast */
      virtual const BaseWidget3D* ToBaseWidget3D() const;

      /* prevents the dynamic cast */
      virtual BaseWidget3D* ToBaseWidget3D();

      /* returns true if this is a 3D widget */
      virtual bool Is3D() const;

      /* returns the associated node for 3D widgets or NULL otherwise */
      virtual Candera::Node* GetNode3D() const;

      /* Wakes up all render components */
      void triggerUpdate();

      /* Called when the value of a property is changed */
      virtual void OnChanged(FeatStd::UInt32 propertyId);

      /* used for cloning inside the list widget */
      virtual bool CloneFrom(const Candera::WidgetBase*, WidgetCloneHelper&);

      /* prepares this widget for cloning, returns true if it is allowed to clone it */
      virtual bool PrepareForCloning();

      /* enables or disables the usage of dirty rectangle for all widgets */
      static void useDirtyRectangle(bool useDirtyRect);

      /*  */
      static bool isDirtyRectangleEnabled();

      /* returns the widget name as it was in CGi Studio 2.x (for example CompositeNodeName/CompositeNodeName/WidgetName) */
      virtual const FeatStd::Char* GetLegacyName();

      /* returns the id of the parent view or string null if no parent view is set. */
      const FeatStd::Char* GetViewName() const;

      /* The function prototype for detecting a touched 2D widget */
      typedef bool(*IsInsideBoundingBox2DFct)(Widget* view2D, const Courier::TouchInfo& info);

      /* Allow setting an own implementation of the IsInsideBoundingBox2DFct */
      static void SetBoundingBoxCheckFunction(IsInsideBoundingBox2DFct fct);

      /* for those widgets that can be identified via an instance id and receive/support forwarded touch */
      virtual bool IsVirtuallyTouched(FeatStd::UInt32 instanceID);

      /* inform about received touch */
      virtual bool OnTouchGeneric(const Candera::CanderaObject& camera, const Candera::Vector2& pos, FeatStd::Float dist = 0.0F);

      /* returns the touchable rectangle in scene coordinates */
      virtual Candera::Rectangle GetWorldTouchableRectangle() const;

      /* used to sort widgets when a touch session is started */
      virtual FeatStd::UInt32 GetEffectiveTouchPriority() const;

      /* return list of supported gestures */
      virtual hmibase::input::gesture::GestureListener::Gestures GetGestureList() const override;

      /* returns true if the associated node is effectively rendered. */
      virtual bool IsEffectiveVisible() const;

      /************************************************************************/
#define DefWidgetFlag(FlagName, PropId)\
      static const FeatStd::UInt32 FlagName##PropertyId = PropId;\
      typedef bool FlagName##Type;\
      inline bool Is##FlagName() const\
      {\
         return _flags.test(static_cast<size_t>(::hmibase::widget::flags::FlagName));\
      }\
      inline void Set##FlagName##Internal(bool value)\
      {\
         _flags.set(static_cast<size_t>(::hmibase::widget::flags::FlagName), value);\
      }\
      void Set##FlagName(bool value)\
      {\
         SetFlagValue(FlagName##PropertyId, static_cast<size_t>(::hmibase::widget::flags::FlagName), value);\
      }


#define DefWidgetProperty(PropName, PropType, PropId)\
      static const FeatStd::UInt32 PropName##PropertyId = PropId;\
      typedef PropType PropName##Type;\
      inline const PropType& Get##PropName() const\
      {\
         return _##PropName.Value;\
      }\
      inline void Set##PropName##Internal(const PropType& value)\
      {\
         _##PropName.Value = value;\
      }

#define DefWidgetProperty_Field(PropName)\
      struct PropName##InternalFieldType\
      {\
         PropName##InternalFieldType() : Value(PropName##Type()) {}\
         PropName##Type Value;\
      } _##PropName;\

#define DefWidgetProperty_NonConstGetter(PropName)\
      inline PropType& Get##PropName()\
      {\
         return _##PropName.Value;\
      }\

#define DefWidgetProperty_SimpleSetter(PropName)\
      void Set##PropName(const PropName##Type& value)\
      {\
         Set##PropName##Internal(value);\
      }

#define DefWidgetProperty_SetterWithNotifier(PropName)\
      void Set##PropName(const PropName##Type& value)\
      {\
         Set##PropName##Internal(value);\
         OnChanged(PropName##PropertyId);\
      }

#define DefWidgetProperty_SetterWithCheckAndNotifier(PropName)\
      void Set##PropName(const PropName##Type& value)\
      {\
         if (Get##PropName() != value)\
         {\
            Set##PropName##Internal(value);\
            OnChanged(PropName##PropertyId);\
         }\
      }

#define DefWidgetProperty_VisibilityTest(PropName)\
      inline bool IsPropertyVisible_##PropName() const\
      {\
         return IsPropertyVisible(PropName##PropertyId);\
      }\

      /* FLAGS: Enabled, Visible, etc */
      DefWidgetFlag(Active, 0x39E1550C)
      DefWidgetFlag(Enabled, 0x769B5CD3)
      DefWidgetFlag(Focused, 0x1E0A7678)
      DefWidgetFlag(Pressed, 0x336C3A04)
      DefWidgetFlag(Touched, 0xF14D9671)
      DefWidgetFlag(Visible, 0xE1A806A2)
      DefWidgetFlag(ControllerValid, 0)
      DefWidgetFlag(ItemTemplate, 0)
      DefWidgetFlag(ItemClone, 0)

      /* user data, data bindable property which is used as additional parameter to many messages */
      DefWidgetProperty(UserData, FeatStd::UInt32, 0xB3F3CDD5)
      DefWidgetProperty_SimpleSetter(UserData)

#define CdaProperty_UserData()\
      CdaBindableProperty(UserData, UserDataType, GetUserData, SetUserData)\
      CdaDescription("Together with the view and the widget identifier, this user data is a parameter to many messages posted by the widgets which can be used in the state machine or in the data model. Use data binding to change this value dynamically and store extra information in the widgets.")\
      CdaBindablePropertyEnd()

   protected:
      virtual void SetFlagValue(FeatStd::UInt32 propertyId, size_t pos, bool value);

      /* returns true if the property should be visible in scene composer. widgets should overwrite this. */
      virtual bool IsPropertyVisible(FeatStd::UInt32 propertyId) const;

      std::bitset<flags::_MaxValue> _flags;

      DefWidgetProperty_Field(UserData)

      static IsInsideBoundingBox2DFct _fkt_IsInsideBoundingBox2DFunction;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(Widget);

      static bool _useDirtyRectangle;
};


}

namespace view {
typedef hmibase::widget::Widget Widget;
}


}

/** Candera expects that the injected widget base class is named CanderaInjectedWidgetBase */

//FEATSTD_LINT_NEXT_EXPRESSION(1960, "this is a correct global typedef to guarantee backwards compatibility")
typedef hmibase::widget::Widget CanderaInjectedWidgetBase;

#endif
