/* ***************************************************************************************
* FILE:          DelegateBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetController.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "DelegateBase.h"
#include "WidgetController.h"
#include <CanderaWidget/WidgetBase/WidgetBase.h>

#include <hmi_trace_if.h>
#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_WIDGET
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DelegateBase.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {


/*****************************************************************************/
DelegateBase::DelegateBase() : _controllerInfo(NULL)
{
}


/*****************************************************************************/
DelegateBase::~DelegateBase()
{
   if (_controllerInfo != NULL)
   {
      if (_controllerInfo->Controller != NULL)
      {
         ETG_TRACE_COMP(("Controller %p with id %d not properly detached [%p]", _controllerInfo->Controller, _controllerInfo->ControllerId, this));
      }

      FEATSTD_DELETE(_controllerInfo);
      _controllerInfo = NULL;
   }
}


/*****************************************************************************/
void DelegateBase::ClearController(DelegateWidget& widget)
{
   SetController(widget, NULL);
}


/*****************************************************************************/
void DelegateBase::SetControllerId(DelegateWidget& widget, ControllerIdType id)
{
   if (_controllerInfo != NULL)
   {
      if (_controllerInfo->ControllerId != id)
      {
         _controllerInfo->ControllerId = id;
         widget.SetControllerValidInternal(false);
         UpdateController(widget);
         widget.OnChanged(ControllerIdPropertyId);
      }
   }
   else
   {
      //happens for widgets in list when CloneFrom is called
      if (id == 0)
      {
         UpdateController(widget);
      }
      else if (id > 0)
      {
         _controllerInfo = FEATSTD_NEW(ControllerInfo)(id);
         widget.SetControllerValidInternal(false);
         UpdateController(widget);
         widget.OnChanged(ControllerIdPropertyId);
      }
      else
      {
         //nothing to do
      }
   }
}


/*****************************************************************************/
void DelegateBase::SetController(DelegateWidget& widget, WidgetController* controller)
{
   if ((_controllerInfo != NULL) && (_controllerInfo->Controller != controller))
   {
      // detach from old controller
      if (_controllerInfo->Controller != NULL)
      {
         _controllerInfo->Controller->Detach(widget);
      }

      _controllerInfo->Controller = controller;

      //attach to new controller
      if (_controllerInfo->Controller != NULL)
      {
         _controllerInfo->Controller->Attach(widget);
      }
   }
}


/*****************************************************************************/
bool DelegateBase::UpdateController(DelegateWidget& widget)
{
   bool result = true;
   if (!widget.IsControllerValid())
   {
      widget.SetControllerValidInternal(true);

      WidgetController* controller = NULL;
      if ((_controllerInfo == NULL) || (_controllerInfo->ControllerId == 0))
      {
         controller = widget.GetDefaultController();
         if (controller != NULL)
         {
            ETG_TRACE_USR1(("UpdateController default controller=%p %s", controller, HMIBASE_TO_STRING_VW(&widget)));

            _controllerInfo = FEATSTD_NEW(ControllerInfo);
         }
      }
      else if (_controllerInfo->ControllerId > 0)
      {
         controller = WidgetControllerRegistry::getItem(static_cast<WidgetControllerRegistry::IdType>(static_cast<FeatStd::Int>(_controllerInfo->ControllerId)));//many casts to make lint satisfied
         if (controller != NULL)
         {
            ETG_TRACE_USR1(("UpdateController controllerId=%d controller=%p %s",
                            _controllerInfo->ControllerId, controller, HMIBASE_TO_STRING_VW(&widget)));
         }
         else
         {
            ETG_TRACE_ERR(("UpdateController No controller registered for id=%d %s",
                           _controllerInfo->ControllerId, HMIBASE_TO_STRING_VW(&widget)));
            result = false;
         }
      }
      else
      {
         //no controller
      }

      SetController(widget, controller);
   }
   return result;
}


}
}
