/* ***************************************************************************************
* FILE:          WindowInputEventHook.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WindowInputEventHook.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(AppPlatform_Impl_WindowInputEventHook_h)
#define AppPlatform_Impl_WindowInputEventHook_h


#include <View/CGI/InputHandling/Wayland/WaylandEventHook.h>
#include "AppUtils/Timer.h"

#include "AppBase/ScreenBrokerClient/IUserInputObserver.h"
#include "hmibase/util/Semaphore.h"

class KeyMappingBase;

namespace Courier {
namespace TouchHandling {
class TouchEventPreprocessor;
}


}

class WindowInputEventHook : public Courier::InputHandling::Wayland::WaylandEventHook
{
   private:
      struct PointerInfo
      {
         bool valid; // indicate if position information is valid
         bool pointerDownPending; // true if pointer down received but no valid position at that time
         int x; // last known x position
         int y; // last known y position

         PointerInfo()
         {
            Reset();
         }

         void Reset()
         {
            valid = false;
            pointerDownPending = false;
            x = 0;
            y = 0;
         }

         void SetPosition(wl_fixed_t xPos, wl_fixed_t yPos)
         {
            valid = true;
            x = wl_fixed_to_int(xPos);
            y = wl_fixed_to_int(yPos);
         }
      };

      PointerInfo _pointerInfo;

      int _lLastTouchXCoord;
      int _lLastTouchYCoord;
      std::string _appName;
      uint32_t _pointerTouchedSurfaceID;
      uint32_t _touchTouchedSurfaceID;
      uint32_t _keyboardSurfaceID;
      KeyMappingBase* _mKeyMapping;
      uint32_t _unifiedKeyCode;
      bool _bkeyBoardFocusGained;

      ::Util::Timer _longPressKeyTimer;
      ::Util::Timer _repeatKeyTimer;

      ::hmibase::util::Mutex _criticalSec;

      ::hmibase::input::IUserInputObserver* _uiObserver;
      Courier::TouchHandling::TouchEventPreprocessor* _touchEventPreprocessor;

      bool _multiFingerInputEnabled;

      uint32_t GetUnifiedKeyCode(uint32_t key) const;
      uint32_t GetTouchedSurface(void* data, struct wl_surface* surface) const;

      //Timers Start and Stop
      void StartLongPressKeyTimers();
      void StopLongPressKeyTimers();
      void StartRepeatKeyTimer();
      void StopRepeatKeyTimer();

      //check if this key is an key that aborts running SB Animations
      //and send the corresponding request to SBClient in this case
      void CheckAnimAbortKey(uint32_t key) const;

   public:

      WindowInputEventHook();

      void initHook(KeyMappingBase* keyMapping);

      virtual ~WindowInputEventHook();

      /// For detailed interface description see Wayland client protocol documentation (wl_pointer_listener)
      virtual Courier::Message* OnPointerEventEnter(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_pointer* pointer,
            uint32_t serial,
            struct wl_surface* surface,
            wl_fixed_t surface_x,
            wl_fixed_t surface_y);

      /// For detailed interface description see Wayland client protocol documentation (wl_pointer_listener)
      virtual Message* OnPointerEventLeave(Courier::InputHandling::Wayland::WLSeat* wlseat,
                                           struct wl_pointer* pointer,
                                           uint32_t serial,
                                           struct wl_surface* surface);

      //handles touch input, till Wayland input device detection is fixed
      virtual Courier::Message* OnPointerEventMotion(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_pointer* pointer,
            uint32_t time,
            wl_fixed_t surface_x,
            wl_fixed_t surface_y);

      //handles touch input, till Wayland input device detection is fixed, if touch contact it set/removed,
      //this method is called
      virtual Courier::Message* OnPointerEventButton(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_pointer* pointer,
            uint32_t serial,
            uint32_t time,
            uint32_t button,
            uint32_t state);

      //handles Encoder
      virtual Courier::Message* OnPointerEventAxis(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_pointer* pointer,
            uint32_t time,
            uint32_t axis,
            wl_fixed_t value);

      //is called if a key event occured and teh keyboard focus is set to this surface
      virtual Courier::Message* OnKeyboardEventKey(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_keyboard* keyboard,
            uint32_t serial,
            uint32_t time,
            uint32_t key,
            uint32_t state);

      /// is called once the surface gains the keyboard focus
      virtual Courier::Message* OnKeyboardEventEnter(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_keyboard* keyboard,
            uint32_t serial,
            struct wl_surface* surface,
            struct wl_array* keys);

      /// is called once the surface looses the keyboard focus
      virtual Courier::Message* OnKeyboardEventLeave(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_keyboard* keyboard,
            uint32_t serial,
            struct wl_surface* surface);

      //handles touch
      virtual Courier::Message* OnTouchEventMotion(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_touch* touch,
            uint32_t time,
            int32_t id,
            wl_fixed_t x,
            wl_fixed_t y);

      //handles touch down
      virtual Courier::Message* OnTouchEventDown(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_touch* touch,
            uint32_t serial,
            uint32_t time,
            struct wl_surface* surface,
            int32_t id,
            wl_fixed_t x,
            wl_fixed_t y);

      //handles touch up
      virtual Courier::Message* OnTouchEventUp(Courier::InputHandling::Wayland::WLSeat* wlseat,
            struct wl_touch* touch,
            uint32_t serial,
            uint32_t time,
            int32_t id);

      void setUserInputObserver(::hmibase::input::IUserInputObserver* uiObserver)
      {
         _uiObserver = uiObserver;
      }

      ///
      void SetTouchEventPreprocessor(Courier::TouchHandling::TouchEventPreprocessor* touchEventPreprocessor)
      {
         _touchEventPreprocessor = touchEventPreprocessor;
      };

      void ResetInputKeyEvent(uint32_t leavingSurfaceID);

      bool EnableMultiFingerInput(bool enable)
      {
         _multiFingerInputEnabled = enable;
         return true;
      }
      // callback function for repeat timer
      static void RepeatTimerCallBack(Courier::Message* msg);
};


#endif
