/* ***************************************************************************************
* FILE:          WindowInputEventHook.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WindowInputEventHook.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "hmi_trace_if.h"

#include "Courier/Foundation/Component.h"
#include "AppBase/SystemConfiguration.h"
#include "View/CGI/CgiExtensions/CourierMessageMapper.h"
//#include "AppBase/app_base_defines.h"
#include <Courier/Diagnostics/Log.h>
#include <Courier/Platform/MessageFactory.h>
#include <View/CGI/InputHandling/Wayland/WaylandContext.h>
#include <View/CGI/InputHandling/DisplayInputContext.h>
#include <View/CGI/InputHandling/InputHandler.h>
#include <CanderaPlatform/Device/Genivi/Target/ILM/GeniviTargetDisplay.h>
#include <ilm/ilm_types.h>
#include "WindowInputEventHook.h"
#include "AppBase/ScreenBrokerClient/KeyMappingBase.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"

#include "View/CGI/TouchHandling/TouchEventPreprocessor.h"

#include "FeatStd//Platform/PerfCounter.h"

//////// TRACE IF ///////////////////////////////////////////////////
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/WindowInputEventHook.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

//////// TRACE IF ///////////////////////////////////////////////////

using namespace ::hmibase;

// These defines are added here, since no other class need it.
#define WL_KEY_RELEASE			0
#define WL_KEY_PRESS			1

// ------------------------------------------------------------------------------------------------
WindowInputEventHook::WindowInputEventHook() :
   _lLastTouchXCoord(-1),
   _lLastTouchYCoord(-1),
   _pointerTouchedSurfaceID(0),
   _touchTouchedSurfaceID(0),
   _keyboardSurfaceID(0),
   _mKeyMapping(0),
   _unifiedKeyCode(hmibase::app::base::keymapbase::KEY_CODE_INVALID),
   _bkeyBoardFocusGained(false),
   _uiObserver(0),
   _touchEventPreprocessor(0),
   _multiFingerInputEnabled(false)
{
   _longPressKeyTimer.setName("WindowInputEventHook", "LongPressTimer");
   _repeatKeyTimer.setName("WindowInputEventHook", "RepeatKeyTimer");
}


// ------------------------------------------------------------------------------------------------
void WindowInputEventHook::initHook(KeyMappingBase* keyMapping)
{
   _mKeyMapping = keyMapping;
   // set environment MULTIFINGER_SUPPORT to enable multi finger input
   _multiFingerInputEnabled = ::getenv("MULTIFINGER_SUPPORT") != 0 ? true : false;
}


WindowInputEventHook::~WindowInputEventHook()
{
   _mKeyMapping = 0;
   _uiObserver = 0;
   _touchEventPreprocessor = 0;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnPointerEventEnter(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_pointer* pointer,
      uint32_t serial,
      struct wl_surface* surface,
      wl_fixed_t surface_x,
      wl_fixed_t surface_y)
{
   FEATSTD_UNUSED2(serial, pointer);
   _pointerTouchedSurfaceID = GetTouchedSurface(wlseat->GetParentWlContext(), surface);
   wlseat->SetPointerSurfaceID(_pointerTouchedSurfaceID);

   if (0 == _pointerTouchedSurfaceID)
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::OnPointerEventEnter touched surface is '0'! app=%50s", hmibase::trace::getAppName().c_str()));
   }
   else
   {
      _pointerInfo.SetPosition(surface_x, surface_y);
   }

   if (_uiObserver)
   {
      _uiObserver->onInputEvent();
   }
   return Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnPointerEventLeave(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_pointer* pointer,
      uint32_t serial,
      struct wl_surface* surface)
{
   FEATSTD_UNUSED3(pointer, serial, surface);

   ETG_TRACE_USR1_THR(("WindowInputEventHook::OnPointerEventLeave is called pointer focus is with surfaceID=%d and app=%50s",
                       wlseat->GetPointerSurfaceID(), hmibase::trace::getAppName().c_str()));
   //reset seat's surfaceID when leaving surface
   wlseat->SetPointerSurfaceID(0);

   _pointerInfo.Reset();

   return Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnPointerEventMotion(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_pointer* pointer,
      uint32_t time,
      wl_fixed_t surface_x,
      wl_fixed_t surface_y)
{
   FEATSTD_UNUSED2(pointer, time);

   _pointerTouchedSurfaceID = wlseat->GetPointerSurfaceID();

   _pointerInfo.SetPosition(surface_x, surface_y);

   // Parameter: State, xPos, yPos, PointerID (ID of finger in case of multi touch),
   // SourceId (ID of connected display, if more than one HW display is connected)
   Courier::Message* lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Move, _pointerInfo.x, _pointerInfo.y, FeatStd::Internal::PerfCounter::Now(), static_cast<Courier::PointerId>(0), _pointerTouchedSurfaceID);

   if (_pointerInfo.pointerDownPending)
   {
      // this is the first move after a down event was received, so send down event first
      Courier::Message* lMsgDown = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down, _pointerInfo.x, _pointerInfo.y, FeatStd::Internal::PerfCounter::Now(), static_cast<Courier::PointerId>(0), _pointerTouchedSurfaceID);
      if (lMsgDown)
      {
         if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsgDown)))
         {
            //lMsgDown = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
         }
         else
         {
            Courier::InputHandling::InputHandler::Post(lMsgDown);
         }
      }

      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnPointerEventButton x=%d y=%d DOWN, pointerID=%d surfaceID=%d app=%50s", _pointerInfo.x, _pointerInfo.y, 0,
                          _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
      _pointerInfo.pointerDownPending = false;
   }

   ETG_TRACE_USR1_THR(("WindowInputEventHook::OnPointerEventMotion x=%d y=%d, pointerID=%d surfaceID=%d app=%50s", _pointerInfo.x, _pointerInfo.y, 0,
                       _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

   if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsg)))
   {
      lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
   }

   if (_uiObserver)
   {
      _uiObserver->onInputEvent();
   }
   return lMsg;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnPointerEventButton(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_pointer* pointer,
      uint32_t serial,
      uint32_t time,
      uint32_t button,
      uint32_t state)
{
   FEATSTD_UNUSED3(pointer, serial, time);
   Courier::Message* lMsg = Courier::InputHandling::InputHandler::cDoDefaultHandling;

   _pointerTouchedSurfaceID = wlseat->GetPointerSurfaceID();

   switch (state)
   {
      // check for now independent of which button was used, since
      // the single touch display does only have one
      case 0:
      {
         if (_pointerInfo.pointerDownPending)
         {
            // no move in between
            _pointerInfo.pointerDownPending = false;

            if (_pointerInfo.valid)
            {
               // ponter information is valid now, so send pending down event right before up
               ETG_TRACE_USR4_THR(("WindowInputEventHook::OnPointerEventButton x=%d y=%d DOWN, pointerID=%d surfaceID=%d app=%50s",
                                   _pointerInfo.x, _pointerInfo.y, 0, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

               Courier::Message* lMsgDown = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down, _pointerInfo.x, _pointerInfo.y, FeatStd::Internal::PerfCounter::Now(), static_cast<Courier::PointerId>(0), _pointerTouchedSurfaceID);
               if (lMsgDown)
               {
                  if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsgDown)))
                  {
                     //lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
                  }
                  else
                  {
                     Courier::InputHandling::InputHandler::Post(lMsgDown);
                  }
               }
            }
         }

         if (_pointerInfo.valid)
         {
            // Contact released, touch up
            // Parameter: State, xPos, yPos, PointerID (ID of finger in case of multi touch),
            // SourceId (ID of connected display, if more than one HW display is connected)
            lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Up, _pointerInfo.x, _pointerInfo.y, FeatStd::Internal::PerfCounter::Now(), static_cast<Courier::PointerId>(0), _pointerTouchedSurfaceID);
            ETG_TRACE_USR4_THR(("WindowInputEventHook::OnPointerEventButton x=%d y=%d UP, pointerID=%d surfaceID=%d app=%50s",
                                _pointerInfo.x, _pointerInfo.y, 0, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

            if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsg)))
            {
               lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
            }
         }
         break;
      }

      case 1:
      {
         if (_pointerInfo.valid)
         {
            ETG_TRACE_USR4_THR(("WindowInputEventHook::OnPointerEventButton DOWN, pointerID=%d surfaceID=%d app=%50s",
                                0, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

            lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down, _pointerInfo.x, _pointerInfo.y, FeatStd::Internal::PerfCounter::Now(), static_cast<Courier::PointerId>(0), _pointerTouchedSurfaceID);
            if (lMsg)
            {
               if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsg)))
               {
                  lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
               }
            }
         }
         else
         {
            // no valid position known, what for position information
            _pointerInfo.pointerDownPending = true;
            ETG_TRACE_USR4_THR(("WindowInputEventHook::OnPointerEventButton DOWN, pointerID=%d surfaceID=%d app=%50s",
                                0, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

            // since the position of the touch is till now not known, the info that a touch down was received, is only stored
            // and send once a motion is received (since then the touch coordinates are known).
         }
         break;
      }

      default:
         ETG_TRACE_ERR_THR(("WindowInputEventHook::OnPointerEventButton Unhandled pointer state (!up/down) '%d': %d! surfaceID=%d app=%50s",
                            button, state, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
         break;
   }

   if (_uiObserver)
   {
      _uiObserver->onInputEvent();
   }
   return lMsg;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnPointerEventAxis(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_pointer* pointer,
      uint32_t time,
      uint32_t axis,
      wl_fixed_t value)
{
   FEATSTD_UNUSED2(pointer, time);
   Courier::Message* lMsg = Courier::InputHandling::InputHandler::cDoDefaultHandling;
   int lVal = (int) wl_fixed_to_double(value);
   uint32_t unifiedEncoderCode = hmibase::app::base::keymapbase::ENCODER_CODE_INVALID;

   _pointerTouchedSurfaceID = wlseat->GetPointerSurfaceID();

   if (0 != _mKeyMapping)
   {
      unifiedEncoderCode = _mKeyMapping->GetHmiEncoderCode(axis, 0);
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::OnPointerEventAxis failed, WindowInputEventHook not yet initialized, returning: axis: '%d' value: %d! surfaceID=%d app=%50s",
                         axis, lVal, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
   }

   if (unifiedEncoderCode == hmibase::app::base::keymapbase::ENCODER_CODE_INVALID)
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::OnPointerEventAxis Unhandled encoder input event axis: '%d' value: %d! surfaceID=%d app=%50s",
                         axis, lVal, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
   }
   else
   {
      lMsg = COURIER_MESSAGE_NEW(EncoderStatusChangedUpdMsg)(unifiedEncoderCode, lVal, _pointerTouchedSurfaceID, 0);
      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnPointerEventAxis encoder code '%d' received, steps: '%d'. surfaceID=%d app=%50s",
                          unifiedEncoderCode, lVal, _pointerTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
   }

   if (_uiObserver)
   {
      _uiObserver->onInputEvent();
   }
   return lMsg;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnKeyboardEventKey(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      uint32_t time,
      uint32_t key,
      uint32_t state)
{
   hmibase::util::ScopedMutex scs(_criticalSec);

   FEATSTD_UNUSED3(keyboard, serial, time);
   Courier::Message* lMsg = Courier::InputHandling::InputHandler::cDoDefaultHandling;
   Courier::Message* lMsgAbort = Courier::InputHandling::InputHandler::cDoDefaultHandling;
   uint32_t newUnifiedKeyCode = hmibase::app::base::keymapbase::KEY_CODE_INVALID;

   _keyboardSurfaceID = wlseat->GetKeyboardSurfaceID();
   uint32_t displayid = ScreenBrokerClient::GetInstance().GetDisplayID(_keyboardSurfaceID);

   switch (state)
   {
      case WL_KEY_RELEASE:
      {
         newUnifiedKeyCode = GetUnifiedKeyCode(key);
         if (_unifiedKeyCode == newUnifiedKeyCode)
         {
            //the stored key code from last PRESS event is the same, so process the key
            if (newUnifiedKeyCode != hmibase::app::base::keymapbase::KEY_CODE_INVALID)
            {
               if (_longPressKeyTimer.multiTimeoutExpired())
               {
                  // since min. 1 long press timer expired, send long release message
                  lMsg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(newUnifiedKeyCode, HARDKEYSTATE_LONGUP, displayid);
                  ETG_TRACE_USR1_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'LONG_RELEASE' of key '%d' received. surfaceID=%d app=%50s",
                                      newUnifiedKeyCode, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
               }
               else
               {
                  //send normal release message
                  lMsg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(newUnifiedKeyCode, HARDKEYSTATE_UP, displayid);
                  ETG_TRACE_USR1_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'RELEASE' of key '%d' received. surfaceID=%d app=%50s",
                                      newUnifiedKeyCode, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'RELEASE' of key '%d' received but no mapping found, so key not handled! surfaceID=%d app=%50s",
                                  key, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
            }

            //stop key timers in any case, if it is not running, nothing will happen
            StopLongPressKeyTimers();
            StopRepeatKeyTimer();
            _unifiedKeyCode = hmibase::app::base::keymapbase::KEY_CODE_INVALID;
         }
         else
         {
            //the stored key code from last PRESS event is different or no PRESS at all was received, so ignore the key
            ETG_TRACE_ERR_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'RELEASE' of key '%d' received but no corresponding press found, so key not handled! surfaceID=%d app=%50s",
                               newUnifiedKeyCode, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
         }
         break;
      }

      case WL_KEY_PRESS:
      {
         if (_unifiedKeyCode != hmibase::app::base::keymapbase::KEY_CODE_INVALID)
         {
            //a key is already processed at the moment, stop this and store the new key
            //stop key timers in any case, if it is not running, nothing will happen
            StopLongPressKeyTimers();
            StopRepeatKeyTimer();
            //send abort message
            lMsgAbort = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_ABORT, displayid);
            if (lMsgAbort)
            {
               Courier::InputHandling::InputHandler::Post(lMsgAbort);
            }
            ETG_TRACE_USR1_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'ABORT' of key '%d' is send since a PRESS of a new key was received. surfaceID=%d app=%50s",
                                _unifiedKeyCode, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
            _unifiedKeyCode = hmibase::app::base::keymapbase::KEY_CODE_INVALID;
         }

         _unifiedKeyCode = GetUnifiedKeyCode(key);

         if (_unifiedKeyCode != hmibase::app::base::keymapbase::KEY_CODE_INVALID)
         {
            //Check if this key is an abort key for animations and send corresponding request to SBClient in this case
            CheckAnimAbortKey(_unifiedKeyCode);
            StartLongPressKeyTimers();
            StartRepeatKeyTimer();

            lMsg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_DOWN, displayid);
            ETG_TRACE_USR1_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'PRESS' of key '%d' received. surfaceID=%d app=%50s",
                                _unifiedKeyCode, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
         }
         else
         {
            ETG_TRACE_ERR_THR(("WindowInputEventHook::OnKeyboardEventKey Key 'PRESS' of key '%d' received but no mapping found, so key not handled! surfaceID=%d app=%50s",
                               key, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
         }
         break;
      }

      default:
         ETG_TRACE_ERR_THR(("WindowInputEventHook::OnKeyboardEventKey Unhandled state of key '%d': %d! surfaceID=%d app=%50s",
                            key, state, _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));
         break;
   }

   if (_uiObserver)
   {
      _uiObserver->onInputEvent();
   }
   return lMsg;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnKeyboardEventEnter(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      struct wl_surface* surface,
      struct wl_array* keys)
{
   FEATSTD_UNUSED3(keyboard, serial, keys);

   hmibase::util::ScopedMutex scs(_criticalSec);

   _keyboardSurfaceID = GetTouchedSurface(wlseat->GetParentWlContext(), surface);
   wlseat->SetKeyboardSurfaceID(_keyboardSurfaceID);

   ETG_TRACE_USR1_THR(("WindowInputEventHook::OnKeyboardEventEnter is called keyboard focus is with surfaceID=%d and app=%50s",
                       _keyboardSurfaceID, hmibase::trace::getAppName().c_str()));

   if (_uiObserver)
   {
      _uiObserver->onInputEvent();
   }
   return Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnKeyboardEventLeave(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      struct wl_surface* surface)
{
   hmibase::util::ScopedMutex scs(_criticalSec);

   FEATSTD_UNUSED3(keyboard, serial, surface);
   uint32_t keySurfaceIdLocal = wlseat->GetKeyboardSurfaceID();

   //stop key timers, send Abort HKstatuschanged message
   ResetInputKeyEvent(keySurfaceIdLocal);

   ETG_TRACE_USR1_THR(("WindowInputEventHook::OnKeyboardEventLeave is called keyboard focus is with surfaceID=%d and app=%50s",
                       keySurfaceIdLocal, hmibase::trace::getAppName().c_str()));

   //reset seat's surfaceID when leaving surface
   wlseat->SetKeyboardSurfaceID(0);
   return Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnTouchEventDown(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_touch* touch,
      uint32_t serial,
      uint32_t time,
      struct wl_surface* surface,
      int32_t id,
      wl_fixed_t x,
      wl_fixed_t y)
{
   FEATSTD_UNUSED3(touch, serial, time);
   int lX = (int) wl_fixed_to_double(x);
   int lY = (int) wl_fixed_to_double(y);
   Courier::Message* lMsg = Courier::InputHandling::InputHandler::cDoDefaultHandling;
   uint32_t timestamp = FeatStd::Internal::PerfCounter::Now();
   bool storeCoordinates = false;

   if (_multiFingerInputEnabled || id <= 0)
   {
      _touchTouchedSurfaceID = GetTouchedSurface(wlseat->GetParentWlContext(), surface);
      wlseat->SetTouchSurfaceID(_touchTouchedSurfaceID);

      // Parameter: State, xPos, yPos, PointerID (ID of finger in case of multi touch),
      // SourceId (ID of connected display, if more than one HW display is connected)
      lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Down, lX, lY, timestamp, static_cast<Courier::PointerId>(id), _touchTouchedSurfaceID);

      if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsg)))
      {
         lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
      }

      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnTouchEventDown x=%d y=%d DOWN, pointerID=%d surfaceID=%d app=%50s", lX, lY, id,
                          _touchTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

      storeCoordinates = true;
   }
   else
   {
      lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnTouchEventDown x=%d y=%d DOWN, pointerID=%d surfaceID=%d app=%50s is ignored, since _multiFingerInputEnabled is not set.", lX, lY, id,
                          _touchTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
   }

   if (_uiObserver)
   {
      _uiObserver->onTouchEventDown(lX, lY, _touchTouchedSurfaceID, id, timestamp);
   }

   if (storeCoordinates)
   {
      _lLastTouchXCoord = lX;
      _lLastTouchYCoord = lY;
   }
   return lMsg;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnTouchEventUp(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_touch* touch,
      uint32_t serial,
      uint32_t time,
      int32_t id)
{
   FEATSTD_UNUSED3(touch, serial, time);
   Courier::Message* lMsg = Courier::InputHandling::InputHandler::cDoDefaultHandling;
   uint32_t timestamp = FeatStd::Internal::PerfCounter::Now();
   bool resetCoordinates = false;
   _touchTouchedSurfaceID = wlseat->GetTouchSurfaceID();
   if (_multiFingerInputEnabled || id <= 0)
   {
      // Parameter: State, xPos, yPos, PointerID (ID of finger in case of multi touch),
      // SourceId (ID of connected display, if more than one HW display is connected)*
      lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Up, _lLastTouchXCoord, _lLastTouchYCoord, timestamp, static_cast<Courier::PointerId>(id), _touchTouchedSurfaceID);

      if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsg)))
      {
         lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
      }

      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnTouchEventUp x=%d y=%d UP, pointerID=%d surfaceID=%d app=%50s", _lLastTouchXCoord, _lLastTouchYCoord, id,
                          _touchTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

      // reset
      resetCoordinates = true;
   }
   else
   {
      lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnTouchEventUp x=%d y=%d UP, pointerID=%d surfaceID=%d app=%50s is ignored, since _multiFingerInputEnabled is not set.",
                          _lLastTouchXCoord, _lLastTouchYCoord, id, _touchTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
   }

   if (_uiObserver)
   {
      _uiObserver->onTouchEventUp(_lLastTouchXCoord, _lLastTouchYCoord, _touchTouchedSurfaceID, id, timestamp);
   }

   if (resetCoordinates)
   {
      // reset
      _lLastTouchXCoord = -1;
      _lLastTouchYCoord = -1;
   }
   return lMsg;
}


// ------------------------------------------------------------------------------------------------
Courier::Message* WindowInputEventHook::OnTouchEventMotion(Courier::InputHandling::Wayland::WLSeat* wlseat,
      struct wl_touch* touch,
      uint32_t time,
      int32_t id,
      wl_fixed_t x,
      wl_fixed_t y)
{
   FEATSTD_UNUSED2(touch, time);
   int lX = (int) wl_fixed_to_double(x);
   int lY = (int) wl_fixed_to_double(y);
   Courier::Message* lMsg = Courier::InputHandling::InputHandler::cDoDefaultHandling;
   uint32_t timestamp = FeatStd::Internal::PerfCounter::Now();
   bool storeCoordinates = false;
   _touchTouchedSurfaceID = wlseat->GetTouchSurfaceID();
   if (_multiFingerInputEnabled || id <= 0)

   {
      // Parameter: State, xPos, yPos, PointerID (ID of finger in case of multi touch),
      // SourceId (ID of connected display, if more than one HW display is connected)
      lMsg = COURIER_MESSAGE_NEW(Courier::TouchMsg)(Courier::TouchMsgState::Move, lX, lY, timestamp, static_cast<Courier::PointerId>(id), _touchTouchedSurfaceID);

      if ((_touchEventPreprocessor != 0) && (_touchEventPreprocessor->Receive(lMsg)))
      {
         lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
      }

      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnTouchEventMotion x=%d y=%d, pointerID=%d surfaceID=%d app=%50s", lX, lY, id,
                          _touchTouchedSurfaceID, hmibase::trace::getAppName().c_str()));

      storeCoordinates = true;
   }
   else
   {
      lMsg = Courier::InputHandling::InputHandler::cIgnoreDefaultHandling;
      ETG_TRACE_USR4_THR(("WindowInputEventHook::OnTouchEventMotion x=%d y=%d, pointerID=%d surfaceID=%d app=%50s is ignored, since _multiFingerInputEnabled is not set.", lX, lY, id,
                          _touchTouchedSurfaceID, hmibase::trace::getAppName().c_str()));
   }

   if (_uiObserver)
   {
      _uiObserver->onTouchEventMove(lX, lY, _touchTouchedSurfaceID, id, timestamp);
   }

   if (storeCoordinates)
   {
      // reset
      _lLastTouchXCoord = lX;
      _lLastTouchYCoord = lY;
   }

   return lMsg;
}


// ------------------------------------------------------------------------------------------------
uint32_t WindowInputEventHook::GetTouchedSurface(void* data, struct wl_surface* surface) const
{
   static t_ilm_surface mTouchedSurface = 0;

   if (0 != data)
   {
      Courier::InputHandling::Wayland::WaylandContext* lContext = static_cast<Courier::InputHandling::Wayland::WaylandContext*>(data);

      if (0 != surface)
      {
         Courier::InputHandling::DisplayInputContext* inputContext =
            reinterpret_cast<Courier::InputHandling::DisplayInputContext*>(lContext->GetData());

         if (0 != inputContext)
         {
            Candera::GeniviTargetDisplay* lDisplay =
               dynamic_cast<Candera::GeniviTargetDisplay*>(inputContext->GetDisplay());

            if (0 != lDisplay)
            {
               // Search for corresponding ILM surface ID
               FeatStd::UInt i = 0;
               Candera::GeniviTargetDisplay::IlmWaylandMapItem* item;
               do
               {
                  item = lDisplay->GetIlmWaylandMapItem(i++);
               }
               while ((0 != item) && (surface != item->m_waylandSurface));

               // Corresponding ILM surface found
               if (0 != item)
               {
                  mTouchedSurface = item->m_ilmSurfaceId;
                  ETG_TRACE_USR1_THR(("WindowInputEventHook::GetTouchedSurface ILM surface for native wayland surface %p found. surfaceID=%d app=%50s",
                                      surface, mTouchedSurface, hmibase::trace::getAppName().c_str()));
               }
               else
               {
                  ETG_TRACE_ERR_THR(("WindowInputEventHook::GetTouchedSurface NO ILM surface for native wayland surface %p found! app=%50s",
                                     surface, hmibase::trace::getAppName().c_str()));
               }
            }
            else
            {
               ETG_TRACE_ERR_THR(("WindowInputEventHook::GetTouchedSurface NO GeniviTargetDisplay for native wayland surface %p found! app=%50s",
                                  surface, hmibase::trace::getAppName().c_str()));
            }// lDisplay != 0
         }
         else
         {
            ETG_TRACE_ERR_THR(("WindowInputEventHook::GetTouchedSurface NO DisplayInputContext for native wayland surface %p found! app=%50s",
                               surface, hmibase::trace::getAppName().c_str()));
         }// inputContext != 0
      }
      else
      {
         ETG_TRACE_ERR_THR(("WindowInputEventHook::GetTouchedSurface native wayland surface is '0'! app=%50s",
                            hmibase::trace::getAppName().c_str()));
      }
      // surface != 0
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::GetTouchedSurface WaylandContext is '0'! app=%50s",
                         hmibase::trace::getAppName().c_str()));
   }
   // data != 0

   return mTouchedSurface;
}


// ------------------------------------------------------------------------------------------------
uint32_t WindowInputEventHook::GetUnifiedKeyCode(uint32_t key) const
{
   uint32_t unifiedKeyCode = hmibase::app::base::keymapbase::KEY_CODE_INVALID;

   if (0 != _mKeyMapping)
   {
      unifiedKeyCode = _mKeyMapping->GetHmiKeyCode(key, 0);
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::GetUnifiedKeyCode failed, WindowInputEventHook not yet initialized, returning: key code: '%d'! app=%50s",
                         unifiedKeyCode, hmibase::trace::getAppName().c_str()));
   }

   return unifiedKeyCode;
}


// ------------------------------------------------------------------------------------------------
void WindowInputEventHook::StartLongPressKeyTimers()
{
   if (0 != _mKeyMapping)
   {
      KeyMappingBase::HkTimers timersconfigured;
      memset(&timersconfigured, 0, sizeof(timersconfigured));
      bool bTimeoutsSet = false;
      bool bReturn = true;

      uint32_t displayid = ScreenBrokerClient::GetInstance().GetDisplayID(_keyboardSurfaceID);

      //to make sure that all timer information is cleared (timer values set to '0', set courier messages to NULL, set repeat time to '0'
      _longPressKeyTimer.resetTimeouts();

      timersconfigured = _mKeyMapping->GetTimerValuesForKey(_unifiedKeyCode, 0);

      if (timersconfigured.long1 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(0, timersconfigured.long1, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG1, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long2 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(1, timersconfigured.long2, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG2, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long3 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(2, timersconfigured.long3, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG3, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long4 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(3, timersconfigured.long4, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG4, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long5 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(4, timersconfigured.long5, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG5, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long6 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(5, timersconfigured.long6, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG6, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long7 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(6, timersconfigured.long7, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG7, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long8 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(7, timersconfigured.long8, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG8, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long9 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(8, timersconfigured.long9, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG9, displayid));
         bTimeoutsSet = true;
      }
      if (timersconfigured.long10 > 0)
      {
         bReturn = _longPressKeyTimer.setTimeout(9, timersconfigured.long10, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_LONG10, displayid));
         bTimeoutsSet = true;
      }

      if (bTimeoutsSet)
      {
         //start timer only if timeouts are defined for this key
         _longPressKeyTimer.start();
         ETG_TRACE_USR1_THR(("WindowInputEventHook::StartLongPressKeyTimers Long press timer started for key: %d app=%50s",
                             _unifiedKeyCode, hmibase::trace::getAppName().c_str()));
      }

      if (bReturn)
      {
         ETG_TRACE_ERR_THR(("WindowInputEventHook::StartLongPressKeyTimers failed, negative return code from _longPressKeyTimer.setTimeout() for min. one timeout. key code: '%d'! app=%50s",
                            _unifiedKeyCode, hmibase::trace::getAppName().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::StartLongPressKeyTimers failed, WindowInputEventHook not yet initialized, returning: key code: '%d'! app=%50s",
                         _unifiedKeyCode, hmibase::trace::getAppName().c_str()));
   }
}


// ------------------------------------------------------------------------------------------------
void WindowInputEventHook::StopLongPressKeyTimers()
{
   _longPressKeyTimer.stop();

   ETG_TRACE_USR1_THR(("WindowInputEventHook::StopLongPressKeyTimers Long press timer stopped for key: %d app=%50s",
                       _unifiedKeyCode, hmibase::trace::getAppName().c_str()));
}


// ------------------------------------------------------------------------------------------------
void WindowInputEventHook::StartRepeatKeyTimer()
{
   if (0 != _mKeyMapping)
   {
      //to make sure that all timer information is cleared (timer values set to '0', set courier messages to NULL, set repeat time to '0')
      _repeatKeyTimer.resetTimeouts();

      unsigned int repeatTimeout = _mKeyMapping->GetRepeatTimeoutForKey(_unifiedKeyCode, 0);

      if (repeatTimeout > 0)
      {
         uint32_t displayid = ScreenBrokerClient::GetInstance().GetDisplayID(_keyboardSurfaceID);
         //a repeat time is set for this key
         if (!_repeatKeyTimer.setTimeoutWithRepeat(repeatTimeout, repeatTimeout, COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_REPEAT, displayid), WindowInputEventHook::RepeatTimerCallBack))
         {
            ETG_TRACE_ERR_THR(("WindowInputEventHook::StartRepeatKeyTimer failed, negative return code from _repeatKeyTimer.setTimeoutWithRepeat(). key code: '%d' repeatTimeout: %d, repeatTime: %d! app=%50s",
                               _unifiedKeyCode, repeatTimeout, repeatTimeout, hmibase::trace::getAppName().c_str()));
         }

         _repeatKeyTimer.start();

         ETG_TRACE_USR1_THR(("WindowInputEventHook::StartRepeatKeyTimer repeat key timer started for key: %d, repeatTimeout: %d, repeatTime: %d, app=%50s",
                             _unifiedKeyCode, repeatTimeout, repeatTimeout,  hmibase::trace::getAppName().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::StartRepeatKeyTimer failed, WindowInputEventHook not yet initialized, returning: key code: '%d'! app=%50s",
                         _unifiedKeyCode, hmibase::trace::getAppName().c_str()));
   }
}


// ------------------------------------------------------------------------------------------------
void WindowInputEventHook::StopRepeatKeyTimer()
{
   _repeatKeyTimer.stop();

   ETG_TRACE_USR1_THR(("WindowInputEventHook::StopRepeatKeyTimer Repeat key timer stopped for key: %d app=%50s",
                       _unifiedKeyCode, hmibase::trace::getAppName().c_str()));
}


// ------------------------------------------------------------------------------------------------
void WindowInputEventHook::CheckAnimAbortKey(uint32_t key) const
{
   if (0 != _mKeyMapping)
   {
      //check if this key is an key that aborts running SB Animations
      if (_mKeyMapping->IsAbortKey(key, 0))
      {
         //send abort to SB
         ScreenBrokerClient::GetInstance().AbortAnimations();
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::CheckAnimAbortKey failed, WindowInputEventHook not yet initialized, returning: key code: '%d'! app=%50s",
                         key, hmibase::trace::getAppName().c_str()));
   }
}


void WindowInputEventHook::ResetInputKeyEvent(uint32_t leavingSurfaceID)
{
   //lock scoped section, since _unifiedKeyCode/_keyboardSurfaceID variables could be updated from multiple threads
   //example: screenbrokerclient thread and Windowinput thread
   hmibase::util::ScopedCriticalSection scs(_criticalSec);

   if ((leavingSurfaceID == _keyboardSurfaceID) &&
         (_unifiedKeyCode != hmibase::app::base::keymapbase::KEY_CODE_INVALID)
      )
   {
      //stop key timers in any case, if it is not running, nothing will happen
      StopLongPressKeyTimers();
      StopRepeatKeyTimer();

      uint32_t displayid = ScreenBrokerClient::GetInstance().GetDisplayID(_keyboardSurfaceID);

      Courier::Message* lMsgAbort = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(_unifiedKeyCode, HARDKEYSTATE_ABORT, displayid);
      //a key is already processed at the moment, stop this and send abort message
      if (lMsgAbort)
      {
         Courier::InputHandling::InputHandler::Post(lMsgAbort);
      }

      ETG_TRACE_USR1_THR(("WindowInputEventHook::ResetInputKeyEvent Key 'ABORT' of key '%d' is send since this app/surface is leavesfocus/UnMapped : surfaceID=%d app=%50s",
                          _unifiedKeyCode, leavingSurfaceID, hmibase::trace::getAppName().c_str()));

      _unifiedKeyCode  = hmibase::app::base::keymapbase::KEY_CODE_INVALID;
      _keyboardSurfaceID = 0;
   }
}


void WindowInputEventHook::RepeatTimerCallBack(Courier::Message* msg)
{
   // check for NUll pointer, callback is also called on Not a valid message
   if (msg != NULL)
   {
      HKStatusChangedUpdMsg* hkMsg = dynamic_cast<HKStatusChangedUpdMsg*>(msg);
      if (hkMsg != NULL)
      {
         //uint32_t displayid = ScreenBrokerClient::GetInstance().GetDisplayID(repeatKbSurface);
         ETG_TRACE_USR1_THR(("WindowInputEventHook::RepeatTimerCallBack for key: %d, Display=%d, app=%50s",
                             hkMsg->GetHKCode(), hkMsg->GetDisplayID(), hmibase::trace::getAppName().c_str()));

         Courier::Message* lMsg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(hkMsg->GetHKCode(), HARDKEYSTATE_REPEAT, hkMsg->GetDisplayID());
         if (lMsg)
         {
            Courier::InputHandling::InputHandler::Post(lMsg);
         }
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("WindowInputEventHook::RepeatTimerCallBack - Not Valid Message! app=%50s", hmibase::trace::getAppName().c_str()));
   }
}
