/* ***************************************************************************************
* FILE:          WaylandInputHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandInputHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "lint_deactivation.h"
#include "WaylandInputHandler.h"

#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Platform/MessageFactory.h>
#include <Courier/Diagnostics/Log.h>
#include <FeatStd/Platform/Thread.h>
#include <View/CGI/InputHandling/DisplayConfiguration.h>
#include <View/CGI/InputHandling/DisplayInputContext.h>

#include "WaylandInc.h"
#include "WaylandEventHook.h"

//#include <stdio.h>
#include <poll.h>

#include "hmi_trace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#include "trcGenProj/Header/WaylandInputHandler.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace Courier {
namespace InputHandling {
namespace Wayland {

// ------------------------------------------------------------------------
WaylandInputHandler::WaylandInputHandler()
{
}


// ------------------------------------------------------------------------
bool WaylandInputHandler::InitContext()
{
   bool lRc = (0 != GetNativeHandle());
   if (lRc)
   {
      ETG_TRACE_USR4_THR(("Initializing wayland context"));
      mContext.Init(reinterpret_cast<wl_display*>(GetNativeHandle()),
                    reinterpret_cast<void*>(GetInputContext()),
                    reinterpret_cast<WaylandEventHook*>(GetEventHook()));
   }
   return (lRc);
}


// ------------------------------------------------------------------------
bool WaylandInputHandler::Listen()
{
   bool lRepeat = true;
   // Check if Wayland context is initialized
   if (!mContext.IsInitialized())
   {
      // Try to initialize Wayland context
      lRepeat = !InitContext();
      if (lRepeat)
      {
         // Wait a short while before trying again, if handle is available in the next thread cycle
         FeatStd::Internal::Thread::Sleep(20);
         return true;
      }
   }

   if (!lRepeat)
   {
      // Check if input handler is initialized
      FEATSTD_DEBUG_ASSERT(0 != mContext.GetDisplay());
   }

   // Creating poll wayland display file descriptor for capturing incoming events
   struct pollfd lPollFd[1];
   lPollFd[0].fd = wl_display_get_fd(mContext.GetDisplay());
   lPollFd[0].events = POLLIN;

   if (!LockDisplayForListen())
   {
      return false;
   }

#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 2))
   // Empty queue before getting exclusive access
   while (-1 == wl_display_prepare_read_queue(mContext.GetDisplay(),
          mContext.GetEventQueue()))
   {
      (void) wl_display_dispatch_queue_pending(mContext.GetDisplay(),
            mContext.GetEventQueue());
   }
#else
   // Empty queue without getting exclusive access (old version)
   (void) wl_display_dispatch_queue_pending(mContext.GetDisplay(),
         mContext.GetEventQueue());
#endif

   // Send buffered requests to server
   (void) wl_display_flush(mContext.GetDisplay());
   UnlockDisplayForListen();
   // Listen on wayland display file descriptor using poll
   Int pollResult = poll(lPollFd, 1, (InputHandlerMode::Blocking == GetMode() ? -1 : 0));
   if (!LockDisplayForListen())
   {
      return false;
   }
   switch (pollResult)
   {
      case -1:    // error
#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 2))
         wl_display_cancel_read(mContext.GetDisplay());
#endif
         break;
      case 0:     // timed out
         (void) wl_display_dispatch_queue_pending(mContext.GetDisplay(),
               mContext.GetEventQueue());
         break;
      default:    // success
         if (lPollFd[0].revents & POLLIN)
         {
#if ((WAYLAND_VERSION_MAJOR >= 1) && (WAYLAND_VERSION_MINOR >= 2))
            (void) wl_display_read_events(mContext.GetDisplay());
            //wl_display_dispatch_pending(mContext.GetDisplay());
            (void) wl_display_dispatch_queue_pending(mContext.GetDisplay(),
                  mContext.GetEventQueue());
#else
            (void) wl_display_dispatch_queue(mContext.GetDisplay(),
                                             mContext.GetEventQueue());
         }
         else
         {
            (void) wl_display_dispatch_queue_pending(mContext.GetDisplay(),
                  mContext.GetEventQueue());
#endif
         }
         break;
   }
   UnlockDisplayForListen();
   return true;
}


}
}


}
