/* ***************************************************************************************
* FILE:          WaylandEventHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandEventHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include <Courier/Base.h>
#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Platform/MessageFactory.h>
#include <View/CGI/InputHandling/DisplayInputContext.h>
#include <View/CGI/InputHandling/InputHandler.h>
#include <CanderaPlatform/Device/Genivi/Target/ILM/GeniviTargetDisplay.h>
#include <FeatStd/Platform/String.h>
#include <FeatStd/Platform/PerfCounter.h>
#include "lint_deactivation.h"
#include "WaylandEventHandler.h"
#include "WaylandEventHook.h"
#include "WaylandContext.h"
#include <ilm/ilm_types.h>


#include "hmi_trace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#include "trcGenProj/Header/WaylandEventHandler.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace FeatStd::Internal;

namespace Courier {
namespace InputHandling {
namespace Wayland {
// ------------------------------------------------------------------------
static UInt32 GetTouchedSurface(void* data, const wl_surface* surface)
{
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   static t_ilm_surface mTouchedSurface = 0;

   if (0 != surface)
   {
      Courier::InputHandling::DisplayInputContext* inputContext =
         reinterpret_cast<Courier::InputHandling::DisplayInputContext*>(lContext->GetData());

      Candera::GeniviTargetDisplay* lDisplay =
         static_cast<Candera::GeniviTargetDisplay*>(inputContext->GetDisplay());

      // Search for corresponding ILM surface ID
      UInt i = 0;
      Candera::GeniviTargetDisplay::IlmWaylandMapItem* item;
      do
      {
         item = lDisplay->GetIlmWaylandMapItem(i++);
      }
      while ((0 != item) && (surface != item->m_waylandSurface));
      if (item)
      {
         mTouchedSurface = item->m_ilmSurfaceId;
      }
   }

   return mTouchedSurface;
}


// ------------------------------------------------------------------------
const struct wl_registry_listener* WaylandEventHandler::GetRegistryEventHandler()
{
   static const struct wl_registry_listener cRegistryEventHandler =
   {
      WaylandEventHandler::RegistryEventGlobal,
      WaylandEventHandler::RegistryEventGlobalRemove
   };

   return &cRegistryEventHandler;
}


// ------------------------------------------------------------------------
const struct wl_seat_listener* WaylandEventHandler::GetSeatEventHandler()
{
   static const struct wl_seat_listener cSeatEventHandler =
   {
      WaylandEventHandler::SeatEventCapabilities,
      WaylandEventHandler::SeatHandleName
   };

   return &cSeatEventHandler;
}


// ------------------------------------------------------------------------
const struct wl_pointer_listener* WaylandEventHandler::GetPointerEventHandler()
{
   static const struct wl_pointer_listener cPointerEventHandler =
   {
      WaylandEventHandler::PointerEventEnter,
      WaylandEventHandler::PointerEventLeave,
      WaylandEventHandler::PointerEventMotion,
      WaylandEventHandler::PointerEventButton,
      WaylandEventHandler::PointerEventAxis
#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 10)
      , WaylandEventHandler::PointerHandleFrame
      , WaylandEventHandler::PointerHandleAxis_source
      , WaylandEventHandler::PointerHandleAxis_stop
      , WaylandEventHandler::PointerHandleAxis_discrete
#endif
   };

   return &cPointerEventHandler;
}


// ------------------------------------------------------------------------
const struct wl_keyboard_listener* WaylandEventHandler::GetKeyboardEventHandler()
{
   static const struct wl_keyboard_listener cKeyboardEventHandler =
   {
      WaylandEventHandler::KeyboardEventKeymap,
      WaylandEventHandler::KeyboardEventEnter,
      WaylandEventHandler::KeyboardEventLeave,
      WaylandEventHandler::KeyboardEventKey,
      WaylandEventHandler::KeyboardEventModifiers
#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 6)
      , WaylandEventHandler::KeyboardHandleRepeatInfo
#endif
   };

   return &cKeyboardEventHandler;
}


// ------------------------------------------------------------------------
const struct wl_touch_listener* WaylandEventHandler::GetTouchEventHandler()
{
   static const struct wl_touch_listener cTouchEventHandler =
   {
      WaylandEventHandler::TouchEventDown,
      WaylandEventHandler::TouchEventUp,
      WaylandEventHandler::TouchEventMotion,
      WaylandEventHandler::TouchEventFrame,
      WaylandEventHandler::TouchEventCancel
#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 13)
      , WaylandEventHandler::TouchEventShape
      , WaylandEventHandler::TouchEventOrientation
#endif
   };

   return &cTouchEventHandler;
}


// ------------------------------------------------------------------------
void WaylandEventHandler::RegistryEventGlobal(void* data,
      struct wl_registry* registry,
      uint32_t name,
      const char* interface,
      uint32_t version)
{
   WaylandContext* lContext = static_cast<WaylandContext*>(data);

   if (lContext == 0)
   {
      FEATSTD_DEBUG_ASSERT_ALWAYS();
      return;
   }

   if (0 == FeatStd::Internal::String::CompareStrings(interface, "wl_seat"))
   {
      struct wl_seat* wlSeat = NULL;

      // Print wayland interface protocol version
      ETG_TRACE_USR2_THR(("Wayland interface version: %d", version));

      struct wl_registry* wrappedRegistry = (struct wl_registry*)  wl_proxy_create_wrapper((void*)registry);

      if (NULL != wrappedRegistry)
      {
         ETG_TRACE_USR2_THR(("WaylandEventHandler: use own queue for wl_seat"));
         wl_proxy_set_queue((struct wl_proxy*)wrappedRegistry, lContext->GetEventQueue());
         wlSeat = (wl_seat*)wl_registry_bind(wrappedRegistry, name, &wl_seat_interface, 2);
         wl_proxy_wrapper_destroy(wrappedRegistry);
      }
      else
      {
         ETG_TRACE_USR2_THR(("WaylandEventHandler: use default queue for wl_seat"));
         wlSeat = (wl_seat*)wl_registry_bind(registry, name, &wl_seat_interface, 2);
      }

      WLSeat* pWlSeat = new WLSeat();
      if ((NULL != pWlSeat) && (NULL != wlSeat))
      {
         pWlSeat->SetWLSeat(wlSeat);
         pWlSeat->SetParentWlContext(lContext);
         pWlSeat->setSeatId(name);

         wl_seat_add_listener(wlSeat, GetSeatEventHandler(), (void*)pWlSeat);
         lContext->AddWlSeat(pWlSeat);
         ETG_TRACE_USR2_THR(("Register seat object '%d' for WaylandContext [%p]", name, lContext));
      }
      else
      {
         ETG_TRACE_ERR_THR(("WaylandEventHandler::RegistryEventGlobal - Unable to allocate a seat object"));
      }

      //commented: below roundtrip call as it block this thread leading to deadlock situation if server is busy in reading or messages in queue to be dispatched
      //caused problem in multi-seat environment
      //wl_display_roundtrip(lContext->GetDisplay());
      ETG_TRACE_USR4_THR(("Wayland roundtrip of seat event handler finished"));
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::RegistryEventGlobalRemove(void* data,
      struct wl_registry* registry,
      uint32_t name)
{
   // Retrieve current wayland context
   WaylandContext* lContext = static_cast<WaylandContext*>(data);
   FEATSTD_DEBUG_ASSERT(lContext != 0);

   ETG_TRACE_USR1_THR(("Remove request for registry object '%d'", name));

   COURIER_UNUSED2(registry, lContext);
}


void WaylandEventHandler::SeatHandleName(void* data, struct wl_seat* seat, const char* name)
{
   FEATSTD_UNUSED(seat);
   ETG_TRACE_USR2_THR(("SeatHandleGetName [%s]", name));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      pWlSeat->SetSeatName(name);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::SeatEventCapabilities(void* data,
      struct wl_seat* seat,
      uint32_t caps)
{
   FEATSTD_UNUSED(seat);

   // Retrieve current WLSeat
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Retrieve current wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();

      if (lContext == 0)
      {
         FEATSTD_DEBUG_ASSERT_ALWAYS();
         return;
      }

      // Check if seat was already initialized
      struct wl_seat* lSeat = pWlSeat->GetWLSeat();
      if (0 == lSeat)
      {
         return;
      }

      // Register / de-register pointer event handler
      struct wl_pointer* lPointer = pWlSeat->GetWLPointer();
      if ((caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_POINTER)) && (0 == lPointer))
      {
         lPointer = wl_seat_get_pointer(lSeat);
         wl_proxy_set_queue((struct wl_proxy*)lPointer, lContext->GetEventQueue());   //lint !e740
         wl_pointer_set_user_data(lPointer, data);
         wl_pointer_add_listener(lPointer, GetPointerEventHandler(), data);
         ETG_TRACE_USR4_THR(("WaylandEventHandler::SeatEventCapabilities pointer registered"));
      }
      else
      {
         if (!(caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_POINTER)) && (0 != lPointer))
         {
            wl_pointer_destroy(lPointer);
            lPointer = 0;
         }
         else
         {
            ETG_TRACE_USR1_THR(("WaylandEventHandler::SeatEventCapabilities pointer failure, caps %u", caps));
         }
      }
      pWlSeat->SetWLPointer(lPointer);

      // Register / de-register keyboard event handler
      struct wl_keyboard* lKeyboard = pWlSeat->GetWLKeyboard();
      if ((caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_KEYBOARD)) && (0 == lKeyboard))
      {
         lKeyboard = wl_seat_get_keyboard(lSeat);
         wl_proxy_set_queue((struct wl_proxy*)lKeyboard, lContext->GetEventQueue());   //lint !e740
         wl_keyboard_set_user_data(lKeyboard, data);
         wl_keyboard_add_listener(lKeyboard, GetKeyboardEventHandler(), data);
         ETG_TRACE_USR4_THR(("WaylandEventHandler::SeatEventCapabilities keyboard registered"));
      }
      else
      {
         if (!(caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_KEYBOARD)) && (0 != lKeyboard))
         {
            wl_keyboard_destroy(lKeyboard);
            lKeyboard = 0;
         }
         else
         {
            ETG_TRACE_USR1_THR(("WaylandEventHandler::SeatEventCapabilities keyboard failure, caps %u", caps));
         }
      }
      pWlSeat->SetWLKeyboard(lKeyboard);

      // Register / de-register touch event handler
      struct wl_touch* lTouch = pWlSeat->GetWLTouch();

      if ((caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_TOUCH)) && (0 == lTouch))
      {
         lTouch = wl_seat_get_touch(lSeat);
         wl_proxy_set_queue((struct wl_proxy*)lTouch, lContext->GetEventQueue());    //lint !e740
         wl_touch_set_user_data(lTouch, data);
         wl_touch_add_listener(lTouch, GetTouchEventHandler(), data);
         ETG_TRACE_USR4_THR(("WaylandEventHandler::SeatEventCapabilities touch registered"));
      }
      else
      {
         if (!(caps & static_cast<uint32_t>(WL_SEAT_CAPABILITY_TOUCH)) && (0 != lTouch))
         {
            wl_touch_destroy(lTouch);
            lTouch = 0;
         }
         else
         {
            ETG_TRACE_USR1_THR(("WaylandEventHandler::SeatEventCapabilities touch failure, caps %u", caps));
         }
      }
      pWlSeat->SetWLTouch(lTouch);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::PointerEventEnter(void* data,
      struct wl_pointer* pointer,
      uint32_t serial,
      struct wl_surface* surface,
      wl_fixed_t surface_x,
      wl_fixed_t surface_y)
{
   // Convert fixed point coordinates
   Int lX = static_cast<Courier::Int>(wl_fixed_to_double(surface_x));
   Int lY = static_cast<Courier::Int>(wl_fixed_to_double(surface_y));

   ETG_TRACE_USR2_THR(("PointerEventEnter: serial(%d), surface(%p), x(%d), y(%d)", serial, surface, lX, lY));

   if (NULL == surface)
   {
      ETG_TRACE_ERR_THR(("Received Pointer enter for destroyed surface"));
      return;
   }
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnPointerEventEnter(pWlSeat,
                                                  pointer,
                                                  serial,
                                                  surface,
                                                  surface_x,
                                                  surface_y);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::PointerEventLeave(void* data,
      struct wl_pointer* pointer,
      uint32_t serial,
      struct wl_surface* surface)
{
   ETG_TRACE_USR2_THR(("PointerEventLeave: serial(%d), surface(%p)", serial, surface));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnPointerEventLeave(pWlSeat,
                                                  pointer,
                                                  serial,
                                                  surface);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::PointerEventMotion(void* data,
      struct wl_pointer* pointer,
      uint32_t time,
      wl_fixed_t surface_x,
      wl_fixed_t surface_y)
{
   // Convert fixed point coordinates
   int lX = (int) wl_fixed_to_double(surface_x);
   int lY = (int) wl_fixed_to_double(surface_y);

   ETG_TRACE_USR2_THR(("PointerEventMotion: x(%d), y(%d)", lX, lY));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnPointerEventMotion(pWlSeat,
                                                   pointer,
                                                   time,
                                                   surface_x,
                                                   surface_y);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::PointerEventButton(void* data,
      struct wl_pointer* pointer,
      uint32_t serial,
      uint32_t time,
      uint32_t button,
      uint32_t state)
{
   ETG_TRACE_USR2_THR(("PointerEventButton: button(%d), state(%d)", button, state));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnPointerEventButton(pWlSeat,
                                                   pointer,
                                                   serial,
                                                   time,
                                                   button,
                                                   state);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::PointerEventAxis(void* data,
      struct wl_pointer* pointer,
      uint32_t time,
      uint32_t axis,
      wl_fixed_t value)
{
   // Convert fixed point coordinates
   int lValue = (int) wl_fixed_to_double(value);
   ETG_TRACE_USR2_THR(("PointerEventAxis: axis(%d), value(%d)", axis, lValue));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnPointerEventAxis(pWlSeat,
                                                 pointer,
                                                 time,
                                                 axis,
                                                 value);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 10)

void WaylandEventHandler::PointerHandleFrame(void* data, struct wl_pointer* pointer)
{
   ETG_TRACE_USR2_THR(("WaylandEventHandler PointerHandleFrame called"));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->PointerHandleFrame(pWlSeat,
                                                 pointer);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


void WaylandEventHandler::PointerHandleAxis_source(void* data, struct wl_pointer* pointer, uint32_t source)
{
   COURIER_UNUSED2(data, pointer);

   const char* axis_source = "";

   switch (source)
   {
      case WL_POINTER_AXIS_SOURCE_WHEEL:
         axis_source = "wheel";
         break;
      case WL_POINTER_AXIS_SOURCE_FINGER:
         axis_source = "finger";
         break;
      case WL_POINTER_AXIS_SOURCE_CONTINUOUS:
         axis_source = "continuous";
         break;
      default:
         axis_source = "<invalid source value>";
         break;
   }

   ETG_TRACE_USR2_THR(("WaylandEventHandler PointerHandleAxis_source: %s", axis_source));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->PointerHandleAxis_source(pWlSeat,
                   pointer,
                   source);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


void WaylandEventHandler::PointerHandleAxis_stop(void* data, struct wl_pointer* pointer,
      uint32_t time, uint32_t axis)
{
   ETG_TRACE_USR2_THR(("WaylandEventHandler PointerHandleAxis_stop: %d, axis: %s",
                       time,
                       axis == (uint32_t)WL_POINTER_AXIS_VERTICAL_SCROLL ? "vertical" : "horizontal"));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->PointerHandleAxis_stop(pWlSeat,
                   pointer,
                   time,
                   axis);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


void WaylandEventHandler::PointerHandleAxis_discrete(void* data, struct wl_pointer* pointer,
      uint32_t axis, int32_t discrete)
{
   ETG_TRACE_USR2_THR(("WaylandEventHandler PointerHandleAxis_discrete: %d value: %d", axis, discrete));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->PointerHandleAxis_discrete(pWlSeat,
                   pointer,
                   axis,
                   discrete);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


#endif


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventKeymap(void* data,
      struct wl_keyboard* keyboard,
      uint32_t format,
      int fd,
      uint32_t size)
{
   ETG_TRACE_USR2_THR(("KeyboardEventKeymap: format(%d), fd(%d), size(%d)", format, fd, size));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnKeyboardEventKeymap(pWlSeat,
                                                    keyboard,
                                                    format,
                                                    fd,
                                                    size);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventEnter(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      struct wl_surface* surface,
      struct wl_array* keys)
{
   ETG_TRACE_USR2_THR(("KeyboardEventEnter: serial(%d), surface(%p)", serial, surface));
   if (NULL == surface)
   {
      ETG_TRACE_ERR_THR(("Received keyboard enter for destroyed surface"));
      return;
   }

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      pWlSeat->SetKeyboardFocus(surface);
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnKeyboardEventEnter(pWlSeat,
                                                   keyboard,
                                                   serial,
                                                   surface,
                                                   keys);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventLeave(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      struct wl_surface* surface)
{
   ETG_TRACE_USR2_THR(("KeyboardEventLeave: serial(%d), surface(%p)", serial, surface));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnKeyboardEventLeave(pWlSeat,
                                                   keyboard,
                                                   serial,
                                                   surface);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventKey(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      uint32_t time,
      uint32_t key,
      uint32_t state)
{
   ETG_TRACE_USR2_THR(("KeyboardEventKey: serial(%d), time(%d), key(%d), state(%d)",
                       serial, time, key, state));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnKeyboardEventKey(pWlSeat,
                                                 keyboard,
                                                 serial,
                                                 time,
                                                 key,
                                                 state);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         ETG_TRACE_USR4_THR(("KeyboardEventKey: Default handling"));
         if (static_cast<uint32_t>(WL_KEYBOARD_KEY_STATE_PRESSED) == state)
         {
            lMsg = COURIER_MESSAGE_NEW(KeyDownMsg)(static_cast<KeyCode>(key));
         }
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::KeyboardEventModifiers(void* data,
      struct wl_keyboard* keyboard,
      uint32_t serial,
      uint32_t mods_depressed,
      uint32_t mods_latched,
      uint32_t mods_locked,
      uint32_t group)
{
   ETG_TRACE_USR2_THR(("KeyboardEventModifiers: serial(%d), mods_depressed(%d), "
                       "mods_latched(%d), mods_locked(%d), group(%d)",
                       serial, mods_depressed, mods_latched, mods_locked, group));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnKeyboardEventModifiers(pWlSeat,
                   keyboard,
                   serial,
                   mods_depressed,
                   mods_latched,
                   mods_locked,
                   group);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 6)

void WaylandEventHandler::KeyboardHandleRepeatInfo(void* data, struct wl_keyboard* keyboard,
      int32_t rate, int32_t delay)
{
   ETG_TRACE_USR2_THR(("WaylandEventHandler KeyboardHandleRepeatInfo: rate(%d), delay(%d)", rate, delay));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->KeyboardHandleRepeatInfo(pWlSeat,
                   keyboard,
                   rate,
                   delay);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


#endif

// ------------------------------------------------------------------------
void WaylandEventHandler::TouchEventDown(void* data,
      struct wl_touch* touch,
      uint32_t serial,
      uint32_t time,
      struct wl_surface* surface,
      int32_t id,
      wl_fixed_t x,
      wl_fixed_t y)
{
   // Convert fixed point coordinates
   int lX = (int) wl_fixed_to_double(x);
   int lY = (int) wl_fixed_to_double(y);

   ETG_TRACE_USR2_THR(("TouchEventDown: id(%d), surface(%p), x(%d), y(%d)", id, surface, lX, lY));

   if (NULL == surface)
   {
      ETG_TRACE_ERR_THR(("Received touch_down for destroyed surface"));
      return;
   }
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnTouchEventDown(pWlSeat,
                                               touch,
                                               serial,
                                               time,
                                               surface,
                                               id,
                                               x,
                                               y);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         ETG_TRACE_USR4_THR(("TouchEventDown: Default handling"));
         lMsg = COURIER_MESSAGE_NEW(TouchMsg)(TouchMsgState::Down, lX, lY, FeatStd::Internal::PerfCounter::Now(), static_cast<PointerId>(id), GetTouchedSurface(data, surface));
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::TouchEventUp(void* data,
                                       struct wl_touch* touch,
                                       uint32_t serial,
                                       uint32_t time,
                                       int32_t id)
{
   ETG_TRACE_USR2_THR(("TouchEventUp: id(%d)", id));
   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnTouchEventUp(pWlSeat,
                                             touch,
                                             serial,
                                             time,
                                             id);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         ETG_TRACE_USR4_THR(("TouchEventUp: Default handling"));
         lMsg = COURIER_MESSAGE_NEW(TouchMsg)(TouchMsgState::Up, -1, -1, FeatStd::Internal::PerfCounter::Now(), static_cast<PointerId>(id), GetTouchedSurface(data, 0));
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::TouchEventMotion(void* data,
      struct wl_touch* touch,
      uint32_t time,
      int32_t id,
      wl_fixed_t x,
      wl_fixed_t y)
{
   // Convert fixed point coordinates
   int lX = (int) wl_fixed_to_double(x);
   int lY = (int) wl_fixed_to_double(y);

   ETG_TRACE_USR2_THR(("TouchEventMotion: id(%d), x(%d), y(%d)", id, lX, lY));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnTouchEventMotion(pWlSeat,
                                                 touch,
                                                 time,
                                                 id,
                                                 x,
                                                 y);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         ETG_TRACE_USR4_THR(("TouchEventMotion: Default handling"));
         lMsg = COURIER_MESSAGE_NEW(TouchMsg)(TouchMsgState::Move, lX, lY, FeatStd::Internal::PerfCounter::Now(), static_cast<PointerId>(id), GetTouchedSurface(data, 0));
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::TouchEventFrame(void* data,
      struct wl_touch* touch)
{
   ETG_TRACE_USR2_THR(("TouchEventFrame"));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnTouchEventFrame(pWlSeat,
                                                touch);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


// ------------------------------------------------------------------------
void WaylandEventHandler::TouchEventCancel(void* data,
      struct wl_touch* touch)
{
   ETG_TRACE_USR2_THR(("TouchEventCancel"));

   WLSeat* pWlSeat = static_cast<WLSeat*>(data);
   if (pWlSeat)
   {
      // Provide wayland context
      WaylandContext* lContext = pWlSeat->GetParentWlContext();
      FEATSTD_DEBUG_ASSERT(lContext != 0);

      // Call hook method
      Message* lMsg = InputHandler::cDoDefaultHandling;
      if (0 == lContext)
      {
         ETG_TRACE_ERR_THR(("Wayland context is invalid, event hooks ignored!"));
      }
      else
      {
         // Call hook method
         WaylandEventHook* eventHook = lContext->GetEventHook();
         if (0 != eventHook)
         {
            lMsg = eventHook->OnTouchEventCancel(pWlSeat,
                                                 touch);
         }
      }

      // If msg is still set to default handling, then do it
      if (InputHandler::DoDefaultHandling(lMsg))
      {
         // PUT DEFAULT HANDLING OF EVENT HERE!!!
      }

      // If event yielded a valid message, post it (if not, it is ignored)
      InputHandler::Post(lMsg);
   }
}


#if (WAYLAND_VERSION_MAJOR >=1 && WAYLAND_VERSION_MINOR >= 13)
void WaylandEventHandler::TouchEventShape(void* data,
      struct wl_touch* wl_touch,
      int32_t id,
      wl_fixed_t major,
      wl_fixed_t minor)
{
   FEATSTD_UNUSED5(data, wl_touch, id, major, minor);
   //dummy function since this is never called from compositor
}


void WaylandEventHandler::TouchEventOrientation(void* data,
      struct wl_touch* wl_touch,
      int32_t id,
      wl_fixed_t orientation)
{
   FEATSTD_UNUSED4(data, wl_touch, id, orientation);
   //dummy function since this is never called from compositor
}


#endif
}


}
}
