/* ***************************************************************************************
* FILE:          WaylandContext.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandContext.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include <Courier/Diagnostics/Diagnostics.h>
#include "WaylandContext.h"
#include "WaylandEventHandler.h"

COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

namespace Courier {
namespace InputHandling {
namespace Wayland {

// ------------------------------------------------------------------------
WaylandContext::WaylandContext()
   : mDisplay(0)
   , mRegistry(0)
   , mEventQueue(0)
   , mData(0)
   , mEventHook(0)
   , mInitialized(false)
   , _wlSeatList()

{
}


// ------------------------------------------------------------------------
WaylandContext::~WaylandContext()
{
   Destroy();
}


// ------------------------------------------------------------------------
bool WaylandContext::Init(wl_display* display, void* data, WaylandEventHook* eventHook)
{
   if (!mInitialized)
   {
      COURIER_LOG_DEBUG("Initializing wayland context for input handling");
      mInitialized = true;

      if (0 != display)
      {
         mDisplay = display;
         mData = data;
         mEventHook = eventHook;

         COURIER_LOG_DEBUG("Create wayland input event handler for wl_display(%p)", mDisplay);
         mRegistry = wl_display_get_registry(mDisplay);

         if (0 != mRegistry)
         {
            wl_registry_add_listener(mRegistry, WaylandEventHandler::GetRegistryEventHandler(), this);

            mEventQueue = wl_display_create_queue(mDisplay);
            COURIER_LOG_DEBUG("Propagate all issued requests to server");
            wl_display_dispatch(mDisplay);
            wl_display_roundtrip(mDisplay);
            COURIER_LOG_DEBUG("Wayland roundtrip of global event handler finished");

            if (0 != mEventQueue)
            {
               COURIER_LOG_DEBUG("Created wayland input event queue(%p) for wl_display(%p)",
                                 mEventQueue,
                                 mDisplay);
            }
            else
            {
               COURIER_LOG_ERROR("Failed to create wayland input event queue for wl_display(%p)!",
                                 mDisplay);
               mInitialized = false;
            }
         }
         else
         {
            COURIER_LOG_ERROR("Failed to create wayland registry listener for wl_display(%p)!",
                              mDisplay);
            mInitialized = false;
         }
      }
      else
      {
         COURIER_LOG_ERROR("Invalid wayland display retrieved!");
         mInitialized = false;
      }

      if (mInitialized)
      {
         COURIER_LOG_INFO("Wayland context for input handling successful initialized");
      }
   }
   else
   {
      COURIER_LOG_DEBUG("Wayland context for input handling ALREADY initialized");
   }

   return mInitialized;
}


// ------------------------------------------------------------------------
void WaylandContext::Destroy()
{
   //delete WLSeat objects that was created on RegistryEventGlobal
   std::list<WLSeat*>::iterator it;
   for (it = _wlSeatList.begin(); it != _wlSeatList.end(); ++it)
   {
      delete(*it);
   }
   _wlSeatList.clear();

   if (0 != mRegistry)
   {
      wl_registry_destroy(mRegistry);
      mRegistry = 0;
   }
//   if (0 != mEventQueue)         // TODO: Check if wl_event_queue_destroy(...) must be called additional
//   {
//      wl_event_queue_destroy(mEventQueue);
   mEventQueue = 0;
//   }
//   if (0 != mDisplay)             // TODO: Check if wl_display_disconnect(...) must be called additional
//   {
//      wl_display_disconnect(mDisplay);
   mDisplay = 0;
//   }
   mData = 0;
   mEventHook = 0;
}


}
}


}
