/* ***************************************************************************************
* FILE:          SurfaceInputRegionManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SurfaceInputRegionManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef __HMIBASE_SURFACE_INPUT_REGION_MANAGER__
#define __HMIBASE_SURFACE_INPUT_REGION_MANAGER__

#include <BaseContract/generated/BaseTypesSurfaceInputRegion.h>

namespace Courier {
class View;
}


namespace hmibase {
namespace view {
class ViewScene2D;
}


namespace widget {
class Widget;
}


namespace input {

typedef Courier::View SurfaceInputRegionViewType;
typedef hmibase::widget::Widget SurfaceInputRegionWidgetType;

class SurfaceInputRegionContainer;
class SurfaceInputRegionViewConfig;

class SurfaceInputRegionWidgetConfig
{
   public:
      virtual ~SurfaceInputRegionWidgetConfig() {}
      virtual hmibase::enSurfaceInputRegionOperation::Enum getInputRegionOperation() const = 0;
      virtual Candera::Rectangle getInputRegionRectangle() const = 0;
};


/*****************************************************************************/
/* Provides support to configure touchable input regions for surfaces in Gen4 using Wayland.
* The views will register themselves and mark as active. The input region widgets will registers themselves when updated.
* Before rendering, the registered widgets from the active views will be used to determine the touchable input regions which will are then sent to Wayland.
*/
/*****************************************************************************/
class SurfaceInputRegionManager
{
   public:
      typedef SurfaceInputRegionViewType ViewType;
      typedef SurfaceInputRegionWidgetType WidgetType;

      /* destructor */
      ~SurfaceInputRegionManager();

      /* returns the singleton instance of the surface input region manager */
      static SurfaceInputRegionManager& getInstance();

      /* creates the input region configuration for the specified view, must be done when the view is created */
      SurfaceInputRegionViewConfig* registerView(ViewType* view);

      /* destroys the input region configuration for the specified view, must be done when the view is destroyed */
      void unregisterView(ViewType* view);

      /* marks the specified view as active, its input region widgets will be used for determining the input regions for Wayland*/
      void onViewActivated(ViewType* view, bool isEntireViewportTouchable);

      /* marks the specified view as inactive, it will not be used for determining the input regions */
      void onViewDeactivated(ViewType* view);

      /* removes all the touchable widgets from the specified view */
      void clearTouchableWidgets(ViewType* view);

      /* registers the specified widget to be used for determining the input regions */
      void addInputRegionWidget(WidgetType* widget, SurfaceInputRegionWidgetConfig* widgetConfig);

      /* sends the input regions of the registered touchable widgets in the registered views to wayland */
      void sendInputRegions();

   private:
      /* private constructor, use getInstance to obtain the singleton instance */
      SurfaceInputRegionManager();

      /* gets the input region configuration for the specified view */
      SurfaceInputRegionViewConfig* getViewConfig(ViewType* view);

      typedef std::map<const ViewType*, SurfaceInputRegionViewConfig*> ViewConfigMapType;
      ViewConfigMapType _allViewConfigs;
      ViewConfigMapType _activeViewConfigs;
};


}
}


#endif
