/* ***************************************************************************************
* FILE:          InputThread.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  InputThread.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(InputHandling_InputThread_h)
#define InputHandling_InputThread_h

#include <Courier/Base.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Platform/Thread.h>

// ------------------------------------------------------------------------
namespace Courier {
namespace InputHandling {
class DisplayInputContext;
class InputHandler;
class InputThreadHook;

/** The InputThread thread is a utility class for using InputHandler.
    It provides its own thread context and starts listening to the attached InputHandler exclusively.
    Additionally it provides a hooking mechanism to hook the thread state stages like startup, execution, and termination. */
class InputThread : public FeatStd::Internal::Thread
{
   public:
      ///
      InputThread();

      ///
      virtual ~InputThread()
      {
         mHook = 0;
         mData = 0;
         mInputContext = 0;
      }

      /** Sets a thread hook, and returns the old one (if there was set one before).
          @param hook New thread hook to set.
          @return Previous thread hook.*/
      InputThreadHook* SetHook(InputThreadHook* hook);

      /** Sets a thread user data, and returns the old one (if there was set one before).
          This user data is simply forwarded to the hook methods for any user specific purpose.
          @param hook New thread data to set.
          @return Previous thread data.*/
      void* SetData(void* data);

      /** @param inputContext External input context, which is related to this external input thread instance. */
      void SetInputContext(DisplayInputContext* inputContext);

      /** Performs a graceful termination (exits the thread loop) of the input handler thread.
          @note If the attached input handler performs a blocking read on a event queue, an immediate
                termination may fail means it may wait for the next input event to be able to resolve
                the thread queue. */
      void Terminate();

   protected:
      /** The main thread function. */
      virtual Int ThreadFn();

      /** Resets the internal state of input thread. This method is called when thread is terminated. */
      FEATSTD_LINT_NEXT_EXPRESSION(1511, "the base class reset method is not in the public interface")
      virtual void Reset();

   private:
      FEATSTD_LINT_NEXT_EXPRESSION(1516, "the base class member is not in the public interface")
      FeatStd::Internal::CriticalSection mCs;
      InputThreadHook* mHook;
      void* mData;
      DisplayInputContext* mInputContext;
      bool mRunning;
};


}
}


#endif
