/* ***************************************************************************************
* FILE:          InputThread.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  InputThread.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "InputThread.h"
#include <Courier/Diagnostics/Log.h>
#include <View/CGI/InputHandling/DisplayInputContext.h>
#include <View/CGI/InputHandling/InputHandler.h>
#include <View/CGI/InputHandling/InputThreadHook.h>

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/InputThread.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

// ------------------------------------------------------------------------
namespace Courier {
namespace InputHandling {

// ------------------------------------------------------------------------
InputThread::InputThread() :
   mHook(0),
   mData(0),
   mInputContext(0),
   mRunning(false)
{
}


// ------------------------------------------------------------------------
InputThreadHook* InputThread::SetHook(InputThreadHook* hook)
{
   FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
   InputThreadHook* lPrevHook = mHook;
   mHook = hook;
   return lPrevHook;
}


// ------------------------------------------------------------------------
void* InputThread::SetData(void* data)
{
   FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
   void* lPrevData = mData;
   mData = data;
   return lPrevData;
}


// ------------------------------------------------------------------------
void InputThread::SetInputContext(DisplayInputContext* inputContext)
{
   FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
   mInputContext = inputContext;
}


// ------------------------------------------------------------------------
void InputThread::Terminate()
{
   FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
   mRunning = false;
}


// ------------------------------------------------------------------------
Int InputThread::ThreadFn()
{
   {
      FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
      mRunning = (0 != mInputContext);
      if (0 != mInputContext)   // lint !
      {
         InputHandler* lInputHandler = mInputContext->GetInputHandler();
         mRunning = mRunning && (0 != lInputHandler);
         if (!mRunning)
         {
            ETG_TRACE_ERR_THR(("Invalid input handler linked to input thread '%s' - not starting input listening...", GetName()));
         }
      }
      else
      {
         ETG_TRACE_ERR_THR(("Invalid input context linked to input thread '%s' - not starting input listening...", GetName()));
      }

      // Log info about starting input thread
      if (mRunning && (0 != mInputContext))
      {
         ETG_TRACE_USR4_THR(("Start %5s blocking input handler in receiver thread '%s'.",
                          ((InputHandlerMode::Blocking == mInputContext->GetInputHandler()->GetMode()) ? "" : " non"),
                          GetName()));
      }

      // Startup hook method
      if (0 != mHook)
      {
         mRunning = mRunning && mHook->OnStartup(mData);
      }
   }

   // The external input event processing loop
   while (mRunning)
   {
      {
         // Execution hook method
         FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
         if (0 != mHook)
         {
            mRunning =  mHook->OnExecute(mData);
         }
      }

      // Listen to external input events
      if (0 != mInputContext)
      {
         InputHandler* lInputHandler = mInputContext->GetInputHandler();
         if (0 != lInputHandler)
         {
            bool running = lInputHandler->Listen();
            if (!running)
            {
               mRunning = false;
            }
         }
         else
         {
            mRunning = false;
            ETG_TRACE_ERR_THR(("Invalid input handler linked to input thread '%s', terminating input thread...", GetName()));
         }
      }
      else
      {
         mRunning = false;
         ETG_TRACE_ERR_THR(("Invalid input context linked to input thread '%s', terminating input thread...", GetName()));
      }
   }

   ETG_TRACE_USR4_THR(("Input handler in receiver thread '%s' has ended.", GetName()));

   {
      // Termination hook method
      FeatStd::Internal::CriticalSectionLocker lLock(&mCs);
      if (0 != mHook)
      {
         mHook->OnShutdown(mData);
      }
   }

   Reset();

   return 0;
}


// ------------------------------------------------------------------------
void InputThread::Reset()
{
   if (!mRunning)
   {
      mHook = 0;
      mData = 0;
      mInputContext = 0;
   }
   else
   {
      ETG_TRACE_ERR_THR(("Cannot reset input thread, while still running!"));
   }
}


}
}
