/* ***************************************************************************************
* FILE:          InputHandler.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  InputHandler.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(InputHandling_InputHandler_h)
#define InputHandling_InputHandler_h

#include <Courier/Base.h>

// Forward declarations
namespace Courier {
class Message;
}


namespace Courier {
namespace InputHandling {
class DisplayInputContext;

/** Working modes for external input handler. */
namespace InputHandlerMode {
enum Enum
{
   /// Just checks the event queue and continues processing, if no event is available (non-blocking mode).
   Polling,
   /// Waits, when reading the event queue, until event available.
   Blocking
};


}
/// Base class for external communication window input handler.
class InputHandler
{
   public:
      /// (Message *) return value for respective input event hook signalling to perform default event handling
      static Message* cDoDefaultHandling;

      /// (Message *) return value for respective input event hook signalling to ignore default event handling
      static Message* cIgnoreDefaultHandling;

      /// (Message *) return value for respective input event hook signalling to terminate input handler
      static Message* cTerminateInputHandler;

      /** @param shallBlock Defines if the event queue listening shall block, when reading the input event queue
          (or poll the input event queue otherwise). */
      InputHandler();

      ///
      virtual ~InputHandler()
      {
         mEventHook = 0;
         mInputContext = 0;
      }

      /** @param mode Describes the listen policy of input handler. */
      virtual void SetMode(InputHandlerMode::Enum mode);

      /** @param hook Input event hook, to overwrite input handler's default handling. */
      void SetEventHook(void* eventHook);

      /** @param inputContext External input context, which is related to this external input handler instance. */
      void SetInputContext(DisplayInputContext* inputContext);

      /** Listen to input event queue.
          @return If <em>true</em> then stop calling this Listen() method in the embedding input thread,
                  if <em>false</em> then continue calling the Listen() method.*/
      virtual bool Listen() = 0;

      /** Resets the internal state of the input handler. */
      virtual void Reset();

      /** @return Mode describing the listen policy of input handler. */
      InputHandlerMode::Enum GetMode() const
      {
         return mMode;
      }

      /** @return Event hook for custom event handling. */
      FEATSTD_LINT_NEXT_EXPRESSION(1763, "the object itself is still const")
      void* GetEventHook() const
      {
         return mEventHook;
      }

      /** @return External input context, which is related to this external input handler instance. */
      FEATSTD_LINT_NEXT_EXPRESSION(1763, "the object itself is still const")
      DisplayInputContext* GetInputContext() const
      {
         return mInputContext;
      }

      /** @return Native handle for event queue listening. */
      void* GetNativeHandle() const;

      /** Posts the message if it is not 0 and does some logging.
          @param msg Pointer to message instance (can be 0). */
      static void Post(Message* msg);

      /// Checks if input handler shall apply a default handling of incoming event
      static bool DoDefaultHandling(const Message* msg)
      {
         return (cDoDefaultHandling == msg);
      }

      /// Checks if input handler shall terminate
      static bool TerminateInputHandler(const Message* msg)
      {
         return (cTerminateInputHandler == msg);
      }

      /// Checks if message pointer is valid for posting
      static bool IsValidMessage(const Message* msg);

   private:
      InputHandlerMode::Enum mMode;
      void* mEventHook;
      DisplayInputContext* mInputContext;
};


/// @}
}


}

#endif
