/* ***************************************************************************************
* FILE:          InputHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  InputHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "InputHandler.h"
#include "DisplayInputContext.h"

#include <Courier/Diagnostics/Log.h>
#include <Courier/Messaging/Message.h>

namespace Courier {
namespace InputHandling {
COURIER_LOG_SET_REALM(Courier::Diagnostics::LogRealm::Messaging);

FEATSTD_LINT_NEXT_EXPRESSION(923, "this cast is required")
Message* InputHandler::cDoDefaultHandling = reinterpret_cast<Message*>(0);
FEATSTD_LINT_NEXT_EXPRESSION(923, "this cast is required")
Message* InputHandler::cIgnoreDefaultHandling = reinterpret_cast<Message*>(-1);
FEATSTD_LINT_NEXT_EXPRESSION(923, "this cast is required")
Message* InputHandler::cTerminateInputHandler = reinterpret_cast<Message*>(-2);

// ------------------------------------------------------------------------
InputHandler::InputHandler() : mMode(InputHandlerMode::Blocking), mEventHook(0), mInputContext(0)
{
}


// ------------------------------------------------------------------------
void InputHandler::SetMode(InputHandlerMode::Enum mode)
{
   mMode = mode;
}


// ------------------------------------------------------------------------
void InputHandler::SetEventHook(void* eventHook)
{
   mEventHook = eventHook;
}


// ------------------------------------------------------------------------
void InputHandler::SetInputContext(DisplayInputContext* inputContext)
{
   mInputContext = inputContext;
}


// ------------------------------------------------------------------------
void InputHandler::Reset()
{
   mMode = InputHandlerMode::Blocking;
   mEventHook = 0;
   mInputContext = 0;
}


// ------------------------------------------------------------------------
void InputHandler::Post(Message* msg)
{
   if (IsValidMessage(msg))
   {
      COURIER_LOG_DEBUG("Post incoming %s", msg->GetName());
      bool lRc = msg->Post();
      if (!lRc)
      {
         COURIER_LOG_ERROR("Post incoming %s failed!", msg->GetName());
      }
   }
}


// ------------------------------------------------------------------------
void* InputHandler::GetNativeHandle() const
{
   FEATSTD_DEBUG_ASSERT(0 != mInputContext);
   void* lNativeHandle = 0;
   if (0 != mInputContext)
   {
      lNativeHandle = mInputContext->GetNativeHandle();
   }
   return lNativeHandle;
}


// ------------------------------------------------------------------------
bool InputHandler::IsValidMessage(const Message* msg)
{
   return ((cDoDefaultHandling != msg) &&
           (cIgnoreDefaultHandling != msg) &&
           (cTerminateInputHandler != msg));
}


}
}
