/* ***************************************************************************************
* FILE:          CgiApplicationBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CgiApplicationBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#define ENABLE_CONTROLLER_THREAD 0
//#define ENABLE_EXTCOMM_THREAD    1
//#define ENABLE_MODEL_THREAD      1
#define ENABLE_RENDER_THREAD     0
#define ENABLE_VIEW_THREAD       0

#include "gui_std_if.h"
#include "CgiApplicationBase.h"
#include "AppEnvironment.h"
#include "View/CGI/CgiExtensions/TouchInput.h"
#include "Focus/FManagerConfig.h"
#include "Focus/Default/FDefaultManagerConfig.h"
#include "Trace/ViewTraverser.h"
#include "View/CGI/Widget/Widget.h"

#include "Courier/DataBinding/StdTypeConverter.h"

#if defined(CANDERA_LAYOUT_MONITOR_ENABLED)
#include "Candera/Engine2D/Layout/LayoutLogger.h"
#endif

#ifndef WIN32
#include "Gadget/SyncBlockConsumer.h"
#include "Gadget/SyncBlockProducerFactory.h"
#endif

#include "Trace/StartupInvestigation.h"

#include "AppBase/AppConfigRegistry.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_CGI
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CgiApplicationBase.cpp.trc.h"
#endif

CgiApplicationBase::CgiApplicationBase(const std::vector<std::string>& cAssetFileNames, hmibase::services::hmiappctrl::ProxyHandler& hmiAppCtrlProxyHandler):
   mViewHandler(AppViewHandler::getInstance()),
   mDemoTouchSession(),
   mAppEnvironment(0),
   mAssetFileNames(cAssetFileNames),
   _hmiAppCtrlProxyHandler(hmiAppCtrlProxyHandler),
   _focusManagerConfigurator(0)
{
}


CgiApplicationBase::~CgiApplicationBase()
{
   mAppEnvironment = 0;
   if (0 != _focusManagerConfigurator)
   {
      delete _focusManagerConfigurator;
      _focusManagerConfigurator = 0;
   }
#ifndef WIN32
   hmibase::gadget::SyncBlockConsumer::kill();
#endif
}


bool CgiApplicationBase::Init(AppPlatform::AppEnvironment* appEnvironment, ::asf::core::BaseComponent* baseComponent)
{
   /// for debug puposes    Courier::Diagnostics::LogControl::SetLogLevel(Courier::Diagnostics::LogLevel::Debug);
   HMI_APP_ASSERT(0 != appEnvironment);

   mAppEnvironment = appEnvironment;
#ifdef WIN32
#else
   _hmiAppCtrlProxyHandler.setBaseComponent(baseComponent);
   _hmiAppCtrlProxyHandler.setSurfaceIds(mAppEnvironment->getAppSettings().getScreenBrokerSettings().ids);
#endif
   Util::Timer::createTimerThread();
   // Setup platform specific application environment

   HMI_APP_ASSERT(mAppEnvironment->Setup(mAssetFileNames) == true);

   //Added for Bugfix from FEAT so that Popups get Touch-Messages, 13.11.13
   //FEAT's patch function for finding widgets filtered based on surface id
   Courier::ViewScene2D::SetTouchedWidget2DFunction(hmibase::input::ScreenIdFilteredGetTouchedWidget2D);
   Courier::ViewScene3D::SetTouchedWidget3DFunction(hmibase::input::ScreenIdFilteredGetTouchedWidget3D);
   hmibase::widget::Widget::SetBoundingBoxCheckFunction(hmibase::input::isInsideBoundingBox2D);

   //Use the widget order as it was defined in SceneComposer
   Courier::ViewScene::SetWidgetSortingStrategy(Courier::ViewScene::NoneWidgetSorting);

   setDefaultRenderConfiguration();

   // create type converter objects
   Courier::RegisterStdTypeConverters();
   //SetupSampleTypeConverter();

#ifdef FEATSTD_MEMORYPOOL_ENABLED
   FeatStd::MemoryManagement::MemoryPoolManager::SetOnDestructPoolsCallback(&onDestructPoolsCallback);
#endif
#ifndef WIN32
   if (mAppEnvironment->getAppSettings().getSyncBlockID() > 0)
   {
      hmibase::app::base::AppConfigValRegistry::registerItem(hmibase::app::base::ConfigItems::SYNCBLOCK_ID, mAppEnvironment->getAppSettings().getSyncBlockID());
      hmibase::gadget::SyncBlockConsumer::initialize(static_cast<int>(mAppEnvironment->getAppSettings().getSyncBlockID()),
            static_cast<int>(mAppEnvironment->getAppSettings().getNoOfSyncBlockProducerInstances()));
   }

   // lazy initialization during first access
   //SyncBlockProducerFactory::Init();
#endif

   // Initialize the ViewHandler:
   // The Courier::AssetConfiguration is retrieved from the platform specific app-environment.
   // Assets have to be added to the Courier::AssetConfiguration, this is done inside app-environment.
   HMI_APP_ASSERT(mViewHandler.Init(getViewFactory(), const_cast<Courier::AssetConfiguration*>(mAppEnvironment->GetAssetConfiguration()), &mRenderer, "") == true);

   // set the view controller factory (optional)
   mViewHandler.SetViewControllerFactory(getViewControllerFactory());
   mViewHandler.SetTouchSession(&mDemoTouchSession);

   //register custom message handlers
   mViewHandler.registerMessageHandler(hmibase::trace::ViewTraverser::getInstance());
   hmibase::trace::ViewTraverser::getInstance().setViewHandler(&mViewHandler);

//#ifdef WIN32
   mViewHandler.registerMessageHandler(_viewMetaInfoCollector);
   _viewMetaInfoCollector.setViewHandler(&mViewHandler);
//#endif
   // Initialize the view component
   HMI_APP_ASSERT(mViewComponent.Init(&mViewHandler) == true);

   // Initialize the render component
   HMI_APP_ASSERT(mRenderComponent.Init(&mViewHandler) == true);

   mAppEnvironment->BeforeRun(mViewHandler);

   HMI_APP_ASSERT(initializeFocusManager() == true);

   // called in derived class
   //HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_GUIFW_INIT_END);

   return true;
}


void CgiApplicationBase::setMaxFPSRenderConfiguration()
{
   Courier::RenderConfiguration renderCfg = mRenderer.GetRenderConfiguration();

   renderCfg.UseInvalidation(false);
   renderCfg.UseWakeUpRenderMechanism(false);
   renderCfg.UseCourierSwapping(true);
   hmibase::widget::Widget::useDirtyRectangle(false);
   mRenderer.SetRenderConfiguration(renderCfg);
}


void CgiApplicationBase::setDefaultRenderConfiguration()
{
   Courier::RenderConfiguration myRenderConfiguration;

   setupRenderTargetConfiguration(myRenderConfiguration);

   myRenderConfiguration.UseInvalidation(true);
   myRenderConfiguration.UseWakeUpRenderMechanism(true); // prevents continually msgQue polling
   myRenderConfiguration.UseCourierSwapping(true);

   //Clear the render target content while Gdu loading
   myRenderConfiguration.ClearRenderTargetOnUpload(true);

   mRenderer.SetRenderConfiguration(myRenderConfiguration);

   ETG_TRACE_FATAL_THR(("RenderConfiguration for maximum fps set for %s", hmibase::trace::getAppName().c_str()));
}


bool CgiApplicationBase::Run()
{
   //ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_PERFORMANCE_MP, "CGI_Run START [%20s] %u ms", HmiAsfComponentBase::getAppName(), hmibase::util::Ticker::getTickCountMsec()));
   HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_START_GUI_THREADS_BEGIN);

   bool setupOk = (setupMsgReceiverRenderer() &&
                   setupMsgReceiverView() &&
                   setupMsgReceiverSM());
   HMI_APP_ASSERT(setupOk);

   // Before sending the start up (resp. the first) message be sure to activate all message receivers
   mMsgReceiver.SetName("View_Controller_MsgReceiver");
   mMsgReceiver.Activate();
   HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_START_GUI_THREADS_END);

   if (setupOk)
   {
      HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_GUI_MAINLOOP_START);

      // Startup the system by posting the startup message after all components are in place
      Courier::Message* msg = COURIER_MESSAGE_NEW(Courier::StartupMsg)();
      if (0 != msg)
      {
         msg->Post();
      }

      while (mMsgReceiver.Process() && !mMsgReceiver.IsShutdownTriggered())
      {
#if defined(CANDERA_LAYOUT_MONITOR_ENABLED)
         Candera::LayoutLogger::Clear();
#endif
      }
      mAppEnvironment->AfterRun();
   }
   return setupOk;
}


bool CgiApplicationBase::Finalize()
{
   HMI_APP_ASSERT(0 != mAppEnvironment);
   if (0 == mAppEnvironment)
   {
      return false;
   }

   finalizeFocusManager();

   mViewHandler.Finalize();

   bool lRc = mAppEnvironment->Cleanup(mAssetFileNames);

#ifndef WIN32
   hmibase::gadget::SyncBlockProducerFactory::Deinit();
#endif

   terminateThreads();

   return lRc;
}


bool CgiApplicationBase::PostMsg(Courier::Message* msg)
{
   return (0 != msg) && (msg->Post());
}


void CgiApplicationBase::terminateThreads()
{
   bool lTerminate = false;
   do
   {
      lTerminate = true;
#if (0 != ENABLE_VIEW_THREAD)
      lTerminate = lTerminate && (Courier::Platform::ThreadStatus::Terminated == lViewThread.GetStatus());
#endif
#if (0 != ENABLE_RENDER_THREAD)
      lTerminate = lTerminate && (Courier::Platform::ThreadStatus::Terminated == lRenderThread.GetStatus());
#endif
#if (0 != ENABLE_CONTROLLER_THREAD)
      lTerminate = lTerminate && (Courier::Platform::ThreadStatus::Terminated == lControllerThread.GetStatus());
#endif
   }
   while (!lTerminate);

   //ETG_TRACE_USR1_THR_DCL((COURIER_APPLICATION_TRACE_GROUP.traceClass_Debug, "Main loop processing ended normal."));
}


bool CgiApplicationBase::setupMsgReceiverRenderer()
{
#if (0 != ENABLE_RENDER_THREAD)
   // Start view in separate thread
   lRenderThread.SetName("Render");
#if defined(USE_TWO_RENDER_THREADS)
   lRenderThread.Attach(mRenderComponent2d);
   lRenderThread.Attach(mRenderComponent3d);
#else
   lRenderThread.Attach(mRenderComponent);
#endif
   lRenderThread.Activate();
   return lRenderThread.Run();
#else
#if defined(USE_TWO_RENDER_THREADS)
   // Start two own render "threads" (this wont work under Win32 because because they have to run in main message loop)
   // For using real multi threading the class ComponentMessageReceiverThread could be used for other components
   // in this sample code.
   mMsgReceiver.Attach(&mRenderComponent2d);
   mMsgReceiver.Attach(&mRenderComponent3d);
   return true;
#else
   // Attach render components to main message receiver (has to run in main loop at least for Win32)
   mMsgReceiver.Attach(&mRenderComponent);
   return true;
#endif
#endif
}


bool CgiApplicationBase::setupMsgReceiverView()
{
   // Attach view component to respective message receiver
#if (0 != ENABLE_VIEW_THREAD)
   // Start view in separate thread
   lViewThread.SetName("View");
   lViewThread.Attach(mViewComponent);
   if (mAppEnvironment->GetWindowEventExtCommComponent())
   {
      lViewThread.Attach(*mAppEnvironment->GetWindowEventExtCommComponent());
   }
   lViewThread.Activate();
   return lViewThread.Run();
#else
   mMsgReceiver.Attach(&mViewComponent);
   if (mAppEnvironment && (0 != const_cast<Courier::TouchHandling::TouchEventPreprocessor*>(mAppEnvironment->GetTouchEventPreprocessor())))
   {
      mMsgReceiver.AttachPreprocessor(const_cast<Courier::TouchHandling::TouchEventPreprocessor*>(mAppEnvironment->GetTouchEventPreprocessor()));
   }
   return true;
#endif
}


#ifdef FEATSTD_MEMORYPOOL_ENABLED
/* Memory statistics - Dump file names */
#define HMI_C_PERM_BLOCK_FILE "PermanentBlockList_Shutdown.csv"

void CgiApplicationBase::onDestructPoolsCallback()
{
#ifdef FEATSTD_MEMORYPOOL_STATISTICS_ENABLED
#define HMI_C_POOL_STATISTICS_FILE "MemoryPoolStatistics.csv"
#define HMI_C_BIN_STATISTICS_FILE "MemoryPoolBinStatistics.csv"
   FEATSTD_MEMORYPOOL_LEAK_DETECTION_CODE(FeatStd::MemoryManagement::MemoryPoolManager::DumpAllocatedBlocks());
   FEATSTD_MEMORYPOOL_STATISTICS_CODE(FeatStd::MemoryManagement::MemoryPoolManager::DumpStatistics());

   StatisticsFileDumper::Dump(HMI_C_POOL_STATISTICS_FILE, HMI_C_BIN_STATISTICS_FILE);
#endif
   PermanentBlockFileDumper::Dump(HMI_C_PERM_BLOCK_FILE);
}


#endif

bool CgiApplicationBase::initializeFocusManager()
{
   bool result = false;
   if (_focusManagerConfigurator == NULL)
   {
      _focusManagerConfigurator = createFocusManagerConfigurator();
      if (_focusManagerConfigurator != NULL)
      {
         result = _focusManagerConfigurator->initialize();

         mViewHandler.registerMessageHandler(Focus::FManager::getInstance());
      }
   }
   return result;
}


bool CgiApplicationBase::finalizeFocusManager()
{
   bool result = false;
   if (_focusManagerConfigurator != NULL)
   {
      mViewHandler.unregisterMessageHandler(Focus::FManager::getInstance());

      _focusManagerConfigurator->finalize();
      delete _focusManagerConfigurator;
      _focusManagerConfigurator = NULL;
   }
   return result;
}


Focus::FManagerConfigurator* CgiApplicationBase::createFocusManagerConfigurator()
{
   return new Focus::FDefaultManagerConfigurator(Focus::FManager::getInstance(), &mViewHandler);
}
