/* ***************************************************************************************
* FILE:          CgiApplication.hpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CgiApplication.hpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(_CGI_Application_hpp_)
#define _CGI_Application_hpp_

#include "View/CGI/GuiBase/CgiApplicationBase.h"
#include <View/CGI/CgiExtensions/ImageUploadCache.h>
#include <Cit/VisualStateGlueLayer/VisualStateGlueLayer.h>
#include "Trace/TraceQueryHelper.hpp"
#include "AppBase/ITraceQueryHandler.h"
#include "Trace/TraceUtils.h"
#include "Trace/GuiInfo.h"
#include "Trace/StartupInvestigation.h"
#include "hmibase/util/StringUtils.h"

#include "hmi_trace_if.h"
#include <Widgets/2D/WidgetFinder2D.h>

#include "Courier/Visualization/RenderingMonitor.h"

#include "Trace/WidgetFinderCallback.h"

#ifdef WIN32
#ifndef VARIANT_S_FTR_ENABLE_LAYOUT_MONITOR
#define VARIANT_S_FTR_ENABLE_LAYOUT_MONITOR
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_LAYOUT_MONITOR
#include "../../../tools/CGIStudioLayoutMonitor/LayoutLogger.h"
#endif

#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
#if (defined VARIANT_S_FTR_ENABLE_GLOBAL_SHADER_CACHE) || (defined VARIANT_S_FTR_ENABLE_APPLICATION_SCOPED_SHADER_CACHE)
#include "View/CGI/CgiExtensions/ShaderStorage.h"
#include <CanderaPlatform/Device/Common/OpenGLES/GlShaderStorageProvider.h>
#endif
#endif

#include "View/CGI/CgiExtensions/RendererStatisticsOverlayWrapper.h"

//namespace hmibase {

template<typename TStatemachine>
class CgiApplication : public CgiApplicationBase, hmibase::app::base::ITraceQueryHandler
{
      friend class TraceQueryHandler;
   public:
      // only possible with C++11
      //      CgiApplication(const FeatStd::Char* cAssetFileName, hmibase::services::hmiappctrl::ProxyHandler& hmiAppCtrlProxyHandler):
      //    	  CgiApplication(std::vector<std::string>(1, cAssetFileName), hmiAppCtrlProxyHandler)
      //      {
      //
      //      }

      CgiApplication(const FeatStd::Char* cAssetFileName, hmibase::services::hmiappctrl::ProxyHandler& hmiAppCtrlProxyHandler):
         CgiApplicationBase(std::vector<std::string>(1, cAssetFileName), hmiAppCtrlProxyHandler),
         _glueLayer(hmiAppCtrlProxyHandler, *this), _rendererStaticsOverlayWrapper(0)
      {
      }

      CgiApplication(const std::vector<std::string>& cAssetFileNames, hmibase::services::hmiappctrl::ProxyHandler& hmiAppCtrlProxyHandler) :
         CgiApplicationBase(cAssetFileNames, hmiAppCtrlProxyHandler),
         _glueLayer(hmiAppCtrlProxyHandler, *this)   // TODO: fr83hi this ptr is here not valid, furthermore TraceQueryHandler implementation needs to be reworked asap !!!
         , _rendererStaticsOverlayWrapper(0)
      {
      }

      virtual ~CgiApplication()
      {
         if (_rendererStaticsOverlayWrapper)
         {
            CANDERA_DELETE(_rendererStaticsOverlayWrapper);
            _rendererStaticsOverlayWrapper = 0;
         }
      }

      virtual bool Init(AppPlatform::AppEnvironment* appEnvironment, ::asf::core::BaseComponent* baseComponent = 0)
      {
         HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_PERFORMANCE_MP, "CGI_Init START [%20s] %u ms", hmibase::trace::getAppName().c_str(), hmibase::util::Ticker::getTickCountMsec());
         HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_GUIFW_INIT_BEGIN);

#ifdef VARIANT_S_FTR_ENABLE_LAYOUT_MONITOR
         Candera::LayoutLogger::Register();
#endif

#ifdef VARIANT_S_FTR_ENABLE_IMAGE_UPLOAD_CACHE
         hmibase::view::utils::ImageUploadCache::getInstance().enable();
#endif

#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
#if (defined VARIANT_S_FTR_ENABLE_GLOBAL_SHADER_CACHE)
         static hmibase::view::ShaderStorage cache(false);
         Candera::GlShaderStorageProvider::SetShaderCache(&cache);
#elif (defined VARIANT_S_FTR_ENABLE_APPLICATION_SCOPED_SHADER_CACHE)
         static hmibase::view::ShaderStorage cache(true);
         Candera::ShaderStorageProvider::SetShaderCache(&cache);
#endif
#endif

         // Initialize the model component
         HMI_APP_ASSERT(CgiApplicationBase::Init(appEnvironment, baseComponent) == true);

#ifdef  VARIANT_S_FTR_ENABLE_TO_DISABLE_CLEARON_SCENELOADING
         Courier::ViewScene::SetClearOnSceneLoading(false);
#else
         //Need not clear the content of a view when 2 cameras are configured
         bool flag = ::getenv("DISABLE_CLEARON_SCENELOADING") != 0 ? true : false;
         if (flag)
         {
            Courier::ViewScene::SetClearOnSceneLoading(false);
         }
#endif

         hmibase::trace::GuiInfo::setViewHandler(&GetViewHandler());

#if defined VARIANT_S_FTR_ENABLE_ARABIC_LAYOUT_PATCH
         mViewHandler.setArabicLayoutPatch(true);
#endif

#if defined VARIANT_S_FTR_ENABLE_DIRTY_RECTANGLE
         BaseWidget2D::useDirtyRectangle(true);
#else
         char* val = ::getenv("DIRTY_RECTANGLE_SUPPORT");
         BaseWidget2D::useDirtyRectangle(val != 0 ? true : false);
#endif

#if defined ( COURIER_RENDERING_MONITOR_ENABLED ) && defined ( FEATSTD_MONITOR_ENABLED )
         //#if defined VARIANT_S_FTR_ENABLE_CGI_RENDER_PERF_ANALYSIS
         //! [RenderMonitor_Init]
         // initialize the rendering monitor, for monitoring purposes (optional)
#ifdef VARIANT_S_FTR_ENABLE_CGI_ANALYZER_ONLINE_MODE
         HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_CGI,  "CGI_Init [%20s] setup rendering monitor in online mode, IP %20s, port %u", hmibase::trace::getAppName().c_str(), FEATSTD_MONITOR_TCPIP_ADDRESS, FEATSTD_MONITOR_TCPIP_PORT);
         HMI_APP_ASSERT(_renderingMonitor.Init(FEATSTD_MONITOR_TCPIP_ADDRESS, FEATSTD_STRINGIZE(FEATSTD_MONITOR_TCPIP_PORT)) == true);
#else
         HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_CGI,  "CGI_Init [%20s] enable Candera perfomance measures", hmibase::trace::getAppName().c_str());
         CANDERA_PERF_SET_ENABLED(true);
#endif
         //if (rc)
         {
            static_cast<Courier::ViewHandler&>(GetViewHandler()).SetRenderingMonitor(&_renderingMonitor);
         }
         //! [RenderMonitor_Init]
         //#endif
#endif

         HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_PERFORMANCE_MP, "CGI_Init END [%20s] %u ms", hmibase::trace::getAppName().c_str(), hmibase::util::Ticker::getTickCountMsec());
         HMI_STARTUP_INVESTIGATION(hmibase::trace::StartupInvestigation::MP_GUIFW_INIT_END);

         return true;
      }

      virtual bool setupMsgReceiverSM()
      {
         mMsgReceiver.Attach(&_glueLayer);
         return true;
      }

      virtual void setupRenderTargetConfiguration(Courier::RenderConfiguration& myRenderConfiguration)
      {
#ifdef  VARIANT_S_FTR_ENABLE_DYNAMIC_RENDERTARGET_LOADING
         myRenderConfiguration.LoadAllRenderTargetsAtStartup(false);
         _hmiAppCtrlProxyHandler.enableDirectApplicationActivation(false);
         mViewHandler.enableDirectApplicationActivation(false);
#else
         myRenderConfiguration.LoadAllRenderTargetsAtStartup(true);
#endif
      }

      void GetVisibleMainScenes(std::vector<std::string>& views)
      {
         hmibase::trace::TraceQueryHelper::GetInstance().PrintActiveViewIDs(mViewHandler.getRenderViewVisitor(), views);
      }

      COURIER_MSG_MAP_BEGIN(0)
      ON_COURIER_MESSAGE(QueryTraceInfoReqMsg)
      ON_COURIER_MESSAGE(SbCurrentStatusUpdMsg)
      ON_COURIER_MESSAGE(SimTouchReqMsg)
      ON_COURIER_MESSAGE(WidgetPropertySetterReqMsg)
      COURIER_MSG_MAP_END()

      bool onCourierMessage(const QueryTraceInfoReqMsg& msg)
      {
         bool result = true;

         switch (msg.GetQueryId())
         {
            case hmibase::GET_CURRENT_MAIN_VIEW:
            {
               std::vector<std::string> views;
               ::hmibase::trace::TMLCurrentViewInfo currentViewInfo = hmibase::trace::TraceQueryHelper::GetInstance().PrintActiveViewIDs(mViewHandler.getRenderViewVisitor(), views);

               for (std::vector<std::string>::iterator it = views.begin(); it != views.end(); it++)
               {
                  hmibase::trace::GuiInfo::printScene(it->c_str());
               }
#ifndef WIN32
               hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload1(
                  ::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                        ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_CURRENT_OTHER_VIEW, currentViewInfo.otherScenes));
               _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload1);

               hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload2(
                  ::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                        ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_CURRENT_MAIN_VIEW, currentViewInfo.mainScenes));
               _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload2);
#endif
            }
            break;

            case hmibase::GET_STATEMACHINE_INFO:
            {
#ifndef WIN32
               std::vector<std::string> s;
               s.push_back(hmibase::trace::TraceQueryHelper::GetInstance().CollectSMInfo(_glueLayer.GetStateMachine(), hmibase::trace::getAppName()));

               hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload(::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                     ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_STATEMACHINE_INFO, s));
               _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload);
#endif
            }
            break;

            case hmibase::ENABLE_ARABIC_PATCH:
            {
               mViewHandler.setArabicLayoutPatch(msg.GetUserData1() == 0 ? false : true);
            }
            break;

            case hmibase::TOGGLE_INVALIDATION:
            {
               static bool flag = true;
               if (flag)
               {
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_CGI,  "[%s]: Use default render configuration", hmibase::trace::getAppName().c_str());
                  setDefaultRenderConfiguration();
               }
               else
               {
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_CGI,  "[%s]: Use max fps render configuration", hmibase::trace::getAppName().c_str());
                  setMaxFPSRenderConfiguration();
               }
               flag = flag ? false : true;
            }
            break;
            case hmibase::TOUCHSESSION_INFO:
            {
               CgiApplicationBase::mDemoTouchSession.printInfo();
            }
            break;
            case hmibase::SET_CGITRACELEVEL:
            {
               FeatStd::Diagnostics::LogLevel::Enum level = static_cast<FeatStd::Diagnostics::LogLevel::Enum>(msg.GetUserData1());
               HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_CGI,  "[%25s] CGI trace level set to %d", hmibase::trace::getAppName().c_str(), ETG_CENUM(FeatStd::Diagnostics::LogLevel::Enum, level));
               FeatStd::Diagnostics::LogControl::SetLogLevel(level);
            }
            break;

            case hmibase::TOGGLE_RENDERSTATISTICS_OVERLAY:
            {
               if (_rendererStaticsOverlayWrapper == 0)
               {
                  _rendererStaticsOverlayWrapper = CANDERA_NEW(hmibase::view::RendererStatisticsOverlayWrapper)();

                  if (_rendererStaticsOverlayWrapper)
                  {
                     _rendererStaticsOverlayWrapper->Init();
                  }
               }

               if (_rendererStaticsOverlayWrapper)
               {
                  _rendererStaticsOverlayWrapper->mIsOverlayEnabled = !_rendererStaticsOverlayWrapper->mIsOverlayEnabled;
                  HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_FATAL, TR_CLASS_HMI_CGI,  "[%25s] RenderStatisticsOverlay %s", hmibase::trace::getAppName().c_str(), _rendererStaticsOverlayWrapper->mIsOverlayEnabled ? "enabled" : "disabled");
               }
            }
            break;

            case hmibase::GET_ALL_LOADEDSCENES:
            {
               ::hmibase::trace::TMLLoadedViewInfo loadedViewInfo;
               const Courier::UInt32 QueryId = msg.GetUserData1();
               hmibase::trace::TraceQueryHelper::GetInstance().GetAllLoadedScenes(QueryId, GetViewHandler(), loadedViewInfo);

#ifndef WIN32
               switch (msg.GetUserData1())
               {
                  case hmibase::LOADED:
                  {
                     hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload1(
                        ::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                              ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_ALL_LOADEDSCENES, loadedViewInfo.loadedScenes));
                     _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload1);
                  }
                  break;
                  case hmibase::ACTIVE_RENDERED:
                  {
                     hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload1(
                        ::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                              ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_ALL_ACTIVE_RENDERED_SCENES, loadedViewInfo.activeRenderedScenes));
                     _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload1);
                  }
                  break;
                  case hmibase::ACTIVE_NOTRENDERED:
                  {
                     hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload1(
                        ::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                              ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_ALL_ACTIVE_NOTRENDERED_SCENES, loadedViewInfo.activeNotRenderedScenes));
                     _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload1);
                  }
                  break;
                  case hmibase::INACTIVE:
                  {
                     hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload1(
                        ::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
                              ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_ALL_INACTIVE_SCENES, loadedViewInfo.inactiveScenes));
                     _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload1);
                  }
                  break;
                  default:
                     //do nothing
                     break;
               }
#endif
            }

            default:
               result = hmibase::trace::TraceQueryHelper::GetInstance().ProcessTraceQueryRequest(msg, GetViewHandler(), mViewHandler.getRenderViewVisitor());
               break;
         }

         return result;
      }

      bool onCourierMessage(const SbCurrentStatusUpdMsg& msg)
      {
         std::string s;

         if (msg.GetViewId() != Courier::ViewId())
         {
            switch (msg.GetRequestId())
            {
               case hmibase::GET_CURRENT_VISIBLE_POPUP_VIEW:
                  s += "TP";
                  break;
               default:
                  // do not handle it here
                  return true;
            }

            char buf[10];
            SNPRINTF(buf, sizeof buf, "%u:%u", msg.GetDisplayId(), msg.GetSurfaceId());
            s += "[";
            s += hmibase::trace::getAppName();
            s += "]:[";
            s += buf;
            s += "], ";
            s += msg.GetStatus() ? "(v)" : "(iv)";
            s += " '";
            s += msg.GetViewId().CStr();
            s += "'";
         }
#ifndef WIN32
         std::vector<std::string>v;
         v.push_back(s);

         hmibase::services::hmiappctrl::UpdateTraceQueryResultPayload payload(::bosch::cm::ai::hmi::appctrl::trace::Commands::TraceQueryData(getpid(),
               ::bosch::cm::ai::hmi::appctrl::trace::Commands::traceQueryId__GET_CURRENT_VISIBLE_POPUP_VIEW, v));
         _hmiAppCtrlProxyHandler.sendLocalMessageHMIBaseTraceCmd(payload);
#endif
         return false; // let other components also process
      }

      bool onCourierMessage(const SimTouchReqMsg& msg)
      {
         return hmibase::trace::TraceQueryHelper::GetInstance().ProcessSimTouchRequest(msg, GetViewHandler(), mViewHandler.getRenderViewVisitor());
      }

      bool onCourierMessage(const WidgetPropertySetterReqMsg& msg)
      {
         return hmibase::trace::TraceQueryHelper::GetInstance().ProcessWidgetPropertySetRequest(msg, GetViewHandler(), mViewHandler.getRenderViewVisitor());
      }

   protected:

      virtual Courier::ViewFactory* getViewFactory()
      {
         return &_viewFactory;
      }

      virtual Courier::ViewControllerFactory* getViewControllerFactory()
      {
         return &_viewControllerFactory;
      }

   private:
      CGIAppViewFactory					_viewFactory;
      CGIAppViewControllerFactory			_viewControllerFactory;

      Cit::VisualStateGlueLayer<TStatemachine, hmibase::services::hmiappctrl::ProxyHandler , CGIAppController> _glueLayer;

#if defined ( COURIER_RENDERING_MONITOR_ENABLED ) && defined ( FEATSTD_MONITOR_ENABLED )
      //#if defined VARIANT_S_FTR_ENABLE_CGI_RENDER_PERF_ANALYSIS
      Courier::RenderingMonitor           _renderingMonitor;
      //#endif
#endif
      hmibase::view::RendererStatisticsOverlayWrapper* _rendererStaticsOverlayWrapper;
};


//}

#endif
