/* ***************************************************************************************
* FILE:          AppEnvironment.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppEnvironment.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(AppPlatform_AppEnvironment_h)
#define AppPlatform_AppEnvironment_h

#include <Courier/Base.h>

#include <CanderaPlatform/Device/Common/Base/Display.h>

#include <FeatStd/Diagnostics/FileAppender.h>

#include <Courier/Visualization/AssetConfiguration.h>
#include <View/CGI/InputHandling/InputThread.h>
#include <View/CGI/InputHandling/DisplayConfiguration.h>
#include <View/CGI/InputHandling/DisplayInputContext.h>
#include <View/CGI/TouchHandling/TouchEventPreprocessor.h>

#include <vector>

#ifndef FEATSTD_LOG_ENABLED
#error Feature FEATSTD_ENABLE_LOG is not enabled. Do not include any FeatStd logging releated header files!
#endif

class IApplicationSettings;

namespace Courier {
class ViewHandler;
}


namespace AppPlatform {
class AppEnvironment
{
   public:
#if defined(FEATSTD_DLT_ENABLED)
      AppEnvironment(FeatStd::Char* cDltRegistration, FeatStd::Char* cDltApplication, const IApplicationSettings&);
#else
      explicit AppEnvironment(const IApplicationSettings&);
#endif

      virtual ~AppEnvironment();

      /** Initializes the application platform specific modules, like asset loading, display creation, external input handling, a.s.o.
          @return <em>true</em> True if successful,
                  <em>false</em> otherwise.*/
      virtual bool Setup(const std::vector<std::string>& cAssetFileNames);

      /** Disposes the application platform specific modules, like asset, display, external input handling, a.s.o.
          @return <em>true</em> True if successful,
                  <em>false</em> otherwise.*/
      virtual bool Cleanup(const std::vector<std::string>& cAssetFileNames);

      /**
       *
       */
      virtual Candera::AssetRepository* GetAssetRepo(const char* assetFileName);

      /** @return The asset configuration, which was read in the platform specific asset loading done in Setup(). */
      const Courier::AssetConfiguration* GetAssetConfiguration()
      {
         return &mAssetConfiguration;
      }

      /** @return Application platform defined number of external input contexts. */
      FeatStd::Int GetDisplayInputContextCount() const;

      /** @return Application platform specific external input context (@see DisplayInputContext). */
      const Courier::InputHandling::DisplayInputContext* GetDisplayInputContext(FeatStd::Int i) const
      {
         return &mDisplayInputContexts[i];
      }

      const Courier::TouchHandling::TouchEventPreprocessor* GetTouchEventPreprocessor()
      {
         return &mTouchEventPreprocessor;
      }

      virtual const struct Courier::InputHandling::DisplayConfiguration* GetDisplayConfiguration(FeatStd::UInt8 i) const
      {
         return &mDisplayConfigurations[i];
      }

      const IApplicationSettings& getAppSettings()
      {
         return _appSettings;
      }

      virtual void BeforeRun(Courier::ViewHandler& /*viewHandler*/)
      {
      }

      virtual void AfterRun()
      {
      }

      /** Indicates that an asset repository was loaded or unloaded. */
      class AssetRepositoryEvent : public FeatStd::Event
      {
         public:
            FEATSTD_TYPEDEF_BASE(FeatStd::Event);
            FEATSTD_RTTI_DECLARATION();

            enum State { Undef, Added, Removed };

            AssetRepositoryEvent(Candera::AssetRepository* assetRepo, const std::string& assetFileName, State state)
               : _assetRepository(assetRepo), _fileName(assetFileName), _state(state)
            {
            }

            virtual ~AssetRepositoryEvent()
            {
               _assetRepository = NULL;
            }

            Candera::AssetRepository* GetAssetRepository() const
            {
               return _assetRepository;
            }

            const std::string& GetFileName() const
            {
               return _fileName;
            }

            State GetState() const
            {
               return _state;
            }

         private:
            Candera::AssetRepository* _assetRepository;
            std::string _fileName;
            State _state;
      };

      /** Asset repository event listeners must be added to this event source */
      static FeatStd::EventSource& GetEventSource();

   protected:
      virtual bool BeforeInputContextSetup() = 0;
      virtual void SetupInputHandler(int) = 0;
      virtual void SetupInputThread(int) = 0;
      virtual bool SyncWindowCreation()
      {
         return false;
      }
      virtual void SetupTouchPreprocessor() {};

      const IApplicationSettings& _appSettings;
      Courier::InputHandling::InputThread mMainWindowInputThread;

#if defined(FEATSTD_LOG_ENABLED)
      FeatStd::Diagnostics::FileAppender mFileAppender;
#endif

      static Courier::InputHandling::DisplayInputContext mDisplayInputContexts[];
      static Courier::InputHandling::DisplayConfiguration mDisplayConfigurations[];

   private:
      /** Checks the platform display configuration against the display configuration in the asset. */
      void CheckAssetDisplayConfiguration();
      Courier::AssetConfiguration mAssetConfiguration;

   protected:
      // Touch event preprocessing, filter out unnecessary Move events
      Courier::TouchHandling::TouchEventPreprocessor mTouchEventPreprocessor;
};


}
#endif
