/* ***************************************************************************************
* FILE:          VisualStateGlueLayer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  VisualStateGlueLayer.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined (VisualStateGlueLayer_h)
#define VisualStateGlueLayer_h

#include <iostream>
#include <Courier/Courier.h>
#include <Courier/Foundation/IComponentNotification.h>
#include "AppBase/ITraceQueryHandler.h"
#include <Cit/VisualStateGlueLayer/DeductResult.h>
#include "Cit/VisualStateGlueLayer/VSTraceHelper.hpp"

namespace Cit {
namespace Internal {
template <class Type>
class HasSEM_NextState
{
      template <typename T, T> struct TypeCheck;
      template <typename T> struct Function
      {
         typedef FeatStd::UInt8(T::*Pointer)(void);
      };

      template <typename T> static FeatStd::Internal::TypeTraitsPrivate::Positive Fn(TypeCheck<typename Function<T>::Pointer, &T::SEM_NextStateChg >*);
      template <typename T> static FeatStd::Internal::TypeTraitsPrivate::Negative Fn(...);

   public:
      enum { Value = (sizeof(Fn<Type>(0)) == sizeof(FeatStd::Internal::TypeTraitsPrivate::Positive)) };
};


template<bool isVisualState7_x_compatible_api, ::FeatStd::UInt8 okCode, typename StateMachineType>
struct VisualStateGlueLayerHelper
{
   typedef unsigned char (StateMachineType::*Pointer)(SEM_EVENT_TYPE EventNo, ...);

   inline static Pointer GetPointer()
   {
      return &StateMachineType::VSDeduct;
   }

   inline static ::Cit::DeductResult::Enum ProcessOutput(StateMachineType& /*stateMachine*/, FeatStd::UInt8 errorCode)
   {
      return (errorCode == okCode) ? ::Cit::DeductResult::DeductOk : ::Cit::DeductResult::DeductError;
   }
};


template< ::FeatStd::UInt8 okCode, typename StateMachineType >
struct VisualStateGlueLayerHelper<true, okCode, StateMachineType>
{
   typedef unsigned char (StateMachineType::*Pointer)(typename StateMachineType::VS_Type_Event/*FeatStd::UInt8*/ EventNo, ...);

   static inline Pointer GetPointer()
   {
      return &StateMachineType::SEM_Deduct;
   }

   inline static ::Cit::DeductResult::Enum ProcessOutput(StateMachineType& stateMachine, ::FeatStd::UInt8  errorCode)
   {
      if (okCode == errorCode)
      {
         bool actionCalled = false;
         typename StateMachineType::VS_Type_ActionExpression actionExpressionNr;
         /* Get resulting action expressions and execute them. */
         VSTraceHelper<StateMachineType> helper(&stateMachine);
         FeatStd::Int localErrorCode = stateMachine.SEM_GetOutput(&actionExpressionNr);;
         while (StateMachineType::VS_Ret_Found == localErrorCode)
         {
            actionCalled = true;
            helper.VSAction(actionExpressionNr);
            localErrorCode = stateMachine.SEM_GetOutput(&actionExpressionNr);
         }
         /* Check for error. */
         if (StateMachineType::VS_Ret_Ok == localErrorCode)
         {
            /* Change the next state vector. */
            localErrorCode = helper.SEM_NextStateChg();
         }
         return (StateMachineType::VS_Ret_Ok == localErrorCode) ? ((actionCalled) ? DeductResult::ExternalTransition_ActionCalled : DeductResult::DeductOk) : DeductResult::DeductError;
      }
      return ::Cit::DeductResult::DeductError;
   }
};


template< ::FeatStd::UInt8 okCode, typename StateMachineType >
struct VSDeductHelper
{
   enum { isVisualState7_x_compatible_api = HasSEM_NextState<StateMachineType>::Value };
   typedef VisualStateGlueLayerHelper < isVisualState7_x_compatible_api != 0, okCode, StateMachineType > Helper;
   typedef typename Helper::Pointer Pointer;
   inline static Pointer GetPointer()
   {
      return Helper::GetPointer();
   }

   inline static ::Cit::DeductResult::Enum ProcessOutput(StateMachineType& stateMachine, FeatStd::UInt8 errorCode)
   {
      return Helper::ProcessOutput(stateMachine, errorCode);
   }
};


}

class CitControllerComponent : public Courier::ControllerComponent
{
   protected:
      virtual bool OnMessage(const Courier::Message& /*msg*/)
      {
         return false;
      }
};


/// @addtogroup CIT
/// @{

/**
 * \brief Glue layer that enables a visualState generated state machine receiving messages.
 *
 * This glue layer is used to receive Courier messages. It triggers the message to event
 * mapping, fires the events into the state machine and processes the output of the state machine.
 *
 * The state machine that is used with this glue layer has to fullfill the following requirements
 * which need to be taken care of whilst code generation in visualState:
 * - it must be in C++,
 * - hence the Basic API has to be used,
 * - the API function 'SEM_InitAll' must be available
 * - using the Courier basic data types is recommended (Coder options -> Types)
 *
 * This template implements a Courier::ControllerComponent to be able to receive Courier
 * messages. It has to be instantiated with the class type of the state machine implementation
 * (i.e. the class that derives from the abstract state machine class generated by visualState).
 * Most likely the state machine implementation is generated using the CIT generator which
 * takes an HMI Contract as input.
 *
 * Example:
 * \code
 * Courier::ComponentMessageReceiver mMsgReceiver;
 * Cit::VisualStateGlueLayer<MyStateMachineImpl> myGlueLayer;
 * mMsgReceiver.Attach(&myGlueLayer);
 * ...
 * // Startup the system by posting the startup message
 * (COURIER_MESSAGE_NEW(Courier::StartupMsg)())->Post();
 * mMsgReceiver.Process();
 * ...
 * // Send messages to state machine
 * (COURIER_MESSAGE_NEW(SampleStateMachineController::SoftKeyPressed)(StationListButtonKeyCode))->Post();
 * mMsgReceiver.Process();
 * ...
 * \endcode
 *
 * This sample shows the basic mechansims how to use the glue layer. Inside the class an instance of
 * the state machine is created and consequently used to process
 * the events coming in with Courier messages. The state machine is initialized on the reception of
 * the Courier::StartupMsg. Therefore it has to be made sure that the component is registered in
 * time to receive the Courier::StartupMsg.
 *
 * The ControllerBaseClass has to provide the following public methods:
 *
 * void SetComponentNotification(::Courier::IComponentNotification * notification);
 * virtual bool OnMessage(const Courier::Message & msg);
 *
 * The method OnMessage has to be called for each message that should be processed by the state machine.
 * The base implementation of the method OnMessage has callable by the VisualStateGlueLayer class.
 * The databinding messages have to be forwarded to the method IComponentNotification::OnDataBindingMsg.
 * Otherwise, data events will not be triggered for the state machine.
 */

class CitDefaultControllerComponent : public Courier::ControllerComponent
{
   protected:
      virtual bool OnMessage(const Courier::Message& /*msg*/)
      {
         return false;
      }
};


template <class StateMachineType, class TProxyHandler, class TCGIAppController = CitDefaultControllerComponent>
class VisualStateGlueLayer : public TCGIAppController
{
   public:
      enum { VS_Ret_Ok = StateMachineType::VS_Ret_Ok};
      enum { VS_Ret_Found = StateMachineType::VS_Ret_Found};
      enum { VS_Evt_Reset = StateMachineType::VS_Evt_Reset};
      enum { VS_Ret_Error = StateMachineType::VS_Ret_Error};

      VisualStateGlueLayer(TProxyHandler& commObj, hmibase::app::base::ITraceQueryHandler& o) : TCGIAppController(commObj), traceQueryHandler(o)
      {
         mDataBindingMessageHandler.Init(this);
      }

      /**
       * \brief Return the internal state machine instance.
       */
      StateMachineType& GetStateMachine()
      {
         return mStateMachine;
      }

      /**
       * \brief Receive a Courier message and perform event mapping.
       *
       * This method receives messages from the Courier messaging system. Subsequently
       * it triggers the message to event mapping, fires the event into the statemachine and
       * processes the output by calling all action functions the state transition might
       * require.
       *
       * \param [in] msg A Courier message
       * \retval true If the message was mapped to an event successfully.
       * \retval false If the message could not be processed.
       */
      virtual bool OnMessage(const Courier::Message& msg)
      {
         bool messageConsumed = false;
         Courier::UInt32 eventId;

         if (HandleEventMapping(msg, eventId, messageConsumed))
         {
            //CIT_LOG_DEBUG("Handle Event %d", eventId);
            ::Cit::Internal::VSTraceHelper<StateMachineType> helper(&mStateMachine);
            DeductResult::Enum errorCode = helper.DeductEvent(msg, eventId);

            switch (errorCode)
            {
               case DeductResult::DeductOk:
                  messageConsumed = false; // there was neither a statechange nor an action called
                  break;
               case DeductResult::DeductError:
                  //CIT_LOG_ERROR("The state machine produced an error (code= %d)", errorCode);
                  break;
               default:
                  break;
            }
         }
         return messageConsumed || TCGIAppController::OnMessage(msg);
      }

   private:
      class DataBindingMessageHandler : public Courier::IComponentNotification
      {
         public:
            DataBindingMessageHandler() :
               mVisualStateGlueLayer(0)
            {
            }

            ~DataBindingMessageHandler()
            {
               if (0 != mVisualStateGlueLayer)
               {
                  mVisualStateGlueLayer->SetComponentNotification(0);
               }
            }

            void Init(VisualStateGlueLayer* visualStateGlueLayer)
            {
               mVisualStateGlueLayer = visualStateGlueLayer;
               mVisualStateGlueLayer->SetComponentNotification(this);
            }

            /**
             * \brief Receive a Courier data binding message and delegate it to OnMessage.
             *
             * This method receives data binding messages from the Courier messaging system.
             * Data binding messages are posted via OnMessage. Therefore, the component has to be
             * registered separately by using the components SetComponentNotification interface.
             * All data binding messages are delegated by this implementation of OnDataBindingMsg
             * to OnMessage.
             *
             * \param [in] cid The component id
             * \param [in] msg A Courier data binding message
             */
            virtual void OnDataBindingMsg(Courier::ComponentId cid, const Courier::Message* msg)
            {
               COURIER_UNUSED(cid);
               mVisualStateGlueLayer->OnMessage(*msg);
            }

         private:
            VisualStateGlueLayer* mVisualStateGlueLayer;
      };

      bool HandleEventMapping(const Courier::Message& msg, Courier::UInt32& eventId, bool& messageConsumed)
      {
         messageConsumed = false;

         if (msg.GetId() == Courier::StartupMsg::ID)
         {
            mStateMachine.SEM_InitAll();
            eventId = VS_Evt_Reset;
            // it's customary to not consume the StartupMsg
            return true;
         }
         else if (traceQueryHandler.OnMessage(msg))
         {
            // TODO:: currently in CgiApplication.hpp following messages handled <> ugly
            // ON_COURIER_MESSAGE(QueryTraceInfoReqMsg)
            // ON_COURIER_MESSAGE(SimTouchByNameReqMsg)
            // ON_COURIER_MESSAGE(SimTouchByIndexReqMsg)
            // consumed trace query command
         }
         else if (msg.GetTag() == Courier::MessageTag::DataBinding)
         {
            // look if we got an AbstractDataItemMsg or some other databinding message (eg ListEventMsg)
            const Courier::AbstractDataItemMsg* dataItemMsg = Courier::message_cast<const Courier::AbstractDataItemMsg*>(&msg);
            if (0 != dataItemMsg)
            {
               if (mStateMachine.HandleDataBindingEventMapping(*dataItemMsg, eventId))
               {
                  messageConsumed = true;
                  return true;
               }
            }
         }
         else if (mStateMachine.HandleEventMapping(msg, eventId))
         {
            messageConsumed = true;
            return true;
         }
         return false;
      }

      StateMachineType mStateMachine;
      DataBindingMessageHandler mDataBindingMessageHandler;
      hmibase::app::base::ITraceQueryHandler& traceQueryHandler;
}; // class VisualStateGlueLayer
/// @}
} // namespace Cit/VisualStateGlueLayer
#endif
