/* ***************************************************************************************
* FILE:          ViewScene.hpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewScene.hpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef HMIBASE_VIEWSCENE_HPP
#define HMIBASE_VIEWSCENE_HPP

#include <hmibase/util/Macros.h>
#include <View/CGI/InputHandling/SurfaceInputRegionManager.h>
#include <BaseContract/generated/BaseTouchSessionMsgs.h>

#include "hmi_trace_if.h"

namespace Candera {
namespace Globalization {
class Culture;
}


}

namespace hmibase {
namespace input {
class SurfaceInputRegionManager;
}


namespace view {

template <typename TBase, typename TCamera>
class ViewSceneBase : public TBase
{
      typedef TBase Base;

   public:
      ViewSceneBase(bool managed = false);
      virtual ~ViewSceneBase();

      void UseDirtyRect(bool useDirtyRect);
      bool IsDirtyRectUsed() const
      {
         return _useDirtyRect;
      }

      void ProcessTouchSessionEventsInOffscreen(bool flag);
      bool IsProcessTouchSessionEventsInOffscreen() const
      {
         return _processTouchSessionEventsInOffscreen;
      }

      void EnableInactiveViewUpdate(bool enable);
      bool IsInactiveViewUpdateEnabled() const
      {
         return _inactiveViewUpdateEnabled;
      }

      /* a view is independent if the content rendered for it is not affected by the content rendered for other views, also there is no invalidation dependency.
      * this feature is required for wait animations rendered into separate views and separate window surfaces where we need max fps with min cpu load. */
      void EnableIndependentUpdate(bool enable);
      bool IsIndependentUpdateEnabled() const
      {
         return _independentUpdateEnabled;
      }

      void SetEntireViewportTouchable(bool touchable);
      bool IsEntireViewportTouchable() const
      {
         return _isEntireViewportTouchable;
      }

      /* returns true if this view scene is being blurred */
      virtual bool IsBlurEnabled()
      {
         return false;
      }

      virtual void Update(Courier::RenderHint* renderHint);

      virtual void Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
      virtual void Invalidate(Courier::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
      virtual void Invalidate(TCamera* camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
      virtual void Invalidate(TCamera* camera, Courier::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());
      virtual void Invalidate(Candera::RenderTarget* renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea = FeatStd::Optional<Candera::Rectangle>());

      virtual bool DistributeMessage(const Courier::Message& msg);
      virtual bool DistributeMessageDirect(const Courier::Message& msg)
      {
         return Base::DistributeMessage(msg);
      }

   protected:
      virtual void ActivateImpl(bool activate);

   private:
      bool _useDirtyRect;
      bool _inactiveViewUpdateEnabled;
      bool _independentUpdateEnabled;
      bool _isEntireViewportTouchable;
      bool _processTouchSessionEventsInOffscreen;
      Candera::Globalization::Culture* _culture;
};


/*****************************************************************************/
template<typename TBase, typename TCamera>
ViewSceneBase<TBase, TCamera>::ViewSceneBase(bool managed) : Base(managed),
   _inactiveViewUpdateEnabled(true/*preserve default behavior from Courier*/),
   _independentUpdateEnabled(false),
   _isEntireViewportTouchable(true),
   _processTouchSessionEventsInOffscreen(true)
{
   _useDirtyRect = view::Widget::isDirtyRectangleEnabled();
   _culture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance();

   hmibase::input::SurfaceInputRegionManager::getInstance().registerView(this);
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
ViewSceneBase<TBase, TCamera>::~ViewSceneBase()
{
   hmibase::input::SurfaceInputRegionManager::getInstance().unregisterView(this);
   _culture = NULL;
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::UseDirtyRect(bool useDirtyRect)
{
   if (_useDirtyRect != useDirtyRect)
   {
      HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_4, TR_CLASS_HMI_FW_VIEW, "change dirty rectangle to %d for scene %s", useDirtyRect, Base::GetId().CStr());
      _useDirtyRect = useDirtyRect;
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::EnableInactiveViewUpdate(bool update)
{
   _inactiveViewUpdateEnabled = update;
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::EnableIndependentUpdate(bool enable)
{
   if (_independentUpdateEnabled != enable)
   {
      HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_FW_VIEW, "EnableIndependentUpdate enable=%u for view '%s'", enable, Base::GetId().CStr());
      _independentUpdateEnabled = enable;
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::ProcessTouchSessionEventsInOffscreen(bool flag)
{
   _processTouchSessionEventsInOffscreen = flag;
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::SetEntireViewportTouchable(bool touchable)
{
   if (_isEntireViewportTouchable != touchable)
   {
      HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_FW_VIEW, "SetEntireViewportTouchable touchable=%u for view '%s'", touchable, Base::GetId().CStr());
      _isEntireViewportTouchable = touchable;
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::ActivateImpl(bool activate)
{
   HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_FW_VIEW, "ActivateImpl activate=%u for view '%s'", activate, Base::GetId().CStr());

   Base::ActivateImpl(activate);

   //detect culture changes while the view was inactive
   if (activate && (_culture != NULL) && (_culture != Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()))
   {
      HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_1, TR_CLASS_HMI_FW_VIEW, "ActivateImpl detected culture change while the view '%s' was inactive", Base::GetId().CStr());

      _culture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance();

      Courier::Message* cultureChangedMsg = (COURIER_MESSAGE_NEW(Courier::CultureChangeIndMsg)(Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture()));
      if (cultureChangedMsg != NULL)
      {
         Courier::MessageReferrer msgRef(cultureChangedMsg);
         DistributeMessage(*cultureChangedMsg);
      }
   }

   if (activate)
   {
      hmibase::input::SurfaceInputRegionManager::getInstance().onViewActivated(this, _isEntireViewportTouchable);
   }
   else
   {
      hmibase::input::SurfaceInputRegionManager::getInstance().onViewDeactivated(this);
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::Update(Courier::RenderHint* renderHint)
{
   hmibase::input::SurfaceInputRegionManager::getInstance().clearTouchableWidgets(this);

   if (Base::IsActive() || IsInactiveViewUpdateEnabled())
   {
      HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_4, TR_CLASS_HMI_FW_VIEW, "Update view '%s'", Base::GetId().CStr());
      Base::Update(renderHint);
   }
   else
   {
      HMIBASE_TRACE_VARG_CLS(ETG_LEVEL_USER_4, TR_CLASS_HMI_FW_VIEW, "Update of inactive view '%s' is not enabled", Base::GetId().CStr());
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if (_useDirtyRect)
   {
      Base::Invalidate(dirtyArea);
   }
   else
   {
      Base::Invalidate();
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::Invalidate(Courier::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if (_useDirtyRect)
   {
      Base::Invalidate(renderCounter, dirtyArea);
   }
   else
   {
      Base::Invalidate(renderCounter);
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::Invalidate(TCamera* camera, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if (_useDirtyRect)
   {
      Base::Invalidate(camera, dirtyArea);
   }
   else
   {
      Base::Invalidate(camera);
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::Invalidate(TCamera* camera, Courier::UInt8 renderCounter, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if (_useDirtyRect)
   {
      Base::Invalidate(camera, renderCounter, dirtyArea);
   }
   else
   {
      Base::Invalidate(camera, renderCounter);
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
void ViewSceneBase<TBase, TCamera>::Invalidate(Candera::RenderTarget* renderTarget, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   if (_useDirtyRect)
   {
      Base::Invalidate(renderTarget, dirtyArea);
   }
   else
   {
      Base::Invalidate(renderTarget);
   }
}


/*****************************************************************************/
template<typename TBase, typename TCamera>
bool ViewSceneBase<TBase, TCamera>::DistributeMessage(const Courier::Message& msg)
{
   bool msgConsumed = false;

   //remember the last culture change notification now when the view is active
   if (msg.GetId() == Courier::CultureChangeIndMsg::ID)
   {
      const Courier::CultureChangeIndMsg* cultureChangedMsg = Courier::message_cast<const Courier::CultureChangeIndMsg*>(&msg);
      if (cultureChangedMsg != NULL)
      {
         _culture = cultureChangedMsg->GetCulturePtr().GetPointerToSharedInstance();
      }
   }

   bool isTouchSessionEvent = (msg.GetId() == hmibase::input::TouchSessionStartEvent::ID) || (msg.GetId() == hmibase::input::TouchSessionStopEvent::ID);

   if (IsProcessTouchSessionEventsInOffscreen() || !isTouchSessionEvent)
   {
      msgConsumed = DistributeMessageDirect(msg);
   }

   return msgConsumed;
}


}
}


#endif // HMIBASE_VIEWSCENE_HPP
