/* ***************************************************************************************
* FILE:          ViewProfilehandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewProfilehandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "ViewProfilehandler.h"


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_VIEW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ViewProfilehandler.cpp.trc.h"
#endif

namespace hmibase {
namespace view {

unsigned int ViewProfilehandler::s_currentprofile = 0;

hmibase::view::IViewMap* ViewProfilehandler::s_viewMap = 0;

ViewProfilehandler::ViewProfilehandler(Courier::ViewHandler* viewhandler, HMIRenderViewVisitor* hmirenderviewvisitor) :
   _viewhandler(viewhandler),
   _hmirenderviewvisitor(hmirenderviewvisitor)
{
   _sceneTransitionMediator.init(_viewhandler);
}


ViewProfilehandler::ViewProfilehandler() :
   _viewhandler(0),
   _hmirenderviewvisitor(0)
{
}


ViewProfilehandler::~ViewProfilehandler()
{
   _viewhandler = 0;
   _hmirenderviewvisitor = 0;
}


bool ViewProfilehandler::onCourierMessage(const ExtendedViewReqMsg& msg)
{
   const Courier::ViewId profileviewid(s_viewMap != 0 ? s_viewMap->ViewIdForProfileID(msg.GetViewId(), s_currentprofile) : msg.GetViewId());

   if (msg.GetCommand() == hmibase::views::CreateAndShow)
   {
      return (ShowView(profileviewid));
   }
   else if (msg.GetCommand() == hmibase::views::HideAndDestroy)
   {
      return (HideView(profileviewid));
   }
   else if (msg.GetCommand() == hmibase::views::TransitionEnter)
   {
      _sceneTransitionMediator.enter(profileviewid, msg.GetHint());
   }
   else if (msg.GetCommand() == hmibase::views::TransitionLeave)
   {
      _sceneTransitionMediator.leave(profileviewid, msg.GetHint());
   }
   return false;
}


bool ViewProfilehandler::onCourierMessage(const ProfileSwitchMsg& msg)
{
   if (s_viewMap == 0)
   {
      ETG_TRACE_ERR_THR(("onProfileSwitchMsg() View map is invalid. Check scene mapping!"));
      return false;
   }

   /* find list of active views from HMIRenderView Visitor and check if views available for corresponding profileID and active view
    * if available, deactivate / destroy current active view and create / activate new profile view
    */
   s_currentprofile = msg.GetProfile();

   if (_hmirenderviewvisitor)
   {
      const HMIRenderViewVisitor::tViewInfoVector& viewInfo = _hmirenderviewvisitor->getActiveViewInfo();

      for (HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfo.begin(); cit != viewInfo.end(); ++cit)
      {
         const Courier::ViewId viewid((const char*)cit->viewName.c_str());
         const Courier::ViewId profileviewid(s_viewMap->ViewIdForProfileID(viewid, msg.GetProfile()));
         if (viewid != profileviewid)
         {
            (void)ShowView(profileviewid);
            (void)HideView(viewid);
         }
      }
   }

   return true;
}


bool ViewProfilehandler::ShowView(const Courier::ViewId& viewID)
{
   bool consumed = false;
   Courier::ViewFacade viewFacade;
   if ((0 != _viewhandler) && viewFacade.Init(_viewhandler))
   {
      Courier::ViewReqMsg viewmsg(Courier::ViewAction::Create, viewID, true, true);
      consumed = viewFacade.OnViewComponentMessage(viewmsg);

      Courier::ActivationReqMsg activatemsg(viewID, true, true);
      consumed = consumed && viewFacade.OnViewComponentMessage(activatemsg);
   }
   return consumed;
}


bool ViewProfilehandler::HideView(const Courier::ViewId& viewID)
{
   bool consumed = false;
   Courier::ViewFacade viewFacade;
   if ((0 != _viewhandler) && viewFacade.Init(_viewhandler))
   {
      //Courier::ActivationReqMsg activatemsg(viewID, false, false);
      //consumed = viewFacade.OnViewComponentMessage(activatemsg);
      Courier::ViewReqMsg viewmsg(Courier::ViewAction::Destroy, viewID, false, false);
      consumed = viewFacade.OnViewComponentMessage(viewmsg);
   }
   return consumed;
}


unsigned int getCurrentViewProfile()
{
   return ViewProfilehandler::s_currentprofile;
}


}
}
