/* ***************************************************************************************
* FILE:          ViewInfo.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewInfo.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef HMIBASE_VIEW_INFO_H
#define HMIBASE_VIEW_INFO_H

namespace Courier {
class ViewController;
}


namespace hmibase {
namespace view {

/*****************************************************************************/
struct ViewInfo
{
   /*****************************************************************************/
   FeatStd::UInt32 NameHash;
   const FeatStd::Char* Name;
   bool Is2D;
   Courier::ViewController* (*CreateControllerFunc)(const Courier::ViewId&);
   FeatStd::Int32 UserData;

   /*****************************************************************************/
   inline Courier::ViewId GetViewId() const
   {
      return Courier::ViewId(Name);
   }

   /*****************************************************************************/
   static FeatStd::Int Compare(FeatStd::UInt32 nameHash, const ViewInfo& viewInfo)
   {
      return (nameHash == viewInfo.NameHash) ? 0 : ((nameHash < viewInfo.NameHash) ? -1 : 1);
   }
};


/*****************************************************************************/
class ViewInfoManager
{
   public:
      /*****************************************************************************/
      typedef const ViewInfo* (*ViewInfoFunc)(FeatStd::UInt32 viewNameHash);

      static bool Initialize(ViewInfoFunc viewInfoFunc);
      static bool IsInitialized();

      /*****************************************************************************/
      static Courier::ViewId GetViewId(const FeatStd::Char* viewName);
      static Courier::ViewId GetViewId(FeatStd::UInt32 viewNameHash);

      /*****************************************************************************/
      static const ViewInfo* GetViewInfo(const FeatStd::Char* viewName);
      static const ViewInfo* GetViewInfo(FeatStd::UInt32 viewNameHash);
      static const ViewInfo* GetViewInfo(const Courier::ViewId& viewId);

   private:
      /*****************************************************************************/
      static ViewInfoFunc& GetViewInfoFunc()
      {
         static ViewInfoFunc _viewInfoFunc;
         return _viewInfoFunc;
      }
};


/*****************************************************************************/
typedef ViewControllerBase DefaultViewController;

Courier::ViewController* CreateDefaultViewController(const Courier::ViewId&);

template <typename TController> Courier::ViewController* CreateViewController(const Courier::ViewId&)
{
   return FEATSTD_NEW(TController);
}


}
}


/*****************************************************************************/
#define HMIBASE_VIEW_DEFAULT_VIEW_CONTROLLER_FUNC ::hmibase::view::CreateDefaultViewController

/*****************************************************************************/
#define HMIBASE_VIEW_INFO_BEGIN(viewCount)\
namespace hmibase { namespace view { namespace generated {\
static const ::hmibase::view::ViewInfo* GetViewInfo(::FeatStd::UInt32 viewNameHash) {\
static ::hmibase::view::ViewInfo _viewInfo[viewCount + 1] = {\
{ 0, "", false, &::hmibase::view::CreateViewController< ::hmibase::view::DefaultViewController>, 0 },


/*****************************************************************************/
#define HMIBASE_VIEW_INFO_END()\
};\
const ::hmibase::view::ViewInfo* viewInfo = ::Courier::BinarySearch(viewNameHash, _viewInfo, sizeof(_viewInfo) / sizeof(_viewInfo[0]), &::hmibase::view::ViewInfo::Compare);\
return ((viewInfo != NULL) && (viewNameHash == viewInfo->NameHash)) ? viewInfo : NULL;\
}\
static bool _initViewInfoManager = ::hmibase::view::ViewInfoManager::Initialize(&GetViewInfo);\
}}}//namespace


/*****************************************************************************/
#define HMIBASE_VIEW_INFO_2D(viewClass, userData) \
{ ViewId_##viewClass, ViewName_##viewClass, true, &ViewControllerFunc_##viewClass, userData }

/*****************************************************************************/
#define HMIBASE_VIEW_INFO_3D(viewClass, userData) \
{ ViewId_##viewClass, ViewName_##viewClass, false, &ViewControllerFunc_##viewClass, userData }

#endif
