/* ***************************************************************************************
* FILE:          ViewFactory.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewFactory.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "ViewFactory.h"
#include "ViewScene2D.h"
#include "ViewScene3D.h"
#include "ViewInfo.h"
#include <View/CGI/CgiExtensions/AppViewHandler.h>

#include <hmi_trace_if.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_VIEW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ViewFactory.cpp.trc.h"
#endif

namespace hmibase {
namespace view {


/************************************************************************/
ViewFactory::CreateDefaultViewSceneFunc ViewFactory::_createDefaultViewSceneFunc = NULL;


/************************************************************************/
Courier::View* ViewFactory::Create(const FeatStd::Char* viewName)
{
   const ViewInfo* viewInfo = ViewInfoManager::GetViewInfo(viewName);
   if (viewInfo != NULL)
   {
      Courier::View* view = Create(*viewInfo);

      // since 3.6, Courier expects the init method to be called for the newly created view otherwise it searches the asset for its name
      if (view != NULL)
      {
         view->Init(&AppViewHandler::getInstance(), viewInfo->Name);
      }

      return view;
   }

   return Base::Create(viewName);
}


/************************************************************************/
void ViewFactory::Destroy(Courier::View* view)
{
   // destroy only the views for which we have info
   if ((view != NULL) && (ViewInfoManager::GetViewInfo(view->GetId()) != NULL))
   {
      FEATSTD_DELETE(view);
   }
   else
   {
      Base::Destroy(view);
   }
}


/************************************************************************/
Courier::ViewScene* ViewFactory::CreateViewScene(bool is2D)
{
   return CreateDefaultViewScene(is2D, true);
}


/************************************************************************/
Courier::View* ViewFactory::Create(const ViewInfo& viewInfo)
{
   ETG_TRACE_USR1_THR(("ViewFactory::Create nameHash=0x%x type=%2s userData=%u name=%s", viewInfo.NameHash, viewInfo.Is2D ? "2D" : "3D", viewInfo.UserData, viewInfo.Name));

   return CreateDefaultViewScene(viewInfo.Is2D, false);
}


/************************************************************************/
void ViewFactory::SetCreateDefaultViewSceneFunc(CreateDefaultViewSceneFunc func)
{
   _createDefaultViewSceneFunc = func;
}


/************************************************************************/
Courier::ViewScene* ViewFactory::CreateDefaultViewScene(bool is2D, bool isManagedByViewHandler)
{
   Courier::ViewScene* view = NULL;

   // use the custom method if available
   if (_createDefaultViewSceneFunc != NULL)
   {
      ETG_TRACE_USR4_THR(("ViewFactory::CreateDefaultViewScene type=%uD (custom create method), managedByViewHandler=%u", is2D ? 2 : 3, isManagedByViewHandler));
      view = _createDefaultViewSceneFunc(is2D, isManagedByViewHandler);
   }
   else
   {
      if (is2D)
      {
         ETG_TRACE_USR4_THR(("ViewFactory::CreateDefaultViewScene type=2D (hmibase::view::ViewScene2D), managedByViewHandler=%u", isManagedByViewHandler));
         view = FEATSTD_NEW(hmibase::view::ViewScene2D)(isManagedByViewHandler);
      }
      else
      {
         ETG_TRACE_USR4_THR(("ViewFactory::CreateDefaultViewScene type=3D (hmibase::view::ViewScene3D), managedByViewHandler=%u", isManagedByViewHandler));
         view = FEATSTD_NEW(hmibase::view::ViewScene3D)(isManagedByViewHandler);
      }
   }

   return view;
}


/************************************************************************/
Courier::ViewController* ViewControllerFactory::Create(const FeatStd::Char* viewName)
{
   const hmibase::view::ViewInfo* viewInfo = hmibase::view::ViewInfoManager::GetViewInfo(viewName);
   if (viewInfo != NULL)
   {
      return Create(*viewInfo);
   }

   return Base::Create(viewName);
}


/************************************************************************/
Courier::ViewController* ViewControllerFactory::Create(const ViewInfo& viewInfo)
{
   ETG_TRACE_USR1_THR(("ViewControllerFactory::Create nameHash=0x%x type=%2s userData=%u name=%s", viewInfo.NameHash, viewInfo.Is2D ? "2D" : "3D", viewInfo.UserData, viewInfo.Name));

   return viewInfo.CreateControllerFunc != NULL ? viewInfo.CreateControllerFunc(viewInfo.GetViewId()) : NULL;
}


/************************************************************************/
void ViewControllerFactory::Destroy(Courier::ViewController* viewController)
{
   if (viewController != NULL)
   {
      FEATSTD_DELETE(viewController);
   }
}


}
}
