/* ***************************************************************************************
* FILE:          ViewAccessor.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewAccessor.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "ViewAccessor.h"
#include "AppViewHandler.h"

#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>

#include <Courier/Visualization/ViewScene2D.h>
#include <Courier/Visualization/ViewScene3D.h>

#include <Trace/ToString.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_VIEW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ViewAccessor.cpp.trc.h"
#endif


namespace hmibase {
namespace view {

/************************************************************************/
ViewAccessor::ViewAccessor(Courier::View* view) : _view(view)
{
   if (view != NULL)
   {
      _viewId = view->GetId();
   }
   else
   {
      ETG_TRACE_USR4_THR(("ViewAccessor view is null!"));
   }
}


/************************************************************************/
ViewAccessor::ViewAccessor(const Courier::ViewId& viewId) : _view(NULL), _viewId(viewId)
{
   if (viewId != Courier::ViewId())
   {
      _view = AppViewHandler::getInstance().FindView(viewId);
      if (_view == NULL)
      {
         ETG_TRACE_USR4_THR(("ViewAccessor view(%s) not found!", HMIBASE_TO_STRING(_viewId)));
      }
   }
   else
   {
      ETG_TRACE_USR4_THR(("ViewAccessor view id is invalid!"));
   }
}


/************************************************************************/
ViewAccessor::~ViewAccessor()
{
   _view = NULL;
}


/************************************************************************/
Courier::View* ViewAccessor::GetView() const
{
   return _view;
}


/************************************************************************/
const Courier::ViewId& ViewAccessor::GetViewId() const
{
   return _viewId;
}


/************************************************************************/
bool ViewAccessor::CreateView()
{
   if (_view == NULL)
   {
      ETG_TRACE_USR1_THR(("CreateView viewId=%s", HMIBASE_TO_STRING(_viewId)));

      if (_viewId == Courier::ViewId())
      {
         ETG_TRACE_ERR_THR(("CreateView view id is invalid!"));
      }
      else if (AppViewHandler::getInstance().ExecuteViewAction(Courier::ViewAction::Create, _viewId, true, true))
      {
         _view = AppViewHandler::getInstance().FindView(_viewId);
         if (_view == NULL)
         {
            ETG_TRACE_ERR_THR(("CreateView view(%s) not found!", HMIBASE_TO_STRING(_viewId)));
         }
      }
      else
      {
         ETG_TRACE_ERR_THR(("CreateView failed to create view(%s)!", HMIBASE_TO_STRING(_viewId)));
      }
   }
   return (_view != NULL);
}


/************************************************************************/
void ViewAccessor::DestroyView()
{
   if (_view != NULL)
   {
      ETG_TRACE_USR1_THR(("DestroyView viewId=%s", HMIBASE_TO_STRING(_viewId)));

      AppViewHandler::getInstance().ExecuteViewAction(Courier::ViewAction::Destroy, _viewId, false, false);
      _view = NULL;
   }
}


/************************************************************************/
void ViewAccessor::ShowView()
{
   if (_view != NULL)
   {
      ETG_TRACE_USR1_THR(("ShowView viewId=%s", HMIBASE_TO_STRING(_viewId)));

      _view->Activate(true);
      _view->EnableRendering(true);
      _view->Invalidate(FeatStd::Optional<Candera::Rectangle>());
   }
   else
   {
      ETG_TRACE_ERR_THR(("ShowView view is null!"));
   }
}


/************************************************************************/
void ViewAccessor::HideView()
{
   if (_view != NULL)
   {
      ETG_TRACE_USR1_THR(("HideView viewId=%s", HMIBASE_TO_STRING(_viewId)));

      _view->Activate(false);
      _view->EnableRendering(false);
   }
   else
   {
      ETG_TRACE_ERR_THR(("HideView view is null!"));
   }
}


/************************************************************************/
Candera::GraphicDeviceUnit* ViewAccessor::FindTextureRT(const FeatStd::Char* nodeName) const
{
   if (_view != NULL)
   {
      //2D
      if (_view->ToViewScene2D() != NULL)
      {
         return GetTextureRT(FindNode(_view->ToViewScene2D(), nodeName));
      }

      //3D
      if (_view->ToViewScene3D() != NULL)
      {
         return GetTextureRT(FindNode(_view->ToViewScene3D(), nodeName));
      }
   }

   return NULL;
}


/************************************************************************/
FeatStd::SizeType ViewAccessor::GetCameraCount() const
{
   if (_view != NULL)
   {
      //2D
      if (_view->ToViewScene2D() != NULL)
      {
         return _view->ToViewScene2D()->GetCameraPtrVector().Size();
      }

      //3D
      if (_view->ToViewScene3D() != NULL)
      {
         return _view->ToViewScene3D()->GetCameraPtrVector().Size();
      }
   }
   return 0;
}


/************************************************************************/
Candera::CanderaObject* ViewAccessor::GetCamera(FeatStd::SizeType index) const
{
   if (_view != NULL)
   {
      //2D
      if ((_view->ToViewScene2D() != NULL) && (_view->ToViewScene2D()->GetCameraPtrVector().Size() > index))
      {
         return _view->ToViewScene2D()->GetCameraPtrVector()[index];
      }

      //3D
      if ((_view->ToViewScene3D() != NULL) && (_view->ToViewScene3D()->GetCameraPtrVector().Size() > index))
      {
         return _view->ToViewScene3D()->GetCameraPtrVector()[index];
      }
   }

   return NULL;
}


/************************************************************************/
FeatStd::Optional<Candera::GraphicDeviceUnit*> ViewAccessor::SetCameraRT(Candera::CanderaObject* camera, Candera::GraphicDeviceUnit* gdu)
{
   Candera::Camera2D* camera2D = Candera::Dynamic_Cast<Candera::Camera2D*>(camera);
   if (camera2D != NULL)
   {
      Candera::GraphicDeviceUnit* oldGdu = (camera2D->GetRenderTarget() != NULL) ? camera2D->GetRenderTarget()->GetGraphicDeviceUnit() : NULL;
      Candera::RenderTarget2D* rt2D = (gdu != NULL) ? gdu->ToRenderTarget2D() : NULL;

      ETG_TRACE_USR1_THR(("SetCameraRT rt=%p gdu=%50s %s", rt2D,
                          HMIBASE_TO_STRING_GDU(gdu),
                          HMIBASE_TO_STRING_SPN2D(camera2D)));
      camera2D->SetRenderTarget(rt2D);
      return FeatStd::Optional<Candera::GraphicDeviceUnit*>(oldGdu);
   }

   Candera::Camera* camera3D = Candera::Dynamic_Cast<Candera::Camera*>(camera);
   if (camera3D != NULL)
   {
      Candera::GraphicDeviceUnit* oldGdu = (camera3D->GetRenderTarget() != NULL) ? camera3D->GetRenderTarget()->GetGraphicDeviceUnit() : NULL;
      Candera::RenderTarget3D* rt3D = (gdu != NULL) ? gdu->ToRenderTarget3D() : NULL;

      ETG_TRACE_USR1_THR(("SetCameraRT rt=%p gdu=%50s %s", rt3D,
                          HMIBASE_TO_STRING_GDU(gdu),
                          HMIBASE_TO_STRING_SPN3D(camera3D)));
      camera3D->SetRenderTarget(rt3D);
      return FeatStd::Optional<Candera::GraphicDeviceUnit*>(oldGdu);
   }

   ETG_TRACE_USR1_THR(("SetCameraRT invalid camera %s!", camera != NULL ? camera->GetName() : "null"));
   return FeatStd::Optional<Candera::GraphicDeviceUnit*>();
}


/************************************************************************/
Candera::Node2D* ViewAccessor::FindNode(Courier::ViewScene2D* viewScene, const FeatStd::Char* nodeName)
{
   Candera::Node2D* node = NULL;
   if ((nodeName != NULL) && (viewScene != NULL)
         && (viewScene->GetScene2DContext() != NULL)
         && (viewScene->GetScene2DContext()->GetScene() != NULL))
   {
      node = viewScene->GetScene2DContext()->GetScene()->GetDescendant(nodeName);
   }

   ETG_TRACE_USR4_THR(("FindNode nodeName=%50s view=%75s node=%s", nodeName,
                       HMIBASE_TO_STRING_V(viewScene),
                       HMIBASE_TO_STRING_PN2D(node)));

   return node;
}


/************************************************************************/
Candera::GraphicDeviceUnit* ViewAccessor::GetTextureRT(Candera::Node2D* node)
{
   Candera::GraphicDeviceUnit* gdu = NULL;
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(node);
   if (renderNode != NULL)
   {
      FeatStd::UInt8 index = 0;
      Candera::Effect2D* effect = renderNode->GetEffect(index);
      while ((gdu == NULL) && (effect != NULL))
      {
         Candera::BitmapBrush* brush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(effect->GetBrushEffect2D());
         if (brush != NULL)
         {
            Candera::ProxyImage2D* proxyImage = dynamic_cast<Candera::ProxyImage2D*>(brush->Image().Get());
            if ((proxyImage != NULL) && (proxyImage->GetImageSource() != NULL))
            {
               gdu = proxyImage->GetImageSource()->GetGraphicDeviceUnit();
            }
         }
         ++index;
         effect = renderNode->GetEffect(index);
      }
   }

   ETG_TRACE_USR1_THR(("GetTextureRT gdu=%50s node=%s",
                       HMIBASE_TO_STRING_GDU(gdu),
                       HMIBASE_TO_STRING_SPN2D(node)));
   return gdu;
}


/************************************************************************/
Candera::Node* ViewAccessor::FindNode(Courier::ViewScene3D* viewScene, const FeatStd::Char* nodeName)
{
   Candera::Node* node = NULL;
   if ((nodeName != NULL) && (viewScene != NULL)
         && (viewScene->GetScene3DContext() != NULL)
         && (viewScene->GetScene3DContext()->GetScene() != NULL))
   {
      node = viewScene->GetScene3DContext()->GetScene()->GetDescendant(nodeName);
   }

   ETG_TRACE_USR4_THR(("FindNode nodeName=%50s view=%75s node=%s", nodeName,
                       HMIBASE_TO_STRING_V(viewScene),
                       HMIBASE_TO_STRING_PN3D(node)));

   return node;
}


/************************************************************************/
Candera::GraphicDeviceUnit* ViewAccessor::GetTextureRT(Candera::Node* node)
{
   Candera::GraphicDeviceUnit* gdu = NULL;
   if ((node != NULL) && (!node->GetAppearance().PointsToNull()))
   {
      FeatStd::UInt index = 0;
      Candera::Texture* texture = node->GetAppearance()->GetTexture(index).GetPointerToSharedInstance();
      while ((gdu == NULL) && (texture != NULL))
      {
         if (!texture->GetTextureImage().PointsToNull())
         {
            Candera::ImageSource3D* imageSource = texture->GetTextureImage()->ToImageSource3D();
            if (imageSource != NULL)
            {
               gdu = imageSource->GetGraphicDeviceUnit();
            }
         }
         ++index;
         texture = node->GetAppearance()->GetTexture(index).GetPointerToSharedInstance();
      }
   }

   ETG_TRACE_USR1_THR(("GetTextureRT gdu=%50s node=%s",
                       HMIBASE_TO_STRING_GDU(gdu),
                       HMIBASE_TO_STRING_SPN3D(node)));
   return gdu;
}


}
}
