/* ***************************************************************************************
* FILE:          TouchInput.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TouchInput.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "TouchInput.h"

#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>
#include <Candera/Engine2D/Core/Camera2D.h>

#include "View/CGI/Widget/Widget.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TouchInput.cpp.trc.h"
#endif

using namespace Courier;

namespace hmibase {
namespace input {

class Camera2DDynamicProperties
{
   public:
      static bool ClearUseForTouch(Candera::DynamicProperties::DynamicPropertyHost& object)
      {
         return object.ClearValue(CdaDynamicPropertyInstance(UseForTouch));
      }

      static bool SetUseForTouch(Candera::DynamicProperties::DynamicPropertyHost& object, bool value)
      {
         return object.SetValue(CdaDynamicPropertyInstance(UseForTouch), value);
      }

      static bool GetUseForTouch(const Candera::DynamicProperties::DynamicPropertyHost& object)
      {
         return object.GetValue(CdaDynamicPropertyInstance(UseForTouch));
      }

      static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost*)
      {
         return NULL;
      }

   private:
      CdaDynamicProperties(Camera2DDynamicProperties, Candera::Camera2D);

      CdaDynamicProperty(UseForTouch, bool);
      CdaDynamicPropertyDefaultValue(true);
      CdaDynamicPropertyEnd();

      CdaDynamicPropertiesEnd();
};


bool ShouldUseCameraForTouch(const Candera::Camera2D* camera)
{
   return (camera != NULL) ? Camera2DDynamicProperties::GetUseForTouch(*camera) : false;
}


void SetUseCameraForTouch(Candera::Camera2D* camera, bool value)
{
   ETG_TRACE_USR1_THR(("[%25s] SetUseCameraForTouch camera=%p.%50s value=%u",
                       hmibase::trace::getAppName().c_str(), camera, (camera != NULL) && (camera->GetName() != NULL) ? camera->GetName() : "null", value));

   if (camera != NULL)
   {
      Camera2DDynamicProperties::SetUseForTouch(*camera, value);
   }
}


/*
 * FEAT's patch function for finding widgets filtered based on surface id
 */
bool EqualsSurfaceId(const Candera::Surface* surface, FeatStd::SizeType surfaceId)
{
   if (0 == surface)
   {
      return false;
   }

   Candera::GraphicDeviceUnit* gdu = surface->GetGraphicDeviceUnit();
   if (0 == gdu)
   {
      return false;
   }
   const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(gdu->GetUnitType());
   if (0 == metaInfo)
   {
      return false;
   }
   Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
   if (0 == propMetaInfo)
   {
      // the following code was received from SESA and should fix the touch not working for blurred cameras
      // if no SurfaceId is available, query for ownerGDU and ask for its SurfaceId
      Candera::Internal::GraphicDeviceUnitOwnerAccess gduAccess(gdu);
      Candera::GraphicDeviceUnit* ownerGdu = gduAccess.GetGraphicDeviceUnitOwner();
      if (0 != ownerGdu)
      {
         Candera::RenderTarget3D* ownerRT = ownerGdu->ToRenderTarget3D();
         if ((0 != ownerRT) && (surface != ownerRT))
         {
            return EqualsSurfaceId(ownerRT, surfaceId);
         }
      }

      return surfaceId == 0;
   }
   FeatStd::SizeType id = 0;
   Char buffer[32];
   buffer[0] = '\0';
   if (!propMetaInfo->Get(gdu, buffer, 32))
   {
      return false;
   }
#ifdef WIN32
   ::sscanf(buffer, "%Iu", &id);
#else
   ::sscanf(buffer, "%zu", &id);
#endif
   return surfaceId == id;
}


/**
 * Check for virtual surface id and the occurance of a DirectTextureProviderWidget
 */
bool CheckForVirtualSurfaceTouch(Courier::ViewScene* view, const Courier::TouchInfo& info)
{
   // check for direct touch assignment if source is not a surface
   bool isValid = false;
   const ViewScene::FrameworkWidgetPtrVector& widgets = view->GetFrameworkWidgetPtrVector();

   if (info.mSourceId >= static_cast<UInt32>(SURFACEID_VIRTUAL_SURFACE_OFFSET))
   {
      for (FeatStd::SizeType i = 0; i < widgets.Size() && !isValid; ++i)
      {
         // find matching DirectTextureInstance
         FrameworkWidget* fw = widgets[i];
         hmibase::widget::Widget* wb = static_cast<hmibase::widget::Widget*>(fw);

         isValid = wb->IsVirtuallyTouched(static_cast<FeatStd::UInt32>(info.mSourceId - SURFACEID_VIRTUAL_SURFACE_OFFSET));
      }
   }

   return isValid;
}


/*
 * FEAT's patch function for finding 2D widgets filtered based on surface id
 */

Courier::FrameworkWidget* ScreenIdFilteredGetTouchedWidget2D(Courier::ViewScene2D* view2D, const Courier::TouchInfo& info)
{
   if (view2D != 0 && view2D->IsActive())
   {
      const ViewScene::FrameworkWidgetPtrVector& widgets = view2D->GetFrameworkWidgetPtrVector();
      const ViewScene2D::CameraPtrVector& cameras = view2D->GetCameraPtrVector();

      // check for direct touch assignment if source is not a surface
      bool isVirtualSurfaceTouched = CheckForVirtualSurfaceTouch(view2D, info);

      for (FeatStd::SizeType i = 0; i < widgets.Size(); ++i)
      {
         FrameworkWidget* fw = widgets[i];

         if (fw->IsTouchable())
         {
            Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(fw);
            Candera::Widget2D* w = Candera::Dynamic_Cast<Candera::Widget2D*>(wb);
            if (w != 0)
            {
               Candera::Node2D* node = w->GetNode();
               if (node != 0)
               {
                  for (FeatStd::SizeType idx = 0; idx < cameras.Size(); idx++)
                  {
                     bool isSurfaceTouched = false;
                     Candera::Camera2D* camera2D = cameras[idx];

                     if (!isVirtualSurfaceTouched)
                     {
                        isSurfaceTouched = EqualsSurfaceId(camera2D->GetRenderTarget(), info.mSourceId);
                     }

                     if (camera2D->IsRenderingEnabled() && ShouldUseCameraForTouch(camera2D) && isSurfaceTouched)
                     {
                        Candera::Widget2D* widget = static_cast<Candera::Widget2D*>(fw);
                        //Touchable2D* touchableWidget = Candera::Dynamic_Cast<Touchable2D*>(widget);
                        const Candera::Vector2 point((Courier::Float)info.mX, (Courier::Float)info.mY);
                        if (widget != 0)
                        {
                           if (widget->OnTouchGeneric(*camera2D, point))
                           {
                              return fw;
                           }
                        }
                        else if (node->IsPickIntersectingBoundingRectangle(*camera2D, point))
                        {
                           return fw;
                        }
                     }
                  }
               }
            }
         }
      }
   }
   return 0;
}


/*
 * FEAT's patch function for finding 3D widgets filtered based on surface id
 */
Courier::FrameworkWidget* ScreenIdFilteredGetTouchedWidget3D(Courier::ViewScene3D* view3D, const Courier::TouchInfo& info)
{
   FrameworkWidget* nearestWidget = 0;
   if (view3D != 0 && view3D->IsActive())
   {
      Float currentDistance = 9999999999.0f;
      Float minimalDistance = currentDistance;

      const ViewScene::FrameworkWidgetPtrVector& widgets = view3D->GetFrameworkWidgetPtrVector();
      const ViewScene3D::CameraPtrVector& cameras = view3D->GetCameraPtrVector();

      // check for direct touch assignment if source is not a surface
      bool isVirtualSurfaceTouched = CheckForVirtualSurfaceTouch(view3D, info);

      for (FeatStd::SizeType i = 0; i < widgets.Size(); ++i)
      {
         FrameworkWidget* fw = widgets[i];
         if (fw->IsTouchable())
         {
            Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(fw);
            Candera::Widget* w = Candera::Dynamic_Cast<Candera::Widget*>(wb);
            if (w != 0)
            {
               Candera::Node* node = w->GetNode();
               if (node != 0)
               {
                  for (FeatStd::SizeType idx = 0; idx < cameras.Size(); idx++)
                  {
                     bool isSurfaceTouched = false;
                     Candera::Camera* camera3D = cameras[idx];

                     if (!isVirtualSurfaceTouched)
                     {
                        isSurfaceTouched = EqualsSurfaceId(camera3D->GetRenderTarget(), info.mSourceId);
                     }

                     if (camera3D->IsRenderingEnabled() && isSurfaceTouched)
                     {
                        bool hit = false;
                        hmibase::widget::Widget* widget = static_cast<hmibase::widget::Widget*>(fw);
                        if (widget != 0)
                        {
                           if (widget->OnTouchGeneric(*camera3D, Candera::Vector2(static_cast<FeatStd::Float>(info.mX), static_cast<FeatStd::Float>(info.mY)), currentDistance))
                           {
                              hit = true;
                           }
                        }
                        // if not touchable
                        //else if (widget->GetNode()->IsPickIntersectingGeometry(*camera3D, (Courier::Int)info.mX, (Courier::Int)info.mY, currentDistance))
                        //{
                        //   hit = true;
                        //}
                        if (hit && (currentDistance < minimalDistance))
                        {
                           minimalDistance = currentDistance;
                           nearestWidget = fw;
                        }
                     }
                  }
               }
            }
         }
      }
   }
   return nearestWidget;
}


bool isInsideBoundingBox2D(hmibase::widget::Widget* widget, const Courier::TouchInfo& msg)
{
   if (widget && widget->GetParentView() != NULL)
   {
      ViewScene2D* viewScene2D = widget->GetParentView()->ToViewScene2D();
      if (viewScene2D != NULL)
      {
         const ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
         const Candera::Vector2 point(static_cast<Courier::Float>(msg.mX), static_cast<Courier::Float>(msg.mY));

         bool isVirtualSurfaceTouched = false;

         if (msg.mSourceId >= static_cast<UInt32>(SURFACEID_VIRTUAL_SURFACE_OFFSET))
         {
            isVirtualSurfaceTouched = CheckForVirtualSurfaceTouch(viewScene2D, msg);

            if (!isVirtualSurfaceTouched)
            {
               // surface is virtual, but no matching direct texture provider widget found in this view
               return false;
            }
         }

         for (ViewScene2D::CameraPtrVector::ConstIterator it = cameras.ConstBegin(); it != cameras.ConstEnd(); ++it)
         {
            Candera::Camera2D* camera2D = *it;
            if ((camera2D != NULL) && camera2D->IsRenderingEnabled() && ShouldUseCameraForTouch(camera2D))
            {
               if (isVirtualSurfaceTouched || EqualsSurfaceId(camera2D->GetRenderTarget(), msg.mSourceId))
               {
                  if (widget->OnTouchGeneric(*camera2D, point))
                  {
                     return true;
                  }
               }
            }
         }
      }
   }
   return false;
}


}
}
