/* ***************************************************************************************
* FILE:          ShaderStorage.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ShaderStorage is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmibase/sys_std_if.h"

#define USE_MD5

#include "ShaderStorage.h"
#ifdef CANDERA_SHADER_PROGRAM_PERSIST_INTERFACE_ENABLED
#include <FeatStd/Platform/File.h>
#include "FeatStd/MemoryManagement/Heap.h"

#include <hmibase/util/Semaphore.h>
#include <hmibase/util/SimpleString.h>
#include <hmibase/util/Permissions.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "hmibase/trace/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ShaderStorage.cpp.trc.h"
#endif

#ifdef WIN32
#include <io.h>
#include <direct.h>
#include <Wincrypt.h>

#define F_OK 0
#define access _access
#define umask _umask
#define mkdir(p,m) _mkdir(p)
#define mode_t int
#define PATH_MAX MAX_PATH

#define S_IRWXU   (0400|0200|0100)
#define S_IRWXG  ((0400|0200|0100) >> 3)
#define S_IROTH (((0400) >> 3) >> 3)
#define S_IXOTH (((0100) >> 3) >> 3)


std::string s_shaderStoragePath = "shader";
#else

#include "RootDaemonClient/RootDaemon.h"

#include <openssl/md5.h>
std::string s_shaderStoragePath = "/var/opt/bosch/dynamic/hmi/shader";
#endif

namespace hmibase {
namespace view {

static bool ChangeOwner(const char* path)
{
   int errCode = 0;

#ifndef WIN32
   if (path != 0)
   {
      CmdData status = HMIBASE_ROOTDAEMON_CALLER_PerformRootOp(E_HMIBASE_ROOTDAEMON_CMD_CODE_CHOWN_DYN_PARTITION, path);
      if (status.errorNo != ERR_NONE)
      {
         ETG_TRACE_ERR_THR(("ChangeOwner failed, errno %d, path %s", errno, path));
         errCode = -1;
      }
   }
#endif

   return (errCode == 0) ? true : false;
}


static bool FileExist(const char* path)
{
   return (access(path, F_OK) != -1) ? true : false;
}


static int MkdirRecursive(const char* path)
{
   char opath[PATH_MAX];
   char* p;
   size_t len;
   int ret = 0;
   mode_t mode = S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH;

   strncpy(opath, path, sizeof(opath));
   opath[sizeof(opath) - 1] = '\0';

   len = strlen(opath);
   if (len == 0)
   {
      return false;
   }
   else
   {
      if (opath[len - 1] == '/')
      {
         opath[len - 1] = '\0';
      }
   }

   // need to lock to avoid concurrent access
   hmibase::util::Semaphore sem(1, "hmibase_folder_create");
   sem.wait(100);

   mode_t old = umask(002);
   for (p = opath; *p; p++)
   {
      if (*p == '/')
      {
         *p = '\0';
         if (access(opath, F_OK))
         {
            ret = mkdir(opath, mode);
            if (ret != 0)
            {
               //
               ETG_TRACE_ERR_THR(("Creating %50s with mode o%o failed, error code %d", opath, mode, errno));
            }
            else
            {
               ChangeOwner(opath);
            }
         }
         *p = '/';
      }
   }

   if (access(opath, F_OK)) // if path is not terminated with /
   {
      ret = mkdir(opath, mode);

      if (ret != 0)
      {
         //
         ETG_TRACE_ERR_THR(("Creating %50s with mode o%o failed, error code %d", opath, mode, errno));
      }
      else
      {
         ChangeOwner(opath);
      }
   }
   umask(old);

   // unlock again
   sem.post();

   return FileExist(opath);
}


static FeatStd::Char ValueToCharacter(FeatStd::UInt8 tmp)
{
   int res = tmp;
   if (res > 9)
   {
      res += 'A' - 10;
   }
   else
   {
      res += '0';
   }
   return static_cast<FeatStd::Char>(res);
}


static void MakeHumanReadable(const FeatStd::UInt8* src, FeatStd::SizeType srcSize, FeatStd::Char* dest, FeatStd::SizeType destSize)
{
   FeatStd::SizeType j = 0;
   FeatStd::SizeType i = 0;
   while (j < srcSize && i < destSize)
   {
      switch (i)
      {
         case 8:
            dest[i] = '-';
            ++i;
            break;
         case 13:
            dest[i] = '-';
            ++i;
            break;
         case 18:
            dest[i] = '-';
            ++i;
            break;
         case 23:
            dest[i] = '-';
            ++i;
            break;
         default:
            dest[i] = ValueToCharacter((src[j] & 0xF0) >> 4);
            ++i;
            dest[i] = ValueToCharacter(src[j] & 0x0F);
            ++i;
            ++j;
            break;
      }
   }

   dest[i] = 0;
}


ShaderStorage::ShaderStorage(bool useApplicationFolder) : _useApplicationFolder(useApplicationFolder)
{
   ETG_TRACE_USR1_THR(("[%25s] ShaderStorage, application scoped %d", hmibase::trace::getAppName().c_str(), _useApplicationFolder));

   std::string folderName;
   if (useApplicationFolder)
   {
      std::string tmp = trace::getAppName().c_str();
      size_t s = tmp.find('.');

      if (s != std::string::npos)
      {
         folderName = tmp.substr(0, s);
      }
      else
      {
         folderName = tmp;
      }
   }

   if (s_shaderStoragePath == "")
   {
      // create folder in working directory
      if (useApplicationFolder)
      {
         _shaderBinaryFolder = folderName;
      }
   }
   else // (s_shaderStoragePath != "")
   {
      if (useApplicationFolder)
      {
         util::SimpleString tmp = hmibase::util::SimpleString::format("%s/%s", s_shaderStoragePath.c_str(), folderName.c_str());
         _shaderBinaryFolder = tmp.cPtr();
      }
      else
      {
         _shaderBinaryFolder = s_shaderStoragePath;
      }
   }

   if (FileExist(_shaderBinaryFolder.c_str()) == false)
   {
      MkdirRecursive(_shaderBinaryFolder.c_str());
   }
}


ShaderStorage::~ShaderStorage()
{
}


bool ShaderStorage::Lock(const char* descriptor)
{
   if (_useApplicationFolder)
   {
      // no need to lock as there will be no concurrent access from other processes
      return true;
   }

   hmibase::util::Semaphore* sem = NULL;
   hmibase::util::SimpleString ss = util::SimpleString::format("shaderbinary_%s", descriptor);
   std::string name(ss.cPtr());
   SemaphoreMap::iterator it = semaphores.find(name);
   if (it == semaphores.end())
   {
      sem = new hmibase::util::Semaphore(1, name.c_str());
      semaphores.insert(std::pair<std::string, hmibase::util::Semaphore*>(name, sem));
   }
   else
   {
      sem = it->second;
   }

   if (sem)
   {
      ETG_TRACE_USR4_THR(("[%25s] Lock %s", hmibase::trace::getAppName().c_str(), ss.cPtr()));
      if (sem->wait(1000))
      {
         ETG_TRACE_USR4_THR(("[%25s] Lock success %s", hmibase::trace::getAppName().c_str(), ss.cPtr()));
         return true;
      }
      else
      {
         ETG_TRACE_USR4_THR(("[%25s] Lock failed/timed out errno %d, %s", hmibase::trace::getAppName().c_str(), errno, ss.cPtr()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("[%25s] Lock: no semaphore for descriptor %s", hmibase::trace::getAppName().c_str(), ss.cPtr()));
   }

   return false;
}


bool ShaderStorage::Unlock(const char* descriptor)
{
   if (_useApplicationFolder)
   {
      // no need to lock as there will be no concurrent access from other processes
      return true;
   }

   hmibase::util::SimpleString ss = util::SimpleString::format("shaderbinary_%s", descriptor);
   std::string name(ss.cPtr());
   SemaphoreMap::iterator it = semaphores.find(name);
   if (it != semaphores.end())
   {
      ETG_TRACE_USR4_THR(("[%25s] UnLock %s", hmibase::trace::getAppName().c_str(), ss.cPtr()));
      if (it->second->post())
      {
         ETG_TRACE_USR4_THR(("[%25s] UnLock success %s", hmibase::trace::getAppName().c_str(), ss.cPtr()));
         return true;
      }
      else
      {
         ETG_TRACE_USR4_THR(("[%25s] Unlock failed/timed out errno %d, %s", hmibase::trace::getAppName().c_str(), errno, ss.cPtr()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("[%25s] Unlock: no semaphore for descriptor %s", hmibase::trace::getAppName().c_str(), ss.cPtr()));
   }

   return false;
}


std::string ShaderStorage::FormatShaderBinaryPath(const FeatStd::Char* filename)
{
   util::SimpleString shaderBinaryPath;

   if (filename && filename[0] != '\0')
   {
      shaderBinaryPath = hmibase::util::SimpleString::format("%s", filename);
   }

   if (_shaderBinaryFolder != "")
   {
      shaderBinaryPath = hmibase::util::SimpleString::format("%s/%s", _shaderBinaryFolder.c_str(), shaderBinaryPath.cPtr());
   }

   //ETG_TRACE_USR4_THR(("FormatShaderBinaryPath returns %s", shaderBinaryPath.cPtr()));

   return shaderBinaryPath.cPtr();
}


bool ShaderStorage::Retrieve(StorageObject& object)
{
   bool success = false;

   if (object.m_data == 0)
   {
      StorageObject emptyObject(NULL, NULL, NULL);
      if (memcmp(object.m_identifier, emptyObject.m_identifier, StorageObject::IdentifierSize) == 0)
      {
         ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Retrieve failed due to unset GUID, shaderName is '%s'",
                            hmibase::trace::getAppName().c_str(),
                            (object.m_shaderName != NULL) ? object.m_shaderName : "Unknown"));
         ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Retrieve failed FragmentShader '%s'",
                             hmibase::trace::getAppName().c_str(),
                             (object.m_fragmentShadertext != NULL) ? object.m_fragmentShadertext : "Unknown"));
         ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Retrieve failed VertexShader '%s'",
                             hmibase::trace::getAppName().c_str(),
                             (object.m_vertexShadertext != NULL) ? object.m_vertexShadertext : "Unknown"));

         return false;
      }

      FeatStd::Char descriptor[DescriptorSize];

      /*
      For identifying shader programs and versions:

      If the identifier is set, use it to identify the shader program.
      If it is not set, you may generate a GUID from a namespace e.g. a hard coded GUID and the shader name from object.
      For shaders that are generated by widgets please make sure that they have a GUID or a name set.
      As shown below the actual code of the shader program can be used to generate a MD5 hash that is used to determine if the code of the shader program has changed.
      */
      MakeHumanReadable(object.m_identifier, StorageObject::IdentifierSize, descriptor, DescriptorSize);

      ::hmibase::util::SimpleString fullQualifiedPath = FormatShaderBinaryPath(descriptor);

      if (FileExist(fullQualifiedPath.cPtr()) == false)
      {
         return false;
      }

      if (Lock(descriptor))
      {
         if (fullQualifiedPath.cPtr() != 0)
         {
            //files shall not be opened with empty name
            FeatStd::FileHandle handle = FeatStd::File::Open(fullQualifiedPath.cPtr(), FeatStd::File::ReadBinary);
            if (handle)
            {
               FeatStd::File::Seek(handle, 0, FeatStd::File::End);
               FeatStd::OffsetType end = FeatStd::File::Tell(handle);

               if (end != -1)
               {
                  object.OnRelease = CleanupProc;
                  object.m_dataSize = end - Candera::GlShaderStorageInterface::StorageObject::IdentifierSize - 4;
                  object.m_data = FEATSTD_NEW_ARRAY(FeatStd::Char, object.m_dataSize);
                  FeatStd::File::Seek(handle, 0, FeatStd::File::Begin);
                  FeatStd::UInt32 type = 0;

#ifdef USE_MD5
                  // check for md5 sum
                  std::string shaderText;
                  FeatStd::UInt8 md5SumFromSource[Candera::GlShaderStorageInterface::StorageObject::IdentifierSize];

                  if (0 != object.m_shaderName)
                  {
                     shaderText += object.m_shaderName;
                  }
                  if (0 != object.m_vertexShadertext)
                  {
                     shaderText += object.m_vertexShadertext;
                  }
                  if (0 != object.m_fragmentShadertext)
                  {
                     shaderText += object.m_fragmentShadertext;
                  }
                  CustomIdentifierMd5(shaderText, md5SumFromSource, sizeof(md5SumFromSource));

                  FeatStd::UInt8 md5SumFromFile[Candera::GlShaderStorageInterface::StorageObject::IdentifierSize];
                  (void)FeatStd::File::Read(handle, &md5SumFromFile, 1, sizeof(md5SumFromFile));

                  if (memcmp(md5SumFromFile, md5SumFromSource, sizeof(md5SumFromFile)) == 0)
#endif
                  {
                     // check sums are identical, so stored binary matches and can be used
                     (void)FeatStd::File::Read(handle, &type, 4, 1);
                     object.m_binaryFormat = type;
                     (void)FeatStd::File::Read(handle, object.m_data, 1, static_cast<size_t>(end));
                     FeatStd::File::Close(handle);

                     ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Retrieve success %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
                     success = true;
                  }
#ifdef USE_MD5
                  else
                  {
                     // check sums differ, so don't reuse stored binary
                     FeatStd::File::Close(handle);

                     ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Retrieve md5Sum compare failed %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
                     success = false;
                  }
#endif
               }
               else
               {
                  // check sums differ, so don't reuse stored binary
                  FeatStd::File::Close(handle);

                  ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Retrieve ftell returned -1, %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
                  success = false;
               }
            }
            else
            {
               ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Retrieve failed to open %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
            }
         }

         (void)Unlock(descriptor);
      }
      else
      {
         ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Retrieve failed to lock file %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Retrieve object data is NULL", hmibase::trace::getAppName().c_str()));
   }

   return success;
}


bool ShaderStorage::Persist(StorageObject& object)
{
   bool ret = false;

   if (object.m_data != 0)
   {
      StorageObject emptyObject(NULL, NULL, NULL);
      if (memcmp(object.m_identifier, emptyObject.m_identifier, StorageObject::IdentifierSize) == 0)
      {
         ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Persist failed due to unset GUID, shaderName is '%s'",
                            hmibase::trace::getAppName().c_str(),
                            (object.m_shaderName != NULL) ? object.m_shaderName : "Unknown"));
         ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Persist failed FragmentShader '%s'",
                             hmibase::trace::getAppName().c_str(),
                             (object.m_fragmentShadertext != NULL) ? object.m_fragmentShadertext : "Unknown"));
         ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Persist failed VertexShader '%s'",
                             hmibase::trace::getAppName().c_str(),
                             (object.m_vertexShadertext != NULL) ? object.m_vertexShadertext : "Unknown"));

         return false;
      }

      FeatStd::Char descriptor[DescriptorSize];

      /*
      For identifying shader programs and versions:
      If the identifier is set, use it to identify the shader program.
      If it is not set, you may generate a GUID from a namespace e.g. a hard coded GUID and the shader name.
      For shaders that are generated by widgets please make sure that they have a GUID or a name set.
      As shown below the actual code of the shader program can be used to generate a MD5 hash that is used to determine if the code of the shader program has changed.
      */
      MakeHumanReadable(object.m_identifier, StorageObject::IdentifierSize, descriptor, DescriptorSize);

      ::hmibase::util::SimpleString fullQualifiedPath = FormatShaderBinaryPath(descriptor);

      if (Lock(descriptor))
      {
         FeatStd::FileHandle handle = FeatStd::File::Open(fullQualifiedPath.cPtr(), FeatStd::File::WriteBinary);
         if (handle)
         {
#ifdef USE_MD5
            std::string shaderText;
            FeatStd::UInt8 md5Identifier[Candera::GlShaderStorageInterface::StorageObject::IdentifierSize];

            if (0 != object.m_shaderName)
            {
               shaderText += object.m_shaderName;
            }
            if (0 != object.m_vertexShadertext)
            {
               shaderText += object.m_vertexShadertext;
            }
            if (0 != object.m_fragmentShadertext)
            {
               shaderText += object.m_fragmentShadertext;
            }
            CustomIdentifierMd5(shaderText, md5Identifier, sizeof(md5Identifier));

            FeatStd::File::Write(handle, &md5Identifier, 1, sizeof(md5Identifier));
#endif
            FeatStd::UInt32 type = object.m_binaryFormat;
            FeatStd::File::Write(handle, &type, 4, 1);
            FeatStd::File::Write(handle, object.m_data, 1, object.m_dataSize);
            FeatStd::File::Close(handle);

            ChangeOwner(fullQualifiedPath.cPtr());

            ETG_TRACE_USR4_THR(("[%25s] ShaderStorage::Persist success %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
            ret = true;
         }
         else
         {
            ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Persist failed with errno %d to open %s", hmibase::trace::getAppName().c_str(), errno, fullQualifiedPath.cPtr()));
         }
         (void)Unlock(descriptor);
      }
      else
      {
         ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Retrieve failed to lock file %s", hmibase::trace::getAppName().c_str(), fullQualifiedPath.cPtr()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("[%25s] ShaderStorage::Persist object data is NULL", hmibase::trace::getAppName().c_str()));
   }

   return ret;
}


void ShaderStorage::CleanupProc(StorageObject* object)
{
   if (object->m_data != 0)
   {
      FEATSTD_DELETE_ARRAY(object->m_data);
      object->m_data = 0;
   }
}


#ifdef FEATSTD_PLATFORM_OS_Win32
bool ShaderStorage::CustomIdentifierMd5(std::string& shaderText, FeatStd::UInt8* object, FeatStd::SizeType size)
{
   DWORD dwStatus = 0;

   HCRYPTPROV hProv = 0;
   HCRYPTHASH hHash = 0;

   DWORD cbHash = 0;

   if (!CryptAcquireContext(&hProv,
                            NULL,
                            NULL,
                            PROV_RSA_FULL,
                            CRYPT_VERIFYCONTEXT))
   {
      dwStatus = GetLastError();
      printf("CryptAcquireContext failed: %d\n", dwStatus);
      return false;
   }

   if (!CryptCreateHash(hProv, CALG_MD5, 0, 0, &hHash))
   {
      dwStatus = GetLastError();
      printf("CryptAcquireContext failed: %d\n", dwStatus);

      CryptReleaseContext(hProv, 0);
      return false;
   }

   if (!CryptHashData(hHash, reinterpret_cast<const BYTE*>(shaderText.c_str()), shaderText.length(), 0))
   {
      dwStatus = GetLastError();
      printf("CryptHashData failed: %d\n", dwStatus);
      CryptReleaseContext(hProv, 0);
      CryptDestroyHash(hHash);
      return false;
   }

   //cbHash = object.IdentifierSize;
   cbHash = size;
   if (!CryptGetHashParam(hHash, HP_HASHVAL, reinterpret_cast<BYTE*>(object), &cbHash, 0))
   {
      dwStatus = GetLastError();
      printf("CryptGetHashParam failed: %d\n", dwStatus);
      CryptDestroyHash(hHash);
      CryptReleaseContext(hProv, 0);
      return false;
   }

   CryptDestroyHash(hHash);
   CryptReleaseContext(hProv, 0);
   return true;
}


#endif //FEATSTD_PLATFORM_OS_Win32

#ifdef FEATSTD_PLATFORM_OS_Posix
bool ShaderStorage::CustomIdentifierMd5(std::string& shaderText, FeatStd::UInt8* object, FeatStd::SizeType size)
{
   if (MD5_DIGEST_LENGTH <= size)
   {
      return (0 != MD5(reinterpret_cast<const unsigned char*>(shaderText.c_str()), shaderText.length(), object));
   }
   return false;
}


#endif // FEATSTD_PLATFORM_OS_Posix
}


}

#endif
