/* ***************************************************************************************
* FILE:          SceneTransitionMediator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SceneTransitionMediator.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "View/CGI/CgiExtensions/SceneTransitionMediator.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SceneTransitionMediator.cpp.trc.h"
#endif

namespace hmibase {
   namespace view {
      SceneTransitionMediator::SceneTransitionMediator() : _viewHandler(0)
      {
      }


      SceneTransitionMediator::~SceneTransitionMediator()
      {
         _sceneMediatorPairList.Clear();
         _viewHandler = 0;
      }


      bool SceneTransitionMediator::init(Courier::IViewHandler* viewHandler)
      {
         _viewHandler = viewHandler;
         return true;
      }


      void SceneTransitionMediator::enter(const Courier::ViewId& viewId, TransitionHint hint)
      {
         hint = 2;
         SceneTransitionMediatorPair* p = findHint(hint);
         if (p == 0)
         {
            p = allocMediatorPair();
         }
         if (p != 0)
         {
            p->_viewIdEnter = viewId;
            p->_hint = hint;
         }
         ETG_TRACE_USR1_THR(("SceneTransitionMediator::enter (viewID:%s)", viewId.CStr()));
         check(hint);
      }


      void SceneTransitionMediator::leave(const Courier::ViewId& viewId, TransitionHint hint)
      {
         hint = 2;
         SceneTransitionMediatorPair* p = findHint(hint);
         if (p == 0)
         {
            p = allocMediatorPair();
         }
         if (p != 0)
         {
            p->_viewIdLeave = viewId;
            p->_hint = hint;
         }
         ETG_TRACE_USR1_THR(("SceneTransitionMediator::leave (viewID:%s)", viewId.CStr()));
         check(hint);
      }


      bool SceneTransitionMediator::check(TransitionHint hint)
      {
         SceneTransitionMediatorPair* p = findHint(hint);
         if (p != 0)
         {
            return p->checkTransition(_viewHandler);
         }
         return false;
      }


      void SceneTransitionMediator::onDataComplete(TransitionHint hint)
      {
         SceneTransitionMediatorPair* p = findHint(hint);
         if (p != 0)
         {
            p->onDataComplete(hint);
         }
      }


      SceneTransitionMediatorPair* SceneTransitionMediator::findHint(TransitionHint hint)
      {
         for (FeatStd::SizeType idx = 0; idx < _sceneMediatorPairList.Size(); ++idx)
         {
            SceneTransitionMediatorPair* p = _sceneMediatorPairList[idx];
            if (p->_hint == hint)
            {
               return p;
            }
         }
         return 0;
      }


      SceneTransitionMediatorPair* SceneTransitionMediator::allocMediatorPair()
      {
         SceneTransitionMediatorPair* p = findHint(0);
         if (p == 0)
         {
            p = new SceneTransitionMediatorPair(_viewHandler);
            _sceneMediatorPairList.Add(p);
         }
         return p;
      }


      // ------------------------------------------------------------------------------

      SceneTransitionMediatorPair::SceneTransitionMediatorPair(Courier::IViewHandler* viewHandler) :
         _hint(0),
         _viewIdEnter(),
         _viewIdLeave(),
         _viewLeave(0),
         _viewEnter(0)
      {
         _viewFacade.Init(viewHandler);
      }


      SceneTransitionMediatorPair::~SceneTransitionMediatorPair()
      {
         _viewLeave = 0;
         _viewEnter = 0;
      }


      void SceneTransitionMediatorPair::cleanup()
      {
         _hint = 0;
         _viewIdEnter = Courier::ViewId();
         _viewIdLeave = Courier::ViewId();
         _viewLeave = 0;
         _viewEnter = 0;
      }


      bool SceneTransitionMediatorPair::checkTransition(Courier::IViewHandler* viewHandler)
      {
         if (_viewIdEnter != Courier::ViewId() /*&& _viewIdLeave != Courier::ViewId()*/)
         {
            _viewLeave = viewHandler->FindView(_viewIdLeave);
            _viewEnter = viewHandler->FindView(_viewIdEnter);

            if (0 == _viewEnter)
            {
               // after processing this message the view is loaded into system ram containing the
               // scene node, device objects (bitmaps), widgets and view controller.
               // The message processing is confirmed by a ViewResMsg
               Courier::ViewReqMsg view2CreateMsg(Courier::ViewAction::CreateAll, _viewIdEnter, true, false);
               _viewFacade.OnViewComponentMessage(view2CreateMsg);

               ETG_TRACE_USR1_THR(("SceneTransitionMediator::Create view on enter (viewID:%s)", _viewIdEnter.CStr()));
               _viewEnter = viewHandler->FindView(_viewIdEnter);
            }

            if (0 != _viewEnter)
            {
               // after processing this message the view (view controller and widgets)
               // receives messages. The message processing is confirmed by a ViewMessagingResMsg
               Courier::ViewMessagingReqMsg view2MessagingRegMsg(_viewIdEnter, true);
               _viewFacade.OnViewComponentMessage(view2MessagingRegMsg);

               if (_viewLeave != 0)
               {
                  // after processing this message the rendering of the view is disabled.
                  // The message processing is confirmed by a ViewRenderingResMsg.
                  Courier::ViewMessagingReqMsg view1MessagingRegMsg(_viewIdLeave, false);
                  _viewFacade.OnViewComponentMessage(view1MessagingRegMsg);
               }
               // Load
               // after processing this message the view is unloaded from system ram including
               // the scene node, device objects (bitmaps), widgets and view controller.
               // The message processing is confirmed by a ViewResMsg
               Courier::LoadReqMsg view2LoadReqMsg(_viewIdEnter, true);
               _viewFacade.OnViewComponentMessage(view2LoadReqMsg);

               // Activate View2 and enable surface to accepts inputs
               ///Courier::ActivationReqMsg activationMsg(_viewIdEnter, true, false);
               //_viewFacade.OnViewComponentMessage(activationMsg);

               // setup a listener, waiting for confirmation that data (e.g from ListDataProvider)
               // is available or a configured timeout (350) finalize the transition with onDataComplete.
               setListenerLeave(false, 0);
               if (setListenerEnter(true, 650) == false)
               {
                  // if on DataCompletionHandler is available, no timeout, finalize the transition,
                  // show the scene.
                  ETG_TRACE_USR1_THR(("SceneTransitionMediator::No DataCompletion handler (viewID:%s)", _viewIdEnter.CStr()));
                  onDataComplete(_hint);  // no DataCompletion Handler, show view
               }
            }
            return true;
         }
         _viewLeave = viewHandler->FindView(_viewIdLeave);
         setListenerLeave(true, 1500);
         return false;
      }


      // As soon as the data is available in the view(widget / view controller)
      // a message has to be sent from the view(widget / view controller) to
      // the controller.This message will then trigger the following :

      void SceneTransitionMediatorPair::onDataComplete(TransitionHint hint)
      {
         ETG_TRACE_USR1_THR(("SceneTransitionMediatorPair::onDataComplete (Hint %d)", hint));
         if (hint == _hint || hint == 0) // todo check zero for timeout etc.
         {
            // stop listener
            setListenerLeave(false, 0);
            setListenerEnter(false, 0);

            if (0 != _viewEnter)
            {
               // Start Rendering for View2
               // after processing this message the rendering of the view is enabled.
               // The message processing is confirmed by a ViewRenderingResMsg.
               Courier::ViewRenderingReqMsg view2RenderingReqMsg(_viewIdEnter, true);
               _viewFacade.OnViewComponentMessage(view2RenderingReqMsg);
            }
            if (0 != _viewLeave)
            {
               // Stop rendering View1
               // after processing this message the rendering of the view is disabled.
               // The message processing is confirmed by a ViewRenderingResMsg
               Courier::ViewRenderingReqMsg view1RenderingReqMsg(_viewIdLeave, false);
               _viewFacade.OnViewComponentMessage(view1RenderingReqMsg);

               // Unload content of View1
               // after processing this message the view is unloaded from the VRAM.
               // The message processing is confirmed by a LoadResMsg.
               Courier::LoadReqMsg view1LoadReqMsg(_viewIdLeave, false);
               _viewFacade.OnViewComponentMessage(view1LoadReqMsg);

               // Destroy View1
               // after processing this message the view is unloaded from system ram including
               // the scene node, device objects (bitmaps), widgets and view controller.
               // The message processing is confirmed by a ViewResMsg
               Courier::ViewReqMsg view1DestroyMsg(Courier::ViewAction::Destroy, _viewIdLeave, false, false);
               _viewFacade.OnViewComponentMessage(view1DestroyMsg);
            }
            cleanup();  // free
         }
      }


      bool SceneTransitionMediatorPair::setListenerLeave(bool add, FeatStd::UInt32 timeout)
      {
         if (0 != _viewLeave)
         {
            ViewControllerBase* viewController = dynamic_cast <ViewControllerBase*>(_viewLeave->GetViewController());
            if (viewController)
            {
               DataCompletionHandler* dataCompletionHandler = viewController->getDataCompletionHandler();
               if (0 != dataCompletionHandler)
               {
                  return dataCompletionHandler->setDataCompletionListener(this, add, timeout);
               }
            }
         }
         return false;
      }


      bool SceneTransitionMediatorPair::setListenerEnter(bool add, FeatStd::UInt32 timeout)
      {
         if (0 != _viewEnter)
         {
            ViewControllerBase* viewController = dynamic_cast <ViewControllerBase*>(_viewEnter->GetViewController());
            if (viewController)
            {
               DataCompletionHandler* dataCompletionHandler = viewController->getDataCompletionHandler();
               if (0 != dataCompletionHandler)
               {
                  return dataCompletionHandler->setDataCompletionListener(this, add, timeout);
               }
            }
         }
         return false;
      }
   }
}