/* ***************************************************************************************
* FILE:          RendererStatisticsOverlayWrapper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RendererStatisticsOverlayWrapper.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "RendererStatisticsOverlayWrapper.h"

#include <CanderaAssetLoader/AssetLoaderBase/AssetIdMacros.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDescriptor.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RendererStatisticsOverlayWrapper.cpp.trc.h"
#endif

namespace hmibase {
namespace view {

static const Candera::Char* refTransVertexShader = {
    "#ifndef GL_ES\n"
    "#define lowp\n"
    "#define mediump\n"
    "#define highp\n"
    "#define precision\n"
    "#endif\n"
    "    uniform mat4 u_MVPMatrix;\n"
    "    attribute vec4 a_Position;\n"
    "    attribute vec4 a_Color;\n"
    "    attribute vec2 a_TextureCoordinate;\n"
    "    varying mediump vec4 v_Color;\n"
    "    varying mediump vec2 v_TexCoord;\n"
    "    /*---------------------------------- MAIN ------------------------------------*/\n"
    "    void main(void)\n"
    "    {\n"
    "        gl_Position = u_MVPMatrix * a_Position;\n"
    "        v_Color = a_Color;\n"
    "        v_TexCoord = a_TextureCoordinate;\n"
    "    }\n"
};

static const Candera::Char* refTexFragmentShader = {
    "#ifndef GL_ES\n"
    "#define lowp\n"
    "#define mediump\n"
    "#define highp\n"
    "#define precision\n"
    "#endif\n"
    "precision mediump float;\n"
    "uniform sampler2D u_Texture;\n"
    "varying vec2 v_TexCoord;\n"
    "varying vec4 v_Color;\n"
    "/*---------------------------------- MAIN ------------------------------------*/\n"
    "void main(void)\n"
    "{\n"
    "    gl_FragColor = texture2D(u_Texture, v_TexCoord) * v_Color;\n"
    "}\n"
};


// RenderTargetEventListener
class RenderTargetEventListener : public FeatStd::EventListener {
public:
    RenderTargetEventListener(RendererStatisticsOverlayWrapper* renderStatisticsOverlayWrapper) : mRenderStatisticsOverlayWrapper(renderStatisticsOverlayWrapper){}
    virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& event)
    {

        const Candera::RenderTarget::BeforeSwapEvent* beforeSwapEvent = Candera::Dynamic_Cast<const Candera::RenderTarget::BeforeSwapEvent*>(&event);
        if (beforeSwapEvent == 0) {
            return FeatStd::EventResult::Proceed;
        }
        Candera::RenderTarget* renderTarget = beforeSwapEvent->GetRenderTarget();

        if ((0 != mRenderStatisticsOverlayWrapper->mRendererStatisticsOverlay) && (0 != renderTarget)) {
            Candera::GraphicDeviceUnit* gdu = renderTarget->GetGraphicDeviceUnit();
            if ((0 != gdu) && (mRenderStatisticsOverlayWrapper->mIsOverlayEnabled)) {
                mRenderStatisticsOverlayWrapper->mRendererStatisticsOverlay->Render(gdu->ToRenderTarget3D());
            }
        }
        return FeatStd::EventResult::Proceed;
    }

private:
    RendererStatisticsOverlayWrapper* mRenderStatisticsOverlayWrapper;
};


RendererStatisticsOverlayWrapper::RendererStatisticsOverlayWrapper() :
mIsOverlayEnabled(false),
mRenderTargetEventListener(0)
{
    Candera::MemoryManagement::SharedPointer<Candera::Shader> shader = Candera::Shader::Create();
    if (!shader->SetVertexShader(refTransVertexShader, 0))
    {
       ETG_TRACE_ERR(("RendererStatisticsOverlayWrapper, SetVertexShader failed"));
    }
    if (!shader->SetFragmentShader(refTexFragmentShader, 0))
    {
       ETG_TRACE_ERR(("RendererStatisticsOverlayWrapper, SetFragmentShader failed"));
    }
    shader->SetName("RefTransRefTex");

    mRendererStatisticsOverlay = Candera::RendererStatisticsOverlay::Create(shader);
    mRendererStatisticsOverlay->GetAppearance()->GetRenderMode()->SetBlendingEnabled(false);
    mRendererStatisticsOverlay->SetTextColor(Candera::Color(0.8F, 0.8F, 0.8F, 1.0F));
    mRendererStatisticsOverlay->SetValueColor(Candera::Color(1.0F, 1.0F, 1.0F, 1.0F));
}

// ---------------------------------------------------------------------
void RendererStatisticsOverlayWrapper::Init()
{
    mRenderTargetEventListener = CANDERA_NEW(RenderTargetEventListener)(this);
    Candera::DefaultAssetProvider& assetProvider = Candera::DefaultAssetProvider::GetInstance();
    const Candera::AssetDescriptor& assetDescriptor = assetProvider.GetAssetDescriptor();
    static bool isRenderStatisticOverlayAttached = false;
    Candera::GraphicDeviceUnit* gdu = 0;

    for (Candera::AssetDescriptor::RenderTargetIdIterator gduList = assetDescriptor.GetRenderTargetIdIterator(); gduList.IsValid(); ++gduList) {
        gdu = assetProvider.GetGraphicDeviceUnitById(*gduList);
        if (gdu != 0) {

            Candera::DevicePackageDescriptor::UnitCategory unitCategory = Candera::DevicePackageDescriptor::GetUnitCategory(gdu->GetUnitType());
            switch (unitCategory) {
                case Candera::DevicePackageDescriptor::DisplayTarget3D:
                case Candera::DevicePackageDescriptor::DisplayTarget2D:
                case Candera::DevicePackageDescriptor::Mixed2D3DDisplayTarget:
                    // attach listener to first valid RT
                    if (!isRenderStatisticOverlayAttached) {
                        Candera::RenderTarget2D* renderTarget = gdu->ToRenderTarget2D();

                        if (0 != renderTarget) {
                            renderTarget->AddEventListener(mRenderTargetEventListener);
                            isRenderStatisticOverlayAttached = true;
                        }
                    }
                    break;
                default:
                    //do nothing;
                    break;
            }
        }
    }
}

}
}
