/* ***************************************************************************************
* FILE:          MemoryMappedAssetRepository.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MemoryMappedAssetRepository.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include <sys/mman.h>
#include <memory.h>

#include "MemoryMappedAssetRepository.h"
#include <CanderaPlatform/OS/StringPlatform.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetLoaderMemoryPool.h>


namespace Candera {

// ----------------------------------------------------------------------------
static const UInt8* MapFailed()
{
   return static_cast<const UInt8*>(MAP_FAILED);
}


// ----------------------------------------------------------------------------
MemoryMappedAssetRepository::MemoryMappedAssetRepository(const Char* fileName) :
   mFileName(0),
   mBasePtr(MapFailed()),
   mEndPtr(MapFailed()),
   mReadPtr(MapFailed()),
   mFileHandle(-1)
{
   mFileName = ASSETLOADER_TRANSIENT_NEW_ARRAY(Char, StringPlatform::Length(fileName) + 1);
   if (mFileName != 0)
   {
      StringPlatform::Copy(mFileName, fileName);
   }
   HMI_APP_ASSERT(mFileName != 0);
}


// ----------------------------------------------------------------------------
MemoryMappedAssetRepository::~MemoryMappedAssetRepository()
{
   (void) finalize();
   ASSETLOADER_DELETE_ARRAY(mFileName);
   mFileName = 0;
}  //lint !e1579  Member pointer mBasePtr, mEndPtr and mReadPtr are set to MAP_FAILED in finalize()

// ----------------------------------------------------------------------------
bool MemoryMappedAssetRepository::Initialize()
{
   bool flag = (mBasePtr == MapFailed()) ? true : false;
   HMI_APP_ASSERT(flag);
   if (mBasePtr != MapFailed())
   {
      // log error log_error("attempt to map %s twice\n", fileName);
      return false;
   }

   mFileHandle = ::open(mFileName, O_RDONLY, 0);
   if (mFileHandle == -1)
   {
      // log_error("failed to open file %s\n", mFileName);
      return false;
   }

   off_t fsize = lseek(mFileHandle, 0, SEEK_END);
   if (fsize < 0)
   {
      // log_error("failed to lseek in file %s\n", mFileName);
      return false;
   }

   mBasePtr = static_cast<const UInt8*>(mmap(NULL, static_cast<size_t>(fsize), PROT_READ, MAP_SHARED, mFileHandle, 0));

   if (mBasePtr != MapFailed())
   {
      mEndPtr = mBasePtr + fsize;
      mReadPtr = mBasePtr;
   }
   else
   {
      finalize();
      // log_error("map file into memory %s\n", fileName);
   }

   return mBasePtr != MapFailed();
}


// ----------------------------------------------------------------------------
bool MemoryMappedAssetRepository::finalize()
{
   if (mBasePtr != MapFailed())
   {
      ::munmap(const_cast<UInt8*>(mBasePtr), static_cast<size_t>(mEndPtr - mBasePtr));
      mBasePtr = MapFailed();
      mEndPtr = MapFailed();
      mReadPtr = MapFailed();
   }

   if (mFileHandle != -1)
   {
      ::close(mFileHandle);
      mFileHandle = -1;
   }
   return true;
}


// ----------------------------------------------------------------------------
MemoryMappedAssetRepository::AssetAccessMode MemoryMappedAssetRepository::GetAssetAccessMode()
{
   return ReadOnlyAddressable;
}


// ----------------------------------------------------------------------------
SizeType MemoryMappedAssetRepository::CapByteSize(SizeType elementSize, SizeType count) const
{
   SizeType nBytes = elementSize * count;
   if ((mReadPtr + nBytes) > mEndPtr)
   {
      nBytes = mEndPtr - mReadPtr;
   }
   return nBytes;
}


// ----------------------------------------------------------------------------
#if ((COURIER_VERSION_MAJOR > 3) || (COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2))
SizeType MemoryMappedAssetRepository::AssetRead(void* buffer, SizeType elementSize, SizeType count)
#else
Int32 MemoryMappedAssetRepository::AssetRead(void* buffer, Int32 elementSize, Int32 count)
#endif
{
   SizeType nBytes = CapByteSize(elementSize, count);
   if (nBytes > 0)
   {
      memcpy(buffer, mReadPtr, nBytes);
      mReadPtr += nBytes;
   }

   return (elementSize != 0) ? nBytes / elementSize : 0;
}


// ----------------------------------------------------------------------------
#if ((COURIER_VERSION_MAJOR > 3) || (COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2))
bool MemoryMappedAssetRepository::AssetSeek(OffsetType offset, AssetSeekMark mark)
#else
bool MemoryMappedAssetRepository::AssetSeek(Int32 offset, AssetSeekMark mark)
#endif
{
   bool flag = (mBasePtr != MapFailed()) ? true : false;
   HMI_APP_ASSERT(flag);

   bool ok;

   switch (mark)
   {
      case Begin:
      {
         ok = ((mBasePtr + offset) <= mEndPtr);
         if (ok)
         {
            mReadPtr = mBasePtr + offset;
         }
         break;
      }

      case Current:
      {
         ok = ((mReadPtr + offset) <= mEndPtr);
         if (ok)
         {
            mReadPtr = mReadPtr + offset;
         }
         break;
      }

      default:
      {
         HMI_APP_ASSERT_ALWAYS();
         ok = false;
         break;
      }
   }
   return ok;
}


// ----------------------------------------------------------------------------
#if ((COURIER_VERSION_MAJOR > 3) || (COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2))
OffsetType MemoryMappedAssetRepository::AssetTell()
#else
Int32 MemoryMappedAssetRepository::AssetTell()
#endif
{
   bool flag = (mBasePtr != MapFailed()) ? true : false;
   HMI_APP_ASSERT(flag);
   return mReadPtr - mBasePtr;
}


// ----------------------------------------------------------------------------
#if ((COURIER_VERSION_MAJOR > 3) || (COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2))
SizeType MemoryMappedAssetRepository::GetConstData(const void** buffer, SizeType elementSize, SizeType count)
#else
Int32 MemoryMappedAssetRepository::GetConstData(const void** buffer, Int32 elementSize, Int32 count)
#endif
{
   bool flag = (mBasePtr != MapFailed()) ? true : false;
   HMI_APP_ASSERT(flag);

   SizeType nBytes = CapByteSize(elementSize, count);
   if (nBytes > 0)
   {
      *buffer = mReadPtr;
      mReadPtr += nBytes;
   }
   return (elementSize != 0) ? nBytes / elementSize : 0;
}


}
