/* ***************************************************************************************
* FILE:          ImageUploadCache.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageUploadCache is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "ImageUploadCache.h"

#include <CanderaPlatform/Device/Common/Internal/EGL/EglContextCreator.h>
#include <Focus/FManager.h>//required only for app id

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_IMAGE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ImageUploadCache.cpp.trc.h"
#endif

#if defined(VARIANT_S_FTR_ENABLE_IMAGE_UPLOAD_CACHE)

namespace hmibase {
namespace view {
namespace utils {


/////////////////////////////////////////////////////////////////////////
#if defined(VARIANT_S_FTR_IMAGE_UPLOAD_CACHE_MAXIMUM_IMAGE_SIZE)
const size_t ImageUploadCache::DefaultMaximumImageSize = static_cast<size_t>(VARIANT_S_FTR_IMAGE_UPLOAD_CACHE_MAXIMUM_IMAGE_SIZE);
#else
const size_t ImageUploadCache::DefaultMaximumImageSize = 500000;
#endif

#if defined(VARIANT_S_FTR_IMAGE_UPLOAD_CACHE_MAXIMUM_CACHED_IMAGES_SIZE)
const size_t ImageUploadCache::DefaultMaximumCachedImagesSize = static_cast<size_t>(VARIANT_S_FTR_IMAGE_UPLOAD_CACHE_MAXIMUM_CACHED_IMAGES_SIZE);
#else
const size_t ImageUploadCache::DefaultMaximumCachedImagesSize = 1000000;
#endif

#if defined(VARIANT_S_FTR_IMAGE_UPLOAD_CACHE_MAXIMUM_TOTAL_IMAGES_SIZE)
const size_t ImageUploadCache::DefaultMaximumTotalImagesSize = static_cast<size_t>(VARIANT_S_FTR_IMAGE_UPLOAD_CACHE_MAXIMUM_TOTAL_IMAGES_SIZE);
#else
const size_t ImageUploadCache::DefaultMaximumTotalImagesSize = 0;
#endif

/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageUploadCache() :
   _hitCount(0),
   _hitSize(StatsSize()),
   _missCount(0),
   _missSize(StatsSize()),
   _activeImagesSize(0),
   _maximumImageSize(DefaultMaximumImageSize),
   _maximumCachedImagesSize(DefaultMaximumCachedImagesSize),
   _maximumTotalImagesSize(DefaultMaximumTotalImagesSize)
{
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getMaximumImageSize() const
{
   return _maximumImageSize;
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::setMaximumImageSize(size_t value)
{
   _maximumImageSize = value;
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getMaximumCachedImagesSize() const
{
   return _maximumCachedImagesSize;
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::setMaximumCachedImagesSize(size_t value)
{
   _maximumCachedImagesSize = value;
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getMaximumTotalImagesSize() const
{
   return _maximumTotalImagesSize;
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::setMaximumTotalImagesSize(size_t value)
{
   _maximumTotalImagesSize = value;
}


/////////////////////////////////////////////////////////////////////////
int ImageUploadCache::getAppId()
{
   //for now take the app id from the focus manager
   return Focus::FManager::getInstance().getCurrentAppId();
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::enable()
{
#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
   if (Candera::BitmapBrush::GetUploadListener() == NULL)
   {
      Candera::BitmapBrush::SetUploadListener(&(ImageUploadCache::getInstance()));
   }
#else
   _TODO("Customizations for BitmapBrush are not available. Most likely the git commits from ai_cgi_delivery are not in your branch.")
   ETG_TRACE_FATAL_THR(("ImageUploadCache can not be enabled because the customizations for BitmapBrush are not available. Most likely the git commits from ai_cgi_delivery are not in your branch."));
#endif

   Candera::EglContextCreator::GetEventSource().AddEventListener(this);
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::disable()
{
#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
   if (Candera::BitmapBrush::GetUploadListener() == &(ImageUploadCache::getInstance()))
   {
      Candera::BitmapBrush::SetUploadListener(NULL);
   }
#endif

   Candera::EglContextCreator::GetEventSource().RemoveEventListener(this);
   reset();
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getActiveImagesCount() const
{
   return _images.size() - _cachedImages.getImagesCount();
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getActiveImagesSize() const
{
   return _activeImagesSize;
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getCachedImagesCount() const
{
   return _cachedImages.getImagesCount();
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getCachedImagesSize() const
{
   return _cachedImages.getImagesSize();
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getTotalImagesCount() const
{
   return getActiveImagesCount() + getCachedImagesCount();
}


/////////////////////////////////////////////////////////////////////////
size_t ImageUploadCache::getTotalImagesSize() const
{
   return getActiveImagesSize() + getCachedImagesSize();
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageName ImageUploadCache::getImageName(const ImageSharedPointer& image)
{
   return (image.PointsToNull() || (image->GetName() == NULL)) ? "<null>" : image->GetName();
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageId ImageUploadCache::getImageId(const ImageSharedPointer& image)
{
   return image.PointsToNull() ? ImageId() : image.GetPointerToSharedInstance();
   //return image.PointsToNull() ? ImageId() : image->GetId();
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageSize ImageUploadCache::getImageSize(const ImageSharedPointer& image)
{
   return image.PointsToNull() ? ImageSize() : (image->GetWidth() * image->GetHeight());
}


/////////////////////////////////////////////////////////////////////////
FeatStd::EventResult::Enum ImageUploadCache::OnEvent(const FeatStd::Event& evt)
{
   const Candera::EglContextCreator::BeforeUnloadEvent* unloadEvent = Candera::Dynamic_Cast<const Candera::EglContextCreator::BeforeUnloadEvent*>(&evt);
   if (unloadEvent != NULL)
   {
      ETG_TRACE_USR1_THR(("<%d>  - Context %p with SharedContext %p Unloaded", getAppId(), unloadEvent->GetEglContext(), unloadEvent->GetSharedContext()));
      reset();
   }
   const Candera::EglContextCreator::AfterUploadEvent* uploadEvent = Candera::Dynamic_Cast<const Candera::EglContextCreator::AfterUploadEvent*>(&evt);
   if (uploadEvent != NULL)
   {
      ETG_TRACE_USR1_THR(("<%d>  - Context %p with SharedContext %p Uploaded", getAppId(), uploadEvent->GetEglContext(), uploadEvent->GetSharedContext()));
   }
   return FeatStd::EventResult::Proceed;
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::OnPostUpload(const ImageSharedPointer& image)
{
   if (image.PointsToNull() || !image->IsTypeOf(Candera::BitmapImage2D::GetTypeId()))
   {
      return;
   }

   ImageId imageId = getImageId(image);
   ImageSize imageSize = getImageSize(image);

   //invalid id can't be cached
   if (imageId == ImageId())
   {
      return;
   }

   //get the existing entry or insert a new one if no entry was there already
   ImageInfo*& imageInfo = _images[imageId];

   ETG_TRACE_USR4_THR(("<%d> onUpload - cached(count=%u, size=%u), active(count=%u, size=%u), image(index=%u, id=%p, size=%u, upCounter=%d, name=%s)",
                       getAppId(),
                       getCachedImagesCount(),
                       getCachedImagesSize(),
                       getActiveImagesCount(),
                       getActiveImagesSize(),
                       (imageInfo == NULL) ? 0 : imageInfo->getOwnerAppendIndex(),
                       imageId,
                       imageSize,
                       (imageInfo == NULL) ? -1 : imageInfo->_uploadCounter,
                       image->GetName()));

   //image info already exist from previous calls
   if (imageInfo != NULL)
   {
      //if (imageInfo->_imageId != imageId)
      //{
      //   //todo: add error handling
      //}

      //size was changed since previous calls
      if (imageInfo->_imageSize != imageSize)
      {
         ETG_TRACE_ERR_THR(("<%d> image size was changed since previous calls - oldSize=%u, newSize=%u, image(index=%u, id=%p, name=%s)",
                            getAppId(),
                            imageInfo->_imageSize,
                            imageSize,
                            imageInfo->getOwnerAppendIndex(),
                            imageInfo->_imageId,
                            image->GetName()));
      }

      //image was cached before and it will be reused
      if (imageInfo->_uploadCounter == 0)
      {
         _cachedImages.removeImage(imageInfo);
         imageInfo->_uploadCounter = 1;
      }
      //image already uploaded
      else
      {
         ++(imageInfo->_uploadCounter);
         return;
      }
   }

   //new image
   bool newImage = false;
   if (imageInfo == NULL)
   {
      newImage = true;
      imageInfo = FEATSTD_NEW(ImageInfo)(image);
      if (imageInfo == NULL)
      {
         //no memory
         return;
      }

      //call upload to increase the upload counter of the new image
      image->Upload();
      imageInfo->_uploadCounter = 1;
   }

   imageInfo->updateInfo();
   _activeImagesSize += imageInfo->_imageSize;

   if (newImage)
   {
      ++_missCount;
      _missSize += StatsSize(imageInfo->_imageSize);

      ETG_TRACE_USR1_THR(("<%d> new image (MISS) - miss(count=%u, size=%u), cached(count=%u, size=%u), image(index=%u, id=%p, size=%u, name=%s)",
                          getAppId(),
                          _missCount,
                          static_cast<unsigned int>(_missSize),
                          getCachedImagesCount(),
                          getCachedImagesSize(),
                          imageInfo->getOwnerAppendIndex(),
                          imageId,
                          getImageSize(image),
                          image->GetName()));
   }
   else
   {
      ++_hitCount;
      _hitSize += StatsSize(imageInfo->_imageSize);

      ETG_TRACE_USR1_THR(("<%d> reuse image (HIT) - hit(count=%u, size=%u), cached(count=%u, size=%u), image(index=%u, id=%p, size=%u, name=%s)",
                          getAppId(),
                          _hitCount,
                          static_cast<unsigned int>(_hitSize),
                          getCachedImagesCount(),
                          getCachedImagesSize(),
                          imageInfo->getOwnerAppendIndex(),
                          imageId,
                          getImageSize(image),
                          image->GetName()));
   }

   //ensure total size does not exceed maximum allowed
   ensureMaximumSize();
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::OnPreUnload(const ImageSharedPointer& image)
{
   if (image.PointsToNull())
   {
      return;
   }

   ImageId imageId = getImageId(image);
   ImageSize imageSize = getImageSize(image);

   //invalid id can't be cached
   if (imageId == ImageId())
   {
      return;
   }

   ImageInfoMap::iterator it = _images.find(imageId);
   if (it == _images.end())
   {
      return;
   }

   ImageInfo* imageInfo = it->second;
   if (imageInfo == NULL)
   {
      _images.erase(it);
      ETG_TRACE_ERR_THR(("<%d> null image info - id=%p, name=%s",
                         getAppId(),
                         imageId,
                         image->GetName()));
      reset(true);
      return;
   }

   ETG_TRACE_USR4_THR(("<%d> onUnload - cached(count=%u, size=%u), active(count=%u, size=%u), image(index=%u, id=%p, size=%u, upCounter=%d, name=%s)",
                       getAppId(),
                       getCachedImagesCount(),
                       getCachedImagesSize(),
                       getActiveImagesCount(),
                       getActiveImagesSize(),
                       (imageInfo == NULL) ? -1 : imageInfo->getOwnerAppendIndex(),
                       imageId,
                       imageSize,
                       (imageInfo == NULL) ? -1 : imageInfo->_uploadCounter,
                       image->GetName()));

   //decrease upload counter
   --(imageInfo->_uploadCounter);
   //still uploaded by other clients
   if (imageInfo->_uploadCounter > 0)
   {
      return;
   }

   _activeImagesSize -= imageInfo->_imageSize;

   imageInfo->updateInfo();
   //image can fit in cache if its size does not exceed MaximumTotalImagesSize and MaximumCachedImagesSize
   if (((_maximumImageSize > 0) && (imageInfo->_imageSize <= _maximumImageSize))
         || ((_maximumCachedImagesSize > 0) && (imageInfo->_imageSize <= _maximumCachedImagesSize))
         || ((_maximumTotalImagesSize > 0) && (imageInfo->_imageSize <= _maximumTotalImagesSize))
      )
   {
      _cachedImages.appendImage(imageInfo);

      ETG_TRACE_USR1_THR(("<%d> add image - cached(count=%u, size=%u), active(count=%u, size=%u), image(index=%u, id=%p, size=%u, name=%s)",
                          getAppId(),
                          getCachedImagesCount(),
                          getCachedImagesSize(),
                          getActiveImagesCount(),
                          getActiveImagesSize(),
                          imageInfo->getOwnerAppendIndex(),
                          imageId,
                          getImageSize(image),
                          image->GetName()));
   }
   //image is too big for this cache
   else
   {
      ETG_TRACE_USR1_THR(("<%d> skip image too big to be cached - cachedMaxSize=%u, totalMaxSize=%u, imageMaxSize=%u, image(index=%u, id=%p, size=%u, name=%s)",
                          getAppId(),
                          _maximumCachedImagesSize,
                          _maximumTotalImagesSize,
                          _maximumImageSize,
                          imageInfo->getOwnerAppendIndex(),
                          imageId,
                          getImageSize(image),
                          image->GetName()));

      _images.erase(it);
      _cachedImages.removeImage(imageInfo);

      deleteImageInfo(imageInfo);
   }

   //ensure total uploaded data does not exceed maximum allowed
   ensureMaximumSize();
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::deleteImageInfo(ImageInfo* imageInfo, bool isError)
{
   if (imageInfo != NULL)
   {
      const ImageSharedPointer& image = imageInfo->_image;

      if (isError)
      {
         ETG_TRACE_ERR_THR(("<%d> delete image info - image(index=%u, id=%p, size=%u, upCounter=%u, name=%s)",
                            getAppId(),
                            imageInfo->getOwnerAppendIndex(),
                            getImageId(image),
                            getImageSize(image),
                            imageInfo->_uploadCounter,
                            image->GetName()));
      }

      if (!image.PointsToNull())
      {
         //don't force unload because the image may be used without cache knowledge
         image->Unload();
      }
      FEATSTD_DELETE(imageInfo);
   }
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::reset(bool isError)
{
   if (isError)
   {
      ETG_TRACE_ERR_THR(("<%d> reset cache - cached(count=%u, size=%u), active(count=%u, size=%u)",
                         getAppId(),
                         getCachedImagesCount(),
                         getCachedImagesSize(),
                         getActiveImagesCount(),
                         getActiveImagesSize()));
   }
   else
   {
      ETG_TRACE_USR1_THR(("<%d> reset cache - cached(count=%u, size=%u), active(count=%u, size=%u)",
                          getAppId(),
                          getCachedImagesCount(),
                          getCachedImagesSize(),
                          getActiveImagesCount(),
                          getActiveImagesSize()));
   }

   //remove all entries from the map
   for (ImageInfoMap::iterator it = _images.begin(); it != _images.end(); ++it)
   {
      ImageInfo* imageInfo = it->second;
      if (imageInfo != NULL)
      {
         _cachedImages.removeImage(imageInfo);
         deleteImageInfo(imageInfo, isError);
      }
   }

   //remove all the extra cached entries
   while (_cachedImages.getOldestImage() != NULL)
   {
      ImageInfo* imageInfo = _cachedImages.getOldestImage();
      if (imageInfo != NULL)
      {
         _cachedImages.removeImage(imageInfo);
         deleteImageInfo(imageInfo, isError);
      }
   }

   _images.clear();
   _cachedImages.reset();
   _activeImagesSize = 0;
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::ensureMaximumSize()
{
   while ((_cachedImages.getOldestImage() != NULL)
          && (((getTotalImagesSize() > _maximumTotalImagesSize) && (_maximumTotalImagesSize > 0))
              || ((getCachedImagesSize() > _maximumCachedImagesSize) && (_maximumCachedImagesSize > 0))))
   {
      ImageInfo* imageInfo = _cachedImages.getOldestImage();
      const ImageSharedPointer& image = imageInfo->_image;
      ImageId imageId = getImageId(image);

      ETG_TRACE_USR1_THR(("<%d> remove image - cached(size=%u, maxSize=%u), total(size=%u, maxSize=%u), image(index=%u, id=%p, size=%u, name=%s)",
                          getAppId(),
                          getCachedImagesSize(),
                          _maximumCachedImagesSize,
                          getTotalImagesSize(),
                          _maximumTotalImagesSize,
                          imageInfo->getOwnerAppendIndex(),
                          imageId,
                          getImageSize(image),
                          image->GetName()));

      if (imageInfo->_uploadCounter == 0)
      {
         //remove entry from map
         ImageInfoMap::iterator it = _images.find(imageId);
         if (it != _images.end())
         {
            _images.erase(it);
         }
         //remove entry from cached images
         _cachedImages.removeImage(imageInfo);

         deleteImageInfo(imageInfo);

         ////check if image is still uploaded by other clients
         //this causes a crash!
         //if (!image.PointsToNull() && image->IsUploaded())
         //{
         //   ETG_TRACE_COMP_THR(("<%d> image unload not completed (it could have other clients) - image(id=%p, size=%u, name=%s)",
         //      getAppId(),
         //      getImageId(image),
         //      getImageSize(image),
         //      image->GetName()));
         //}
      }
      //image is still used
      else
      {
         ETG_TRACE_ERR_THR(("<%d> image in cache but still active - image(index=%u, id=%p, size=%u, upCounter=%u, name=%s)",
                            getAppId(),
                            imageInfo->getOwnerAppendIndex(),
                            imageId,
                            getImageSize(image),
                            imageInfo->_uploadCounter,
                            image->GetName()));

         reset(true);
      }
   }
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageInfo::ImageInfo(const ImageSharedPointer& image) :
   _image(image),
   _imageId(0),
   _imageSize(0),
   _uploadCounter(0),
   _ownerList(NULL),
   _next(NULL),
   _previous(NULL),
   _ownerAppendIndex(0)
{
   updateInfo();
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageInfo::~ImageInfo()
{
   if (_ownerList != NULL)
   {
      ETG_TRACE_ERR_THR(("<%d> delete image info still referenced by a list - image(index=%u, id=%p, size=%u, upCounter=%u, name=%s)",
                         ImageUploadCache::getAppId(),
                         getOwnerAppendIndex(),
                         ImageUploadCache::getImageId(_image),
                         ImageUploadCache::getImageSize(_image),
                         _uploadCounter,
                         ImageUploadCache::getImageName(_image)));
   }

   _ownerList = NULL;
   _next = NULL;
   _previous = NULL;
}


/////////////////////////////////////////////////////////////////////////
void ImageUploadCache::ImageInfo::updateInfo()
{
   _imageId = ImageUploadCache::getImageId(_image);
   _imageSize = ImageUploadCache::getImageSize(_image);
}


/////////////////////////////////////////////////////////////////////////
bool ImageUploadCache::ImageInfo::isActive() const
{
   return _uploadCounter > 0;
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageInfoList::ImageInfoList() : _oldestImage(NULL), _imagesCount(0), _imagesSize(0), _appendIndex(0)
{
}


/////////////////////////////////////////////////////////////////////////
ImageUploadCache::ImageInfoList::~ImageInfoList()
{
   reset();
}


void ImageUploadCache::ImageInfoList::reset()
{
   while (_oldestImage != NULL)
   {
      ImageInfo* temp = _oldestImage;
      removeImage(temp);
      FEATSTD_DELETE(temp);
   }
   _imagesCount = 0;
   _imagesSize = 0;
   //_appendIndex = 0;//don't reset it
}


/////////////////////////////////////////////////////////////////////////
bool ImageUploadCache::ImageInfoList::removeImage(ImageInfo* imageInfo)
{
   if ((imageInfo == NULL) || (imageInfo->_ownerList != this))
   {
      return false;
   }

   //remove oldest entry
   if (imageInfo == _oldestImage)
   {
      //it is also the only entry
      if (_oldestImage->_next == _oldestImage)
      {
         _oldestImage = NULL;
      }
      //there are more entries
      else
      {
         _oldestImage = _oldestImage->_next;
      }
   }

   //adjust links for previous and next entries
   if ((imageInfo->_next != NULL) && (imageInfo->_previous != NULL))
   {
      imageInfo->_previous->_next = imageInfo->_next;
      imageInfo->_next->_previous = imageInfo->_previous;
   }

   //clear previous and next links
   imageInfo->_ownerList = NULL;
   imageInfo->_next = NULL;
   imageInfo->_previous = NULL;

   --_imagesCount;
   //imageInfo->_ownerAppendIndex = 0;//don't reset it
   _imagesSize -= imageInfo->_imageSize;

   return true;
}


/////////////////////////////////////////////////////////////////////////
bool ImageUploadCache::ImageInfoList::appendImage(ImageInfo* imageInfo)
{
   if (imageInfo == NULL)
   {
      return false;
   }

   //if the entry belongs to another list => remove it from there first
   if (imageInfo->_ownerList != NULL)
   {
      imageInfo->_ownerList->removeImage(imageInfo);
   }

   //take ownership
   imageInfo->_ownerList = this;

   //no entry yet => set as head
   if (_oldestImage == NULL)
   {
      imageInfo->_next = imageInfo;
      imageInfo->_previous = imageInfo;

      _oldestImage = imageInfo;
   }
   //other entries
   else
   {
      imageInfo->_previous = _oldestImage->_previous;
      imageInfo->_next = _oldestImage;

      _oldestImage->_previous->_next = imageInfo;
      _oldestImage->_previous = imageInfo;
   }

   ++_imagesCount;
   ++_appendIndex;
   imageInfo->_ownerAppendIndex = _appendIndex;
   _imagesSize += imageInfo->_imageSize;

   return true;
}


}//utils
}//view
}//hmibase

#endif// VARIANT_S_FTR_ENABLE_IMAGE_UPLOAD_CACHE
