/* ***************************************************************************************
* FILE:          ImageLoader.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageLoader.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(_IMAGE_LOADER_H)
#define _IMAGE_LOADER_H

#include "Candera/Engine2D/Core/BitmapImage2D.h"
#include "hmibase/util/Macros.h"

#include <stdint.h>

namespace Candera {
class Bitmap;
}


namespace hmibase {
namespace view {

/* Utility class which provides support to load bitmaps from different sources.
* It uses Developer's Image Library (DevIL) which supports many formats (png, jpg, gif, tga and many more).
* The returned bitmap can be used with an image shared pointer in order to have automatic unloading and deallocation of memory.
*/
class ImageLoader
{
   public:
      /* Returns a shared pointer to a bitmap image from the asset. The returned image is a just a wrapper which will retrieve the actual bitmap during rendering. */
      static Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> getAssetBitmapImage(const Candera::Char* canderaName);

      /* Creates a shared pointer to image which uses the specified bitmap.
      * When all the references to the shared pointer are deleted the bitmap will be unloaded and the memory will be freed.
      */
      static Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> createImage(Candera::MemoryManagement::SharedPointer<Candera::Bitmap> bmp);

      /* Loads a bitmap from a file. This method is thread safe.*/
      static Candera::Bitmap::SharedPointer loadBitmapFile(const Candera::Char* fileName);

      /* Loads a bitmap from a file and creates a bitmap image. This method is thread safe.*/
      static Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> loadBitmapImageFile(const Candera::Char* fileName);

      /* Loads a bitmap from a data buffer. This method is thread safe. */
      static Candera::Bitmap::SharedPointer loadBitmapData(const void* data, size_t dataSize);

      _FEATSTD_DEPRECATED("This method is not thread safe, it should not be used. Returns NULL. Use getAssetBitmapImage() instead.",
                          static Candera::Bitmap* loadBitmapResource(const Candera::Char* canderaName)
                         );

      _FEATSTD_DEPRECATED("This method will be removed next time, please use the SharedPointer type of createImage() instead.",
                          static Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> createImage(Candera::Bitmap* bitmap)
                         );

      static bool saveImage(void* data, uint32_t size, uint32_t width, uint32_t height, const char* filePath = 0, bool fileOverwrite = false);
};


}
}


HMIBASE_DEPRECATED_TYPE_ALIAS(ImageLoader, hmibase::view::ImageLoader);

#endif//_IMAGE_LOADER_H
