/* ***************************************************************************************
* FILE:          CourierMessageMapper.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CourierMessageMapper.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __COURIER_MSG_MAPPER_H_
#define __COURIER_MSG_MAPPER_H_

#include "Courier/Base.h"
#include "Courier/Messaging/Message.h"
#include "Courier/Foundation/FoundationMsgs.h"
#include "View/CGI/CourierMsgTracing/CourierMessageMapTraceIn.h"
#include "View/CGI/CourierMsgTracing/CourierMessageMapTraceOut.h"

#define courier_messages protected     // to define a section for incomming messages

#ifndef TABSIZE
# define TABSIZE(a) (sizeof(a)/sizeof(a[1]))
#endif


/////////////////////////////////////////////////////////////////////////////
//
// Courier message map dispatcher
//
//
//COURIER_MSG_MAP_BEGIN(traceClass)
// ON_COURIER_MESSAGE
// ON_COURIER_MESSAGE
// ON_COURIER_MESSAGE
// COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN
//    COURIER_MSG_DELEGATE_TO_OBJ
//    COURIER_MSG_DELEGATE_TO_CLASS
//    COURIER_MSG_DELEGATE_TO_REF
//    COURIER_MSG_DELEGATE_TO_FKT
// COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS
// COURIER_MSG_MAP_NOTCONSUMED_DELEGATE_BEGIN
//    COURIER_MSG_DELEGATE_TO_OBJ
//    COURIER_MSG_DELEGATE_TO_CLASS
//    COURIER_MSG_DELEGATE_TO_REF
//    COURIER_MSG_DELEGATE_TO_FKT
// COURIER_MSG_MAP_NOTCONSUMED_DELEGATE_ENDS
//COURIER_MSG_MAP_ENDS
//
//


#define GETTRACECLASS getLocalTraceClass

/*lint -emacro(1511, COURIER_DECLARE_TRACECLASS) */
#define COURIER_DECLARE_TRACECLASS(_TRACECLASS_)                        \
/*lint -save -e1736*/                                                   \
     private:                                                           \
        inline uint32_t GETTRACECLASS() const { return _TRACECLASS_; }  \
/*lint -restore*/


/*lint -emacro(1511, COURIER_MSG_MAP_BEGIN) */
#define COURIER_MSG_MAP_BEGIN(_TRACECLASS_)                             \
/*lint -save -e1736*/                                                   \
      private:                                                          \
        inline uint32_t GETTRACECLASS() const { return _TRACECLASS_; }  \
      public:                                                           \
    virtual bool OnMessage(const Courier::Message & Msg)                \
    {   bool consumed = false;                                          \
        switch (Msg.GetId())                                            \
        {
//lint -emacro(774, ON_COURIER_MESSAGE)
//lint -emacro(845, ON_COURIER_MESSAGE)
#define ON_COURIER_MESSAGE(_type_)                                               \
      case _type_::ID: {                                                         \
         const _type_ &  oMsg = static_cast<const _type_ & >(Msg);               \
         CourierMessageMapTraceIn::printIncoming(oMsg,                           \
         __FILE__,__LINE__, GETTRACECLASS());                                    \
         consumed = onCourierMessage(oMsg);                                      \
         if((_type_::ID == Courier::StartupMsg::ID) && (consumed)) return false; \
         if(consumed) return true; else break;                                   \
      }


#define ON_COURIER_MESSAGE_NOTRACE(_type_)                              \
      case _type_::ID: {                                                \
         const _type_ &  oMsg = static_cast<const _type_ & >(Msg);      \
         consumed = onCourierMessage(oMsg);                             \
         if(consumed) return true; else break;                          \
      }


#define COURIER_DUMMY_CASE(x)                                           \
       case x: /* no break*/


#define COURIER_CASE_DUMMY_ENTRY() COURIER_DUMMY_CASE(1)


#define COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()                        \
       default:


#define COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()                         \
       break;                                                           \
    };


#define COURIER_MSG_MAP_NOTCONSUMED_DELEGATE_BEGIN()                    \
    /*if (!consumed)*/ {


#define COURIER_MSG_MAP_NOTCONSUMED_DELEGATE_ENDS()                     \
    }


#define COURIER_MSG_MAP_ENDS()                                          \
    return consumed;                                                    \
    }                                                                   \
/*lint -restore*/

#define COURIER_MSG_DELEGATE_TO_FKT(fkt)                                \
           if (fkt(Msg)) return true;

#define COURIER_MSG_DELEGATE_TO_CLASS(x)                                \
           if (x::OnMessage(Msg)) return true;

#define COURIER_MSG_DELEGATE_TO_REF(x)                                  \
           if (x.OnMessage(Msg)) return true;

#define COURIER_MSG_DELEGATE_TO_OBJ(x)                                  \
           if ((x != 0) && (x->OnMessage(Msg))) return true;


//// old compatible macros, do not use it further more


#define COURIER_MSG_MAP_DELEGATE_DEF_BEGIN()                            \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()

#define COURIER_MSG_MAP_DELEGATE_DEF_END()

#define COURIER_MSG_MAP_DELEGATE_START()                                \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()

#define COURIER_MSG_MAP_END()                                           \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()                             \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()                              \
   COURIER_MSG_MAP_ENDS()


#define COURIER_MSG_MAP_DELEGATE_END()                                  \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()                              \
   COURIER_MSG_MAP_ENDS()

#define COURIER_MSG_MAP_END_DELEGATE(_x_)                               \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()                             \
      COURIER_MSG_DELEGATE_TO_CLASS(_x_)                                \
   COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()                              \
   COURIER_MSG_MAP_ENDS()

/////////////////////////////////////////////////////////////////////////////
//
// Courier message binding dispatcher -> UpdateModelMsg()
//


#define COURIER_BINDING_MAP_BEGIN()                                        \
   bool onCourierMessage(const Courier::UpdateModelMsg& oMsg)              \
   {  unsigned handledcnt=0;                                               \
      for (unsigned i = 0; i < oMsg.RequestCount(); ++i)                   \
      {                                                                    \
         Courier::Request request(oMsg.GetRequest(i));                     \
         switch (request.BindingSourceKey())                               \
         {
#define COURIER_BINDING_ITEM_CHANGE(_item_)                                \
      case ItemKey::_item_:                                                \
          if(onCourierBindingItemChange_##_item_(request)) handledcnt++;   \
          break;                                                           \
 
#define COURIER_BINDING_MAP_END()                                          \
       default:                                                            \
            /* TRACE_ERR(("Item 'ItemKey::%s' not handled","#_item_")) */  \
            break;                                                  \
         } /* switch*/;                                                    \
      } /*for .. */                                                        \
      return handledcnt == oMsg.RequestCount() ? true:false;               \
   } /* OnMessage */


//
// Post of outgoing messages with TRACE
//

#define POST_MSG_NOTRACE(_message_) {                                      \
   Courier::Message * pMsg =  _message_;                                   \
   bool rc = (0 != pMsg);                                                  \
   if (rc) {                                                               \
      pMsg->Post();                                                        \
   }                                                                       \
}


#define POST_MSG(_message_) {                                              \
   Courier::Message * pMsg =  _message_;                                   \
   bool rc = (0 != pMsg);                                                  \
   if (rc) {                                                               \
      CourierMessageMapTraceOut::printOutgoing(*pMsg,                      \
         __FILE__,__LINE__, GETTRACECLASS());                              \
      rc = pMsg->Post();                                                   \
      if (!rc) {                                                           \
         CourierMessageMapTraceOut::printError(                            \
            __FILE__,__LINE__, GETTRACECLASS());                           \
      }                                                                    \
   }                                                                       \
   else {                                                                  \
      CourierMessageMapTraceOut::printError(                               \
         __FILE__,__LINE__, GETTRACECLASS());                              \
   }                                                                       \
}


#define SIMPLE_EMPTY_VIEW_CONTROLLER(tController)           \
class tController : public ::hmibase::view::ViewControllerBase               \
{                                                           \
      COURIER_MSG_MAP_BEGIN(0)                              \
      COURIER_CASE_DUMMY_ENTRY()                            \
      COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()              \
      COURIER_MSG_DELEGATE_TO_CLASS(::hmibase::view::ViewControllerBase)     \
      COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()               \
      COURIER_MSG_MAP_ENDS()                                \
      tController() {}                                      \
      virtual ~tController() {}                             \
      static const char* _name;                             \
      static Courier::ViewController* createInstance()      \
      {                                                     \
         return FEATSTD_NEW(tController)();                 \
      }                                                     \
};


#define EXTENDED_VIEW_CONTROLLER(tController, focuspriority)\
class tController : public ::hmibase::view::ViewControllerBase               \
{                                                           \
      COURIER_MSG_MAP_BEGIN(0)                              \
      COURIER_CASE_DUMMY_ENTRY()                            \
      COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()              \
      COURIER_MSG_DELEGATE_TO_CLASS(::hmibase::view::ViewControllerBase)     \
      COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()               \
      COURIER_MSG_MAP_ENDS()                                \
      tController() : ::hmibase::view::ViewControllerBase(focuspriority){}   \
      virtual ~tController() {}                             \
      static const char* _name;                             \
      static Courier::ViewController* createInstance()      \
      {                                                     \
         return COURIER_NEW(tController)();                 \
      }                                                     \
};


#define SIMPLE_EMPTY_EXT_VIEW_CONTROLLER(tController)       \
class tController : public ::hmibase::view::ViewControllerBase               \
{                                                           \
      COURIER_MSG_MAP_BEGIN(0)                              \
      COURIER_DUMMY_CASE(1)                                 \
      COURIER_MSG_MAP_END()                                 \
      tController() {}                                      \
      virtual ~tController() {}                             \
      static const char* _name;                             \
      static const char* _basename;                         \
      static Courier::ViewController* createInstance()      \
      {                                                     \
         return COURIER_NEW(tController)();                 \
      }                                                     \
};


#endif
