/* ***************************************************************************************
* FILE:          ColorManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ColorManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef __HMIBASE_COLOR_MANAGER__
#define __HMIBASE_COLOR_MANAGER__

#include "ColorAccessor.h"
#include <map>

namespace hmibase {
namespace util {
namespace color {


/************************************************************************/
using Candera::Color;
typedef size_t ColorId;
typedef size_t TableId;


/************************************************************************/
class ColorTable
{
      typedef std::map<ColorId, Color> ColorMapType;

   public:
      const Color& getDefaultColor() const;
      void setDefaultColor(const Color& defaultColor);
      const Color& getColor(ColorId id) const;
      void setColor(ColorId id, const Color& color);

   private:
      ColorMapType _colorMap;
      Color _defaultColor;
};


/************************************************************************/
class ColorManager
{
      friend class ColorManagerInitializer;

   public:
      typedef std::map<TableId, ColorTable*> TableMapType;

      ColorManager();
      ~ColorManager();
      static ColorManager& getInstance();

      const ColorTable* getTable(const TableId& id) const;
      const ColorTable& getDefaultTable() const;
      void setCurrentTable(const ColorTable* table);
      const ColorTable& getCurrentTable() const
      {
         return _currentTable != NULL ? (*_currentTable) : _defaultTable;
      }

      size_t getUpdateSequenceNr() const
      {
         return _updateSequenceNr;
      }

   private:
      void updateSequenceNumber();

      size_t _updateSequenceNr;
      const ColorTable* _currentTable;
      ColorTable _defaultTable;
      TableMapType _tableMap;
};


/************************************************************************/
class ColorManagerInitializer
{
   public:
      ColorManagerInitializer(ColorManager& colorManager);
      virtual bool initialize() = 0;

   protected:
      ColorManagerInitializer(const ColorManagerInitializer&);
      ColorManagerInitializer& operator=(const ColorManagerInitializer&);

      ColorTable& defaultTable();
      ColorManager::TableMapType& tableMap();

      ColorManager& _colorManager;
};


/************************************************************************/
class DefaultColorManagerInitializer : public ColorManagerInitializer
{
      typedef ColorManagerInitializer Base;

   public:
      static const int VERSION = 1;

      DefaultColorManagerInitializer(ColorManager& colorManager, const std::string& colorTableRawResourceName = "");
      virtual bool initialize();

   protected:
      std::string _colorTableRawResourceName;
};


}
}


}


#endif
