/* ***************************************************************************************
* FILE:          AttachedAnimation.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AttachedAnimation.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "AttachedAnimation.h"
#include "AppViewHandler.h"
#include "ViewAccessor.h"

#include <Candera/EngineBase/Animation/AnimationPlayer.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>

#include <Courier/Visualization/AnimationProperties.h>
#include <Courier/Visualization/ViewScene2D.h>
#include <Courier/Visualization/ViewScene3D.h>

#include <Trace/ToString.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_VIEW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AttachedAnimation.cpp.trc.h"
#endif


namespace hmibase {
namespace view {


/************************************************************************/
AttachedAnimationMessageHandler::AttachedAnimationMessageHandler(const FeatStd::Char* defaultAttachedViewName, const FeatStd::Char* textureRTNodeName, const FeatStd::Char* defaultAnimationName)
   : _defaultAttachedViewId(defaultAttachedViewName),
     _textureRTNodeName(textureRTNodeName),
     _defaultAnimationId(defaultAnimationName),
     _autoStart(false)
{
}


/************************************************************************/
AttachedAnimationMessageHandler::~AttachedAnimationMessageHandler()
{
}


/************************************************************************/
const Courier::ViewId& AttachedAnimationMessageHandler::GetDefaultAttachedViewId() const
{
   return _defaultAttachedViewId;
}


/************************************************************************/
const FeatStd::Char* AttachedAnimationMessageHandler::GetTextureRTNodeName() const
{
   return _textureRTNodeName.c_str();
}


/************************************************************************/
const Courier::ItemId& AttachedAnimationMessageHandler::GetDefaultAnimationId() const
{
   return _defaultAnimationId;
}


/************************************************************************/
bool AttachedAnimationMessageHandler::IsAutoStartEnabled() const
{
   return _autoStart;
}


/************************************************************************/
void AttachedAnimationMessageHandler::SetAutoStart(bool enable)
{
   _autoStart = enable;
}


/************************************************************************/
bool AttachedAnimationMessageHandler::OnMessage(ViewControllerBase& viewController, const Courier::Message& msg)
{
   bool consumed = false;
   switch (msg.GetId())
   {
      case Courier::ParentViewActivateEvent::ID:
      {
         const Courier::ParentViewActivateEvent* pMsg = Courier::message_cast<const Courier::ParentViewActivateEvent*>(&msg);
         if (pMsg != NULL)
         {
            OnViewActivate(viewController, pMsg->GetActivated());
         }
         break;
      }

      case AttachedAnimationReqMsg::ID:
      {
         const AttachedAnimationReqMsg* pMsg = Courier::message_cast<const AttachedAnimationReqMsg*>(&msg);
         if (pMsg != NULL)
         {
            consumed = OnAttachedAnimationReqMsg(viewController, *pMsg);
         }
         break;
      }

      case Courier::AnimationIndMsg::ID:
      {
         const Courier::AnimationIndMsg* pMsg = Courier::message_cast<const Courier::AnimationIndMsg*>(&msg);
         if (pMsg != NULL)
         {
            consumed = OnAnimationIndMsg(viewController, *pMsg);
         }
         break;
      }

      default:
         break;
   }
   return consumed;
}


/************************************************************************/
void AttachedAnimationMessageHandler::OnViewActivate(ViewControllerBase& viewController, bool activate)
{
   if (activate && _autoStart)
   {
      StartAnimation(viewController, GetDefaultAttachedViewId(), GetTextureRTNodeName(), GetDefaultAnimationId(), Courier::AnimationProperties());
   }
}


/************************************************************************/
bool AttachedAnimationMessageHandler::OnAttachedAnimationReqMsg(ViewControllerBase& viewController, const AttachedAnimationReqMsg& msg)
{
   bool consumed = false;

   if ((viewController.GetView() != NULL) && ((msg.GetViewId() == Courier::ViewId()) || (msg.GetViewId() == viewController.GetView()->GetId())))
   {
      switch (msg.GetAnimationAction())
      {
         case Courier::AnimationAction::Start:
            consumed = StartAnimation(viewController,
                                      msg.GetAttachedViewId() != Courier::ViewId() ? msg.GetAttachedViewId() : GetDefaultAttachedViewId(),
                                      !msg.GetTextureRTNode().IsEmpty() ? msg.GetTextureRTNode() : _textureRTNodeName.c_str(),
                                      msg.GetAnimationId() != Courier::ItemId() ? msg.GetAnimationId() : GetDefaultAnimationId(),
                                      msg.GetAnimationProperties());
            break;

         default:
            break;
      }
   }

   return consumed;
}


/************************************************************************/
bool AttachedAnimationMessageHandler::OnAnimationIndMsg(ViewControllerBase& viewController, const Courier::AnimationIndMsg& msg)
{
   OnAnimationEnded(viewController, msg.GetAnimationId());

   //don't consume the message
   return false;
}


/************************************************************************/
void AttachedAnimationMessageHandler::OnAnimationEnded(ViewControllerBase& viewController, const Courier::ItemId& animationId)
{
   ETG_TRACE_USR4_THR(("OnAnimationEnded animation=%75s view=%s!",
                       HMIBASE_TO_STRING(animationId),
                       HMIBASE_TO_STRING_V(viewController.GetView())));

   if ((_animationId == Courier::ItemId()) || (animationId != _animationId))
   {
      //not our animation
      return;
   }

   ETG_TRACE_USR1_THR(("OnAnimationEnded _attachedView=%75s _animation=%75s _cameraRestore=[%u,%p] view=%s!",
                       HMIBASE_TO_STRING(_attachedViewId),
                       HMIBASE_TO_STRING(_animationId),
                       _cameraRestoreInfo.IsSet(),
                       HMIBASE_TO_STRING_GDU(*_cameraRestoreInfo),
                       HMIBASE_TO_STRING_V(viewController.GetView())));

   ViewAccessor attachedViewAccessor(_attachedViewId);
   ViewAccessor activeViewAccessor(viewController.GetView());
   RestoreCamera(attachedViewAccessor, activeViewAccessor);

   attachedViewAccessor.DestroyView();

   _attachedViewId = Courier::ViewId();
   _animationId = Courier::ItemId();
   _cameraRestoreInfo = FeatStd::Optional<Candera::GraphicDeviceUnit*>();
}


/************************************************************************/
bool AttachedAnimationMessageHandler::StartAnimation(ViewControllerBase& viewController, const Courier::ViewId& attachedViewId, const FeatStd::String& textureRTNodeName, const Courier::ItemId& animationId, const Courier::AnimationProperties& animationProperties)
{
   ETG_TRACE_USR1_THR(("StartAnimation attachedView=%50s textureRtNode=%50s animation=%50s view=%s!",
                       HMIBASE_TO_STRING(attachedViewId),
                       textureRTNodeName.GetCString(),
                       HMIBASE_TO_STRING(animationId),
                       HMIBASE_TO_STRING_V(viewController.GetView())));

   if (_animationId != Courier::ItemId())
   {
      ETG_TRACE_ERR_THR(("StartAnimation animation (%75s) already started!", HMIBASE_TO_STRING(_animationId)));
      return false;
   }

   if (attachedViewId == Courier::ViewId())
   {
      ETG_TRACE_ERR_THR(("StartAnimation No attached view specified!"));
      return false;
   }

   if (animationId == Courier::ItemId())
   {
      ETG_TRACE_ERR_THR(("StartAnimation No animation specified!"));
      return false;
   }

   ViewAccessor attachedViewAccessor(attachedViewId);
   if (!attachedViewAccessor.CreateView())
   {
      ETG_TRACE_ERR_THR(("StartAnimation Failed to create attached view(%s)!", HMIBASE_TO_STRING(attachedViewId)));
      return false;
   }

   ViewAccessor activeViewAccessor(viewController.GetView());
   if (!PrepareCamera(attachedViewAccessor, textureRTNodeName, activeViewAccessor))
   {
      attachedViewAccessor.DestroyView();
      return false;
   }

   attachedViewAccessor.ShowView();

   static bool _animationIndMsgSubscribed = false;
   if (!_animationIndMsgSubscribed)
   {
      _animationIndMsgSubscribed = true;
      Courier::AnimationIndMsg::Subscribe(Courier::ComponentType::View, Courier::ComponentType::Controller);
   }

   AppViewHandler& viewHandler(AppViewHandler::getInstance());
   if (!viewHandler.ExecuteAnimationAction(Courier::AnimationAction::Start, Courier::ViewId(), Courier::CompositePath(), animationId, animationProperties))
   {
      ETG_TRACE_ERR_THR(("StartAnimation Failed to start animation (%s)!", HMIBASE_TO_STRING(animationId)));

      RestoreCamera(attachedViewAccessor, activeViewAccessor);
      attachedViewAccessor.DestroyView();

      return false;
   }

   _attachedViewId = attachedViewId;
   _animationId = animationId;

   return true;
}


/************************************************************************/
bool AttachedAnimationMessageHandler::PrepareCamera(ViewAccessor& attachedViewAccessor, const FeatStd::String& textureRTNodeName, ViewAccessor& activeViewAccessor)
{
   _cameraRestoreInfo = FeatStd::Optional<Candera::GraphicDeviceUnit*>();

   Candera::GraphicDeviceUnit* gdu = FindTextureRT(attachedViewAccessor, textureRTNodeName);
   if (gdu == NULL)
   {
      ETG_TRACE_ERR_THR(("PrepareCamera Framebuffer not found in attached view (%s)!", HMIBASE_TO_STRING_V(attachedViewAccessor.GetView())));
      return false;
   }

   Candera::CanderaObject* camera = FindCamera(activeViewAccessor);
   if (camera == NULL)
   {
      ETG_TRACE_ERR_THR(("PrepareCamera Camera not found in active view (%s)!", HMIBASE_TO_STRING_V(activeViewAccessor.GetView())));
      return false;
   }

   _cameraRestoreInfo = activeViewAccessor.SetCameraRT(camera, gdu);
   if (!_cameraRestoreInfo.IsSet())
   {
      ETG_TRACE_ERR_THR(("PrepareCamera Failed to set camera render target for view (%s)!", HMIBASE_TO_STRING_V(activeViewAccessor.GetView())));
      return false;
   }

   return true;
}


/************************************************************************/
bool AttachedAnimationMessageHandler::RestoreCamera(ViewAccessor& /*attachedViewAccessor*/, ViewAccessor& activeViewAccessor)
{
   if (!_cameraRestoreInfo.IsSet())
   {
      ETG_TRACE_USR4_THR(("RestoreCamera Camera restore info not valid for view (%s)!", HMIBASE_TO_STRING_V(activeViewAccessor.GetView())));
      return false;
   }

   Candera::GraphicDeviceUnit* gdu = *_cameraRestoreInfo;
   _cameraRestoreInfo = FeatStd::Optional<Candera::GraphicDeviceUnit*>();

   Candera::CanderaObject* camera = FindCamera(activeViewAccessor);
   if (camera == NULL)
   {
      ETG_TRACE_ERR_THR(("RestoreCamera Camera not found in view (%s)!", HMIBASE_TO_STRING_V(activeViewAccessor.GetView())));
      return false;
   }

   if (!activeViewAccessor.SetCameraRT(camera, gdu).IsSet())
   {
      ETG_TRACE_ERR_THR(("RestoreCamera Failed to set camera render target for view (%s)!", HMIBASE_TO_STRING_V(activeViewAccessor.GetView())));
      return false;
   }

   return true;
}


/************************************************************************/
Candera::GraphicDeviceUnit* AttachedAnimationMessageHandler::FindTextureRT(ViewAccessor& attachedViewAccessor, const FeatStd::String& textureRTNodeName)
{
   return attachedViewAccessor.FindTextureRT(textureRTNodeName.GetCString());
}


/************************************************************************/
Candera::CanderaObject* AttachedAnimationMessageHandler::FindCamera(ViewAccessor& activeViewAccessor)
{
   return activeViewAccessor.GetCamera(0);
}


}
}
