/* ***************************************************************************************
* FILE:          AssetCustomIdValidator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AssetCustomIdValidator is part of HMI-Base Framework
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "AssetCustomIdValidator.h"

#include <CanderaAssetLoader/AssetLoaderBase/AssetValidation.h>
#include <View/CGI/GuiBase/AppEnvironment.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/AssetCustomIdValidator.cpp.trc.h>
#endif

namespace hmibase {
namespace util {


/************************************************************************/
AssetCustomIdValidator::AssetCustomIdValidator(FeatStd::Int64 customId, bool assertEnabled)
   : _customId(customId), _assertEnabled(assertEnabled)
{
}


/************************************************************************/
AssetCustomIdValidator::~AssetCustomIdValidator()
{
}


/************************************************************************/
void AssetCustomIdValidator::SetCustomId(FeatStd::Int64 customId)
{
   ETG_TRACE_USR4_THR(("SetCustomId customId=%s", HMIBASE_TO_STRING(customId)));

   _customId = customId;
}


/************************************************************************/
void AssetCustomIdValidator::SetAssertEnabled(bool enable)
{
   ETG_TRACE_USR4_THR(("SetAssertEnabled assertEnabled=%u", enable));

   _assertEnabled = enable;
}


/************************************************************************/
void AssetCustomIdValidator::Register()
{
   ETG_TRACE_USR4_THR(("Register assertEnabled=%u customId=%s", _assertEnabled, HMIBASE_TO_STRING(_customId)));

   AppPlatform::AppEnvironment::GetEventSource().AddEventListener(this);
}


/************************************************************************/
void AssetCustomIdValidator::Unregister()
{
   ETG_TRACE_USR4_THR(("Unregister"));

   AppPlatform::AppEnvironment::GetEventSource().RemoveEventListener(this);
}


/************************************************************************/
FeatStd::EventResult::Enum AssetCustomIdValidator::OnEvent(const FeatStd::Event& event)
{
   const AppPlatform::AppEnvironment::AssetRepositoryEvent* assetRepoEvent = Candera::Dynamic_Cast<const AppPlatform::AppEnvironment::AssetRepositoryEvent*>(&event);
   if ((assetRepoEvent != NULL) && (assetRepoEvent->GetAssetRepository() != NULL)
         && (assetRepoEvent->GetState() == AppPlatform::AppEnvironment::AssetRepositoryEvent::Added))
   {
      Candera::AssetData::AssetVersionInfo versionInfo;
      Candera::AssetValidation::GetAssetVersionInfo(assetRepoEvent->GetAssetRepository(), versionInfo);

      ETG_TRACE_USR4_THR(("OnAssetRepositoryAdded customId=%12s assetFile=%s",
                          HMIBASE_TO_STRING(versionInfo.m_customId),
                          assetRepoEvent->GetFileName().c_str()));

      // validation disabled
      if (_customId == 0LL)
      {
         ETG_TRACE_USR4_THR(("Custom id validation disabled"));
      }
      // match
      else if (versionInfo.m_customId == _customId)
      {
         ETG_TRACE_USR1_THR(("Custom id match '%12s' for asset file '%s'",
                             HMIBASE_TO_STRING(_customId),
                             assetRepoEvent->GetFileName().c_str()));
      }
      // mismatch
      else
      {
         if (_assertEnabled)
         {
            ETG_TRACE_FATAL_THR(("Custom id mismatch 'Expected %12s != Found %12s' for asset file '%s'",
                                 HMIBASE_TO_STRING(_customId),
                                 HMIBASE_TO_STRING(versionInfo.m_customId),
                                 assetRepoEvent->GetFileName().c_str()));

            HMI_APP_ASSERT(versionInfo.m_customId != _customId);
         }
         else
         {
            ETG_TRACE_ERR_THR(("Custom id mismatch 'Expected %12s != Found %12s' for asset file '%s'",
                               HMIBASE_TO_STRING(_customId),
                               HMIBASE_TO_STRING(versionInfo.m_customId),
                               assetRepoEvent->GetFileName().c_str()));
         }
      }
   }

   return FeatStd::EventResult::Proceed;
}


}
}
