/* ***************************************************************************************
* FILE:          ViewTraverser.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewTraverser.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef HMIBASE_UTILS_TRACE_VIEW_TRAVERSE
#define HMIBASE_UTILS_TRACE_VIEW_TRAVERSE

#include <View/CGI/CgiExtensions/MessageHandler.h>
#include <hmibase/util/ItemRegistry.h>
#include <fstream>


/*****************************************************************************/
/* Forward declarations                                                      */
/*****************************************************************************/
namespace Candera {
class Scene2D;
class Node2D;
}


namespace Courier {
class IViewHandler;
class View;
}


class AppViewHandler;
namespace hmibase {
namespace trace {

class ReportWriter;
class ViewVisitor;


/*****************************************************************************/
/* Registry containing the registered view visitors.                         */
/* Applications can register view visitors and associate them to string ids. */
/* The string ids are used when a traverse session begins.                   */
/*****************************************************************************/
typedef hmibase::util::ItemPtrRegistry<std::string, ViewVisitor> ViewVisitorRegistry;


/*****************************************************************************/
/* Traverses all the views available in the current application              */
/* and invokes registered visitors to check each of them.                    */
/*****************************************************************************/
class ViewTraverser : public ::hmibase::view::MessageHandler
{
   public:
      typedef std::vector<ViewVisitorRegistry::IdType> VisitorsType;

      virtual ~ViewTraverser();

      /* Singleton instance */
      static ViewTraverser& getInstance();

      /* ViewHandler is required to retrieve the views. */
      void setViewHandler(AppViewHandler* viewHandler)
      {
         _viewHandler = viewHandler;
      }

      /* overrides from MessageHandler */
      virtual bool onMessage(const Courier::Message& msg)
      {
         return OnMessage(msg);
      }

      /* begins a new traverse session; if a session already exists it will be aborted. */
      void beginTraverse(const VisitorsType& visitorIds, const std::string& viewName);

      /* ends the current traverse session; it is called automatically after all the views were visited.*/
      void endTraverse();

   private:
      ViewTraverser();

      ViewTraverser(const ViewTraverser&);
      ViewTraverser& operator=(const ViewTraverser&);

      COURIER_MSG_MAP_BEGIN(TR_CLASS_HMI_FW)
      ON_COURIER_MESSAGE(Courier::ActivationResMsg)
      ON_COURIER_MESSAGE(TimerExpiredMsg)
      ON_COURIER_MESSAGE(TraverseViewsReqMsg)
      COURIER_MSG_MAP_END()

      bool onCourierMessage(const Courier::ActivationResMsg& msg);
      bool onCourierMessage(const TimerExpiredMsg& msg);
      bool onCourierMessage(const TraverseViewsReqMsg& msg);

      /* (re)starts the timer, also creates it if necessary. */
      void startTimer();
      /* stops the timer. */
      void stopTimer();
      /* reads view names from the asset. */
      void updateViewNames();
      /* requests current view or end the traverse session if all views were visited. */
      void checkCurrentViewIndex();
      /* requests the specified view to be shown, also hides the previous active view.*/
      void switchView(const Courier::ViewId& viewId);
      /* callback when a view became active */
      void onViewActivated(const Courier::ViewId& viewId);
      /* notifies visitors with the current view */
      void notifyVisitors(Courier::View& view);

      /* retrieves the views */
      AppViewHandler* _viewHandler;
      /* a delay is required between each view request */
      Util::Timer* _timer;
      /* view names retrieved from the asset */
      std::vector<std::string> _viewNames;
      /* index of the current view */
      size_t _currentViewIndex;
      /* indicates that a traverse session is in progress */
      bool _traverseInProgress;
      /* id of the previously activated view, used to hide old visited views */
      Courier::ViewId _previousActiveViewId;
      /* id of the visitors used for the current session */
      VisitorsType _visitorIds;
};


/*****************************************************************************/
/* View visitors check a view and write a report about it.                   */
/* Applications can register view visitors and associate them to string ids. */
/*****************************************************************************/
class ViewVisitor
{
   public:
      ViewVisitor(ReportWriter* reportWriter) : _reportWriter(reportWriter) {}

      virtual ~ViewVisitor()
      {
         _reportWriter = NULL;
      }

      /* Checks the specified view and writes a report about it using the report writer. */
      virtual void visitView(Courier::View& view) = 0;

      ReportWriter* getReportWriter() const
      {
         return _reportWriter;
      }
      void setReportWriter(ReportWriter* reportWriter)
      {
         _reportWriter = reportWriter;
      }

   private:
      /* used to write the report for the visited views. */
      ReportWriter* _reportWriter;
};


/*****************************************************************************/
/* Used to write reports using the following sequence:                       */
/* - begin, write, write, ..., end -                                         */
/* Some reporters support an optimized writing using a stream.               */
/*****************************************************************************/
class ReportWriter
{
   public:
      virtual ~ReportWriter() {}

      /* begins a new report  (opens a file report, writes a header into the trace console, etc). */
      virtual bool beginReport(const std::string& itemName) = 0;
      /* ends the current report (close file, writes summary, etc). */
      virtual bool endReport() = 0;
      /* aborts the current report. */
      virtual bool abortReport(const std::string& err) = 0;

      /* writes some information into the report. */
      virtual bool write(const std::string& line) = 0;
      /* optimized way to write into the report, not supported by all writers. */
      virtual std::ostream* getStream() = 0;
};


/*****************************************************************************/
/* Writes reports into the trace console.                                    */
/*****************************************************************************/
class TraceReportWriter : public ReportWriter
{
   public:
      TraceReportWriter() {}
      virtual ~TraceReportWriter() {}

      virtual bool beginReport(const std::string& itemName);
      virtual bool endReport();
      virtual bool abortReport(const std::string& err);
      virtual bool write(const std::string& line);
      virtual std::ostream* getStream()
      {
         return NULL;
      }
};


/*****************************************************************************/
/* Writes reports into the files.                                            */
/*****************************************************************************/
class FileReportWriter : public ReportWriter
{
   public:
      FileReportWriter(const std::string& fileNameFormat = "%s", const std::string& outputDirectory = "");
      virtual ~FileReportWriter();

      /* default output directory used if no specific one is set on this writer. */
      static std::string& getDefaultOutputDirectory();

      virtual bool beginReport(const std::string& itemName);
      virtual bool endReport();
      virtual bool abortReport(const std::string& err);
      virtual bool write(const std::string& line);
      virtual std::ostream* getStream()
      {
         return &_stream;
      }

      /* returns true if the report file is open. */
      bool isFileOpen() const;

      void setFileNameFormat(const std::string& fileNameFormat)
      {
         _fileNameFormat = fileNameFormat;
      }

      const std::string& getFileNameFormat() const
      {
         return _fileNameFormat;
      }

      void setOutputDirectory(const std::string& outputDirectory)
      {
         _outputDirectory = outputDirectory;
      }

      const std::string& getOutputDirectory() const
      {
         return _outputDirectory;
      }

   private:
      bool openFile(const std::string& fileName);
      void closeFile();

      std::string _fileNameFormat;
      std::string _outputDirectory;
      std::ofstream _stream;
};


/*****************************************************************************/
/* Traverses all the nodes in a 2D scene and writes an xml report with layout information. */
/*****************************************************************************/
class NodeLayoutViewVisitor : public ViewVisitor
{
      typedef ViewVisitor Base;

   public:
      NodeLayoutViewVisitor(ReportWriter* itemWriter) : Base(itemWriter) {}
      virtual ~NodeLayoutViewVisitor() {}

      virtual void visitView(Courier::View& view);

   protected:
      /* prints layout information for a node. */
      void printNodeLayout(Candera::Node2D& node, size_t level = 0);
};


/*****************************************************************************/
/* Traverses all the nodes in a 2D scene and writes a Scml report for it.    */
/* The Scml content can be loaded and rendered in SceneComposer              */
/* if the required resources are available in that solution.                 */
/*****************************************************************************/
class ScmlViewVisitor : public ViewVisitor
{
      typedef ViewVisitor Base;

   public:
      ScmlViewVisitor(ReportWriter* itemWriter) : Base(itemWriter) {}
      virtual ~ScmlViewVisitor() {}

      virtual void visitView(Courier::View& view);
};


/*****************************************************************************/
/* Prints a report regarding the widgets in a view.                          */
/*****************************************************************************/
class WidgetReportViewVisitor : public ViewVisitor
{
      typedef ViewVisitor Base;

   public:
      WidgetReportViewVisitor(ReportWriter* itemWriter) : Base(itemWriter) {}
      virtual ~WidgetReportViewVisitor() {}

      virtual void visitView(Courier::View& view);
};


/*****************************************************************************/
/* Registers default visitors.                                               */
/*****************************************************************************/
class DefaultVisitors
{
   public:
      typedef std::vector<std::string> IdsType;

      static void initialize();
      static const IdsType& getIds();
};


}
}


#endif //  HMIBASE_UTILS_TRACE_VIEW_TRAVERSE
